/*************************************************************************
 *                                                                       *
 *       (c) Copyright 2003                                              *
 *       All rights reserved                                             *
 *       Programs written by Jianghui Liu (NJIT)                         *
 *                                                                       *
 *       Permission to use, copy, modify, and distribute this software   *
 *       and its documentation for any purpose and without fee is her-   *
 *       eby granted, provided that this copyright notice appears in     *
 *       all copies.   Programmer(s) makes no representations about      *
 *       the suitability of this software for any purpose.  It is pro-   *
 *       vided "as is" without express or implied warranty.              *
 *                                                                       *
 *       08/06/2003                                                      *
 *************************************************************************/
package RNA;
import java.io.*;
import java.util.*;
import java.text.NumberFormat;

//---------------------------------------------------------------------------
//                   Utility Class for homology analysis:
//---------------------------------------------------------------------------

//--------------------------------------------------------------------------- 
// findLocality:  given a pair of homologies, find the optimal matched
//                pair that could be corresponding to the motif;
//---------------------------------------------------------------------------

public class HomoUtil 
{
    //-----------------------------------------------------------
    // Find the best aligned pair of structures from given DB
    //-----------------------------------------------------------
    public static MultiAlignment getSeedAlignment(String dataDB)
    {
        // set up code translation table and score matrix
        String codeTableFile = "codeTable.properties";
        String MatrixFile = "scoreMat.structure";
        Matcher m = new Matcher(codeTableFile, MatrixFile);
        int Penalty = -6;

        double score = Double.NEGATIVE_INFINITY;
        String doll = new String("-\\|/");
        RNAReader inA = new RNAReader(dataDB);
        RNAReader inB = new RNAReader(dataDB);
        RNA a = inA.getNextRNA();
        RNA b = inB.getNextRNA();

        RNA holderA = null, holderB = null;
        int count = 0;
        System.out.println("\n");
        while( a != null ){
            int[] map1 = new int[a.seqLength()];
            while( b != null) {
                String name1 = a.getName();
                String name2 = b.getName();
                if( (name1.split(":"))[0].equals((name2.split(":"))[0]) == false &&
                    name1.compareTo(name2) < 0) {
                    for(int k=0; k<map1.length; k++)
                        map1[k] = -1;
                    int[] map2 = new int[b.seqLength()];
                    for(int k=0; k<map2.length; k++)
                        map2[k] = -1;

                    double s = RNA.localMatch2D(a, b, map1, map2, m, Penalty);
                    int[] region = HomoUtil.getAlignedRegion(a, map1);
                    String seq1 = a.getSequence().substring(region[0], region[1]+1);
                    region = HomoUtil.getAlignedRegion(b, map2);
                    String seq2 = b.getSequence().substring(region[0], region[1]+1);

                    //if( seq1.equals(seq2) == false)
                    {
                        int[] measure = HomoUtil.getMeasures(a, map1, b, map2);
                        //int value2 = measure[7];  // overall sequence identity
                        if( s > score && measure[1]>(measure[0]+measure[1])*0.4) { // && value2 < 100) {
                            //System.out.println("keep score: " + s);
                            holderA = a;
                            holderB = b;
                            score = s;
                        }
                    } 
                }

                count ++;
                int dummy = count % 4;
                System.out.print("\rConstructing seed alignment " + count + ". Wait ... " + doll.substring(dummy, dummy+1));

                b = inB.getNextRNA();
            }
            a = inA.getNextRNA();
            inB = new RNAReader(dataDB);
            b = inB.getNextRNA();
        }

        if( holderA==null || holderB==null)
            return null;

        int[] aMap = new int[holderA.seqLength()];
        for(int i=0; i<aMap.length; i++)
            aMap[i] = -1;
        int[] bMap = new int[holderB.seqLength()];
        for(int i=0; i<bMap.length; i++)
            bMap[i] = -1;

        score = RNA.localMatch2D(holderA, holderB, aMap, bMap, m, Penalty);
        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(2);
        System.out.println("\nGot seed aligned as:");
        System.out.println("Score: " + nf.format(score) + RNA.alignMatch(holderA, aMap, holderB, bMap, m));

        return new MultiAlignment(holderA, aMap, holderB, bMap);
    }

    // ---------------------------------------------------------------------------
    //  Find out the putative motif region (substructure) of a pair of homologies. 
    //  The seed multi-alignment containting TWO structures is returned.
    //  The defalut parameters for the region detection is:
    //  (1) Scoreing Matrix: scoreMat.homo
    //  (2) Gap Penalty:     -6
    // ---------------------------------------------------------------------------
    public static MultiAlignment getSeedAlignment(String HomoDB1, String HomoID1, String HomoDB2, String HomoID2)
    {
        ArrayList RNAset1 = new ArrayList();
        ArrayList RNAset2 = new ArrayList();

        RNAReader in = null;
        in = new RNAReader(HomoDB1);
        RNA data = in.getNextRNA();
        while( data != null)
        {
            if( HomoID1 == null || data.getName().indexOf(HomoID1) != -1)
                RNAset1.add(data);
            
            data = in.getNextRNA();
        }

        in = new RNAReader(HomoDB2);
        data = in.getNextRNA();
        while( data != null)
        {
            if( HomoID2 == null || data.getName().indexOf(HomoID2) != -1)
                RNAset2.add(data);
            
            data = in.getNextRNA();
        }

        System.out.println(" Got " + RNAset1.size() + " structures for " + HomoID1);
        System.out.println(" Got " + RNAset2.size() + " structures for " + HomoID2);

        if( RNAset1.size() == 0 || RNAset2.size() == 0)
            return null;

        // set up code translation table and score matrix
        String codeTableFile = "codeTable.properties";
        String MatrixFile = "scoreMat.homo";
        Matcher m = new Matcher(codeTableFile, MatrixFile);
        int Penalty = -6;

        RNA a=null, b=null;
        double score = Double.NEGATIVE_INFINITY;
        int count = 0;
        String doll = new String("-\\|/");
        for(int i=0; i<RNAset1.size(); i++){
            RNA rna1 = (RNA)RNAset1.get(i);
            for(int j=0; j<RNAset2.size(); j++) {
                count ++;
                RNA rna2 = (RNA)RNAset2.get(j);

                int[] map1 = new int[rna1.seqLength()];
                for(int k=0; k<map1.length; k++)
                    map1[k] = -1;
                int[] map2 = new int[rna2.seqLength()];
                for(int k=0; k<map2.length; k++)
                    map2[k] = -1;

                double s = RNA.localMatch2D(rna1, rna2, map1, map2, m, Penalty);
                int[] region = HomoUtil.getAlignedRegion(rna1, map1);
                String seq1 = rna1.getSequence().substring(region[0], region[1]+1);
                region = HomoUtil.getAlignedRegion(rna2, map2);
                String seq2 = rna2.getSequence().substring(region[0], region[1]+1);
                if( seq1.equals(seq2) == false){
                    int[] measure = HomoUtil.getMeasures(rna1, map1, rna2, map2);
                    int value1 = (measure[1]*100)/(measure[1]+measure[0]);  // percentage of structured part
                    int value2 = measure[7];  // overall sequence identity
                    if( s > score && value1 >= 50 && value2 < 100) {
                        a = (RNA)RNAset1.get(i); 
                        b = (RNA)RNAset2.get(j);
                        score = s;
                    }
                }

                int dummy = count % 4;
                System.out.print("\rWaiting ... " + doll.substring(dummy, dummy+1));
            }
        }

        if( a==null || b==null)
            return null;

        int[] aMap = new int[a.seqLength()];
        for(int i=0; i<aMap.length; i++)
            aMap[i] = -1;
        int[] bMap = new int[b.seqLength()];
        for(int i=0; i<bMap.length; i++)
            bMap[i] = -1;

        score = RNA.localMatch2D(a, b, aMap, bMap, m, Penalty);
        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(2);
        System.out.println("\nGot seed aligned as:");
        System.out.println("Score: " + nf.format(score) + RNA.alignMatch(a, aMap, b, bMap, m));

        return new MultiAlignment(a, aMap, b, bMap);
    }

    //-----------------------------------------------------------------------------------------------
    // Find out optimal putative motif region(substructure) of a pair of homologies. The last parameter
    // excludes the set of structures from consideration.
    // The defalut parameters used by structure alignment is:
    // (1) Scoreing Matrix: scoreMat.homo
    // (2) Gap Penalty:     -6
    // ---------------------------------------------------------------------------------------------
    public static RNA[] findLocality(String HomoDB1, String HomoID1, String HomoDB2, String HomoID2,
                                     String[] Excludes)
    {
        ArrayList RNAset1 = new ArrayList();
        ArrayList RNAset2 = new ArrayList();

        if( Excludes != null)
            Arrays.sort(Excludes);

        RNAReader in = null;
        in = new RNAReader(HomoDB1);
        RNA data = in.getNextRNA();
        while( data != null)
        {
            if( data.getName().indexOf(HomoID1) != -1){
                if( Excludes != null && Arrays.binarySearch(Excludes, data.getName()) >= 0 )
                    continue;
                else
                    RNAset1.add(data);
            }
            data = in.getNextRNA();
        }

        in = new RNAReader(HomoDB2);
        data = in.getNextRNA();
        while( data != null)
        {
            if( data.getName().indexOf(HomoID2) != -1){
                if( Excludes != null && Arrays.binarySearch(Excludes, data.getName()) >= 0 )
                    continue;
                else
                    RNAset2.add(data);
            }
            data = in.getNextRNA();
        }

        System.out.println(" Got " + RNAset1.size() + " structures for " + HomoID1);
        System.out.println(" Got " + RNAset2.size() + " structures for " + HomoID2);

        if( RNAset1.size() == 0 || RNAset2.size() == 0)
            return null;

        // set up code translation table and score matrix
        String codeTableFile = "codeTable.properties";
        String MatrixFile = "scoreMat.homo";
        Matcher m = new Matcher(codeTableFile, MatrixFile);
        int Penalty = -6;

        RNA a=null, b=null;
        double score = Double.NEGATIVE_INFINITY;
        int count = 0;
        String doll = new String("-\\|/");
        for(int i=0; i<RNAset1.size(); i++){
            RNA rna1 = (RNA)RNAset1.get(i);
            for(int j=0; j<RNAset2.size(); j++) {
                count ++;
                RNA rna2 = (RNA)RNAset2.get(j);

                int[] map1 = new int[rna1.seqLength()];
                for(int k=0; k<map1.length; k++)
                    map1[k] = -1;
                int[] map2 = new int[rna2.seqLength()];
                for(int k=0; k<map2.length; k++)
                    map2[k] = -1;

                double s = RNA.localMatch2D(rna1, rna2, map1, map2, m, Penalty);
                int[] region = HomoUtil.getAlignedRegion(rna1, map1);
                String seq1 = rna1.getSequence().substring(region[0], region[1]+1);
                region = HomoUtil.getAlignedRegion(rna2, map2);
                String seq2 = rna2.getSequence().substring(region[0], region[1]+1);
                if( seq1.equals(seq2) == false){
                    int[] measure = HomoUtil.getMeasures(rna1, map1, rna2, map2);
                    int value1 = (measure[1]*100)/(measure[1]+measure[0]);  // percentage of structured part
                    int value2 = measure[7];  // overall sequence identity
                    if( s > score && value1 >= 50 && value2 < 100) {
                        a = (RNA)RNAset1.get(i); 
                        b = (RNA)RNAset2.get(j);
                        score = s;
                    }
                }

                int dummy = count % 4;
                System.out.print("\rWaiting ... " + doll.substring(dummy, dummy+1));
            }
        }

        if( a==null || b==null)
            return null;

        int[] aMap = new int[a.seqLength()];
        for(int i=0; i<aMap.length; i++)
            aMap[i] = -1;
        int[] bMap = new int[b.seqLength()];
        for(int i=0; i<bMap.length; i++)
            bMap[i] = -1;

        RNA.localMatch2D(a, b, aMap, bMap, m, Penalty);
        
        System.out.println(RNA.alignMatch(a, aMap, b, bMap, m));

        // only extract the aligned regions and return them as the localities.
        int[] region = HomoUtil.getAlignedRegion(a, aMap);
        String seq1 = a.getSequence().substring(region[0], region[1]+1);
        String stru1 = a.getStructLine().substring(region[0], region[1]+1);
        RNA structure1 = new RNA(a.getName(), seq1, stru1, a.getAnnotate());
                                                                                          
        region = HomoUtil.getAlignedRegion(b, bMap);
        String seq2 = b.getSequence().substring(region[0], region[1]+1);
        String stru2= b.getStructLine().substring(region[0], region[1]+1);
        RNA structure2 = new RNA(b.getName(), seq2, stru2, b.getAnnotate());

        RNA[] ret = new RNA[2];
        ret[0] = structure1;
        ret[1] = structure2;

        return ret;
    }

    //-------------------------------------------------------------------------------------------
    // Find ALL locality pairs from the provided two homologies. The number of returned
    // structures is EVEN. From the returned, each consecutive two structures are locality pairs.
    //--------------------------------------------------------------------------------------------
    public static RNA[] getMotifLocality(String HomoDB1, String HomoID1, String HomoDB2, String HomoID2)
    {
	ArrayList RNAset1 = new ArrayList();
	ArrayList RNAset2 = new ArrayList();
	RNAReader in = null;

	in = new RNAReader(HomoDB1);
	RNA data = in.getNextRNA();
	while( data != null) {
	    if( HomoID1 == null || data.getName().indexOf(HomoID1) != -1 )
		RNAset1.add(data);

	    data = in.getNextRNA();
	}

	in = new RNAReader(HomoDB2);
	data = in.getNextRNA();
	while( data != null) {
	    if( HomoID2 == null || data.getName().indexOf(HomoID2) != -1)
		RNAset2.add(data);
            
	    data = in.getNextRNA();
	}

	System.out.println(" Got " + RNAset1.size() + " structures from " + HomoDB1);
	System.out.println(" Got " + RNAset2.size() + " structures from " + HomoDB2);

	if( RNAset1.size() == 0 || RNAset2.size() == 0)
	    return null;

	// set up code translation table and score matrix
	String codeTableFile = "codeTable.properties";
	String MatrixFile = "scoreMat.homo";
	Matcher m = new Matcher(codeTableFile, MatrixFile);
	int Penalty = -6;

	double[] score = new double[RNAset1.size()*RNAset2.size()];
        int[]  SSident = new int[RNAset1.size() * RNAset2.size()];
        
	int count = 0;
        String doll = new String("-\\|/");
	for(int i=0; i<RNAset1.size(); i++){
	    RNA rna1 = (RNA)RNAset1.get(i);
	    for(int j=0; j<RNAset2.size(); j++) {
		count ++;
		RNA rna2 = (RNA)RNAset2.get(j);

		int[] map1 = new int[rna1.seqLength()];
		for(int k=0; k<map1.length; k++)
		    map1[k] = -1;
		int[] map2 = new int[rna2.seqLength()];
		for(int k=0; k<map2.length; k++)
		    map2[k] = -1;

                score[i*RNAset2.size() + j] = RNA.localMatch2D(rna1, rna2, map1, map2, m, Penalty);

                int[] region = HomoUtil.getAlignedRegion(rna1, map1);
                String seq1 = rna1.getSequence().substring(region[0], region[1]+1);
                region = HomoUtil.getAlignedRegion(rna2, map2);
                String seq2 = rna2.getSequence().substring(region[0], region[1]+1);

                if( seq1.equals(seq2) == false || (HomoID1==null && HomoID2==null) ){
		    int[] measure = HomoUtil.getMeasures(rna1, map1, rna2, map2);
                    SSident[i*RNAset2.size() + j] = measure[8];
		    int value1 = (measure[1]*100)/(measure[1]+measure[0]);  // percentage of structured part
		    if( !(value1 > 10 && (measure[0]+measure[1]) >= 15) ) {
		        score[i*RNAset2.size() + j] = -1;   // we don't want this pair !
                        SSident[i*RNAset2.size() + j] = -1;
		    }
                } else {
                    score[i*RNAset2.size() + j] = -1;
                    SSident[i*RNAset2.size() + j] = -1;
                }

                int dummy = count % 4;
                System.out.print("\rWaiting ... " + doll.substring(dummy, dummy+1));
	    }
	}

        ArrayList RetList = new ArrayList();
        boolean stop = false;
        double max = Statistic.getMax(score);
        System.out.println("\nmaxmum score is: " + max);
        double next2Max  = 0;
        while( stop != true ) {
            int s = Statistic.getIndexOfMax(score);
            if( score[s] > 0 && (score[s] >= max || SSident[s] >= 25) ) {
                int first = s / RNAset2.size();
                int second= s % RNAset2.size();
                RNA a1 = (RNA)RNAset1.get(first);
                RNA b1 = (RNA)RNAset2.get(second);
                boolean flag = false;
                for(int k=0; k<RetList.size(); k+=2) {
                    RNA a2 = (RNA)RetList.get(k);
                    RNA b2 = (RNA)RetList.get(k+1);
                    if( a1.getName().equals(a2.getName()) && b1.getName().equals(b2.getName()) ) {
                        flag = true;
                        break;
                    }
                }
                if( flag != true){
                    RetList.add( a1 );
                    RetList.add( b1 );
                }
                score[s] = -1;
            } else{
                stop = true;
                next2Max = score[s];
            }
        }
        //System.out.println("next2Max score: " + next2Max + "\n");

        if( RetList.size() == 0)
            return null;

        //int percentage = (int)(next2Max*100/max);

        RNA[] ret = new RNA[RetList.size()];
        for(int i=0; i<RetList.size(); i+=2) {
            RNA a = (RNA)RetList.get(i);
            RNA b = (RNA)RetList.get(i+1);
            int[] aMap = new int[a.seqLength()];
	    for(int k=0; k<aMap.length; k++)
	        aMap[k] = -1;
	    int[] bMap = new int[b.seqLength()];
	    for(int k=0; k<bMap.length; k++)
	        bMap[k] = -1;

	    RNA.localMatch2D(a, b, aMap, bMap, m, Penalty);

            //System.out.println(RNA.alignMatch(a, aMap, b, bMap, m));

            int[] region = HomoUtil.getAlignedRegion(a, aMap);
	    String seq1 = a.getSequence().substring(region[0], region[1]+1);
	    String stru1 = a.getStructLine().substring(region[0], region[1]+1);

            int start1=0, end1=0;
            String[] seg1 = a.getName().split(":");
            if( seg1.length == 2) {
                String[] ends = seg1[1].split("-");
                if( ends.length == 2) {
                    try{
                        start1 = Integer.parseInt(ends[0]);
                    } catch( NumberFormatException e){
                        start1 = 0;
                    }
                } else{
                    start1 = 0;
                }
            }
            end1 = start1 + region[1];
            start1 = start1 + region[0];

            region = HomoUtil.getAlignedRegion(b, bMap);
            String seq2 = b.getSequence().substring(region[0], region[1]+1);
            String stru2= b.getStructLine().substring(region[0], region[1]+1);
            int start2=0, end2=0;
            String[] seg2 = b.getName().split(":");
            if( seg2.length == 2) {
                String[] ends = seg2[1].split("-");
                if( ends.length == 2) {
                    try{
                        start2 = Integer.parseInt(ends[0]);
                    } catch( NumberFormatException e){
                        start2 = 0;
                    }
                } else{
                    start2 = 0;
                }
            }
            end2 = start2 + region[1];
            start2 = start2 + region[0];

            String newName1 = new String(seg1[0] + ":" + start1 + "-" + end1); 
                                  //+ "\t" + max);
                                                                                          
            String newName2 = new String(seg2[0] + ":" + start2 + "-" + end2 );
                                 // + "\t" + max);

            boolean flag = false;
            for(int k=0; k<i; k+=2) {
                if( ret[k] != null){
                    int startA=0, endA=0;
                    String[] segA = ret[k].getName().split(":");
                    String[] endsA = segA[1].split("-");
                    try{
                        startA = Integer.parseInt(endsA[0]);
                        endA   = Integer.parseInt(endsA[1]);
                    } catch (NumberFormatException e){
                        System.out.println("Error while removing overlap redundancy !!");
                        System.exit(1);
                    }

                    int startB=0, endB=0;
                    String[] segB = ret[k+1].getName().split(":");
                    String[] endsB = segB[1].split("-");
                    try{
                        startB = Integer.parseInt(endsB[0]);
                        endB   = Integer.parseInt(endsB[1]);
                    } catch (NumberFormatException e){
                        System.out.println("Error while removing overlap redundancy !!");
                        System.exit(1);
                    }

                    String name1 = (newName1.split(":"))[0];

                    String name2 = (newName2.split(":"))[0];

                    if( segA[0].equals(name1) && segB[0].equals(name2) && 
                              ((startA <= start1 && start1 <= endA) || (startA <= end1 && end1 <= endA) ) &&
                              ((startB <= start2 && start2 <= endB) || (startB <= end2 && end2 <= endB)) ){
                        flag = true;
                        break;
                    }
                }
            }
            if( flag != true){
                RNA structure1 = new RNA(newName1, seq1, stru1, a.getAnnotate());
                RNA structure2 = new RNA(newName2, seq2, stru2, b.getAnnotate());
                ret[i] = structure1;
                ret[i+1] = structure2;
            }
        }
        return ret;
    }
    

    //--------------------------------------------------------------
    // An integrated independent structure region is decided from
    // the alignment map which is the output of strutural alignment
    // algorithm(s).
    //--------------------------------------------------------------
    public static int[] getAlignedRegion(RNA a, int[] Map)
    {
        int[] region = new int[2];

        // the region SHOULD start AT MOST the first aligned column
        for(int i = 0; i < Map.length; i ++ ) {
            if( Map[i] != -1 ) {
                region[0] = i;
                break;
            }
        }

        // the region SHOULD end AT LEAST at the last aligned column
        for(int i = Map.length - 1; i >= 0; i --) {
            if( Map[i] != -1 ) {
                region[1] = i;
                break;
            }
        }

        // adjust the starting position lest that the structure integrity
        // is ruined by gap-matched stems.
        for(int j = region[0] - 1; j >= 0; j --) {
            int right = a.getPairedBase(j);
            if( right != -1 && right > j && right > region[0] && right < region[1] )
                region[0] = j;
        }

        // adjustment for the ending postion
        for(int j = region[1] + 1; j < Map.length; j ++) {
            int left = a.getPairedBase(j);
            if( left != -1 && left < j && left > region[0] && left < region[1])
                region[1] = j;
        }

        return region;
    }

    //-----------------------------------------------------------------------------------------
    //  Get measures of an structure alignment, the returned values are contained in an 
    //  Array int measure[]:
    //        measure[0]: for the FIRST structure, number of nt.(s) in SS region(s);
    //        measure[1]: for the FIRST structure, number of nt.(s) in DS region(s);
    //        measure[2]: number of OVERALL gaps in SS region(s);
    //        measure[3]: number of OVERALL gaps in DS region(s);
    //        measure[4]: number of OVERALL mismatches in SS region(s);
    //        measure[5]: number of OVERALL mismatches in DS region(s);
    //        measure[6]: percentage of OVERALL structure identity;
    //        measure[7]: percentage of OVERALL sequence identity;
    //        measure[8]: percentage of sequence identity from SS region(s) of the FIRST RNA;
    //        measure[9]: percentage of sequence identity from DS region(s) of the FIRST RNA;
    //-----------------------------------------------------------------------------------------
    public static int[] getMeasures(RNA aRNA, int[] aMap, RNA bRNA, int[] bMap)
     {
         int start1, end1, start2, end2;
         int[] region = HomoUtil.getAlignedRegion(aRNA, aMap);
         start1 = region[0];
         end1   = region[1];

         region = HomoUtil.getAlignedRegion(bRNA, bMap);
         start2 = region[0];
         end2   = region[1];

         int[] measure = new int[10];

         int TotalLen = 0;         // number of total columns
         while( start1<=end1 && start2<=end2) {
             TotalLen ++;
             if( aMap[start1] == start2 + 1) {            // one aligned column
                 measure[6] ++;

                 if( aRNA.getPairedBase(start1) == -1){   // alignment column in SS regoin
                     measure[0] ++;
                     String a = aRNA.getBase(start1);
                     String b = bRNA.getBase(start2);
                     if( a.equals(b) ){
                         measure[7] ++;
                         measure[8] ++;
                     } else
                         measure[4] ++;
                 }
                 else  {                                 // alignment column in DS region 
                     measure[1] ++;
                     String a = aRNA.getBase(start1);
                     String b = bRNA.getBase(start2);
                     if( a.equals(b) ){
                         measure[7] ++;
                         measure[9] ++;
                     } else
                         measure[5] ++;
                 }
                 start1 ++;
                 start2 ++;
             } else if( aMap[start1] > start2 + 1) {     // one gap from the first structure
                 if( bRNA.getPairedBase(start2) == -1)
                     measure[2] ++;
                 else
                     measure[3] ++;

                 start2 ++;
             } else if( bMap[start2] > start1 + 1) {     // one gap from the second structure
                 if( aRNA.getPairedBase(start1) == -1){
                     measure[0] ++;
                     measure[2] ++;
                 } else{
                     measure[1] ++;
                     measure[3] ++;
                 }
                 start1 ++;
             } else if( aMap[start1]==-1 && bMap[start2]==-1 ){   // both gaps
                 if( aRNA.getPairedBase(start1) == -1){
                     measure[0] ++;
                     measure[2] ++;
                 } else{
                     measure[1] ++;
                     measure[3] ++;
                 }
                 start1 ++;

                 if( bRNA.getPairedBase(start2) == -1)
                     measure[2] ++;
                 else
                     measure[3] ++;

                 start2 ++;
            } else {
                System.out.println("Fatal Error with the alignment information !");
                System.exit(1);
            }
        }

        if( start1 <= end1 ) {
            while( start1 <= end1 ) {
                TotalLen ++;
                if( aRNA.getPairedBase(start1) == -1){
                    measure[0] ++;
                    measure[2] ++;
                } else{
                    measure[1] ++;
                    measure[3] ++;
                }
                start1 ++;
            }
        } else if( start2 <= end2) {
            while( start2 <= end2) {
                TotalLen ++;
                if( bRNA.getPairedBase(start1) == -1)
                    measure[2] ++;
                else
                    measure[3] ++;

                start2 ++;
            }
        }

        measure[6] = measure[6]*100/TotalLen;
        measure[7] = measure[7]*100/TotalLen;

        if( measure[0] != 0)
            measure[8] = measure[8]*100/measure[0];
        else
            measure[8] = 0;

        if( measure[1] != 0)
            measure[9] = measure[9]*100/measure[1];
        else
            measure[9] = 0;

        return measure;
    }
}
