#ifndef __CORRECTED_MUTUAL_INFORMATION_SCORER__
#define __CORRECTED_MUTUAL_INFORMATION_SCORER__

#include <string>
#include <Physical.h>
#include <MutualInformationScorer.h>
#include <CharacterAlphabet.h>
#include <AlignmentColumn.h>
#include <Random.h>

using namespace std;

class CorrectedMutualInformationScorer : public MutualInformationScorer {

 public:

  CorrectedMutualInformationScorer(const CharacterAlphabet& _alphabet) : alphabet(_alphabet), correctionMode(true) { }

  virtual CorrectedMutualInformationScorer& operator = (const CorrectedMutualInformationScorer& other) {
    if (&other != this) {
      copy(other);
    }
    return *this;
  }

  /** returns (the error corrected) mutual information and an estimate of the small sample noise as a standard deviation */
  virtual Physical computeMutualInformation(const string& s1, const string& s2) const;

  /** computes the information of an alignment column */
  virtual Physical computeInformation(const string& s) const;

  /** return approximate correction term according to Basharin for an alphabet of size s and n characters */
  static double approximateCorrection(unsigned int s, unsigned int n);

  static double initialUncertainty(unsigned int n);

 private:

  /* PREDICATES */

  /** returns frequency of n'th alphabet character in alignment column (not counting gaps) */
  double frequency2(const AlignmentColumn& col, unsigned int n) const;

  /** returns frequency of character pair (n1, n2) in alignment column (not counting gaps) */
  double frequency2(const AlignmentColumn& col1, char c1,
		    const AlignmentColumn& col2, char c2) const;  

  /** returns frequency of character pair (n1, n2) in alignment column (also counting gaps) */
  double frequency(const AlignmentColumn& col1, char c1,
		   const AlignmentColumn& col2, char c2) const;

  /** Computes uncorrected uncertainty of an alignment column with respect to an alphabet */
  double uncorrectedUncertainty(const AlignmentColumn& col) const;

  /** Computes uncorrected uncertainty of an alignment column with respect to an alphabet.
   * Assumes that columns are already cleaned from gaps! 
   */
  double uncorrectedUncertainty(const AlignmentColumn& col1,
				const AlignmentColumn& col2) const;

  /** used for computation of variance of uncertainty.
   * According to Basharin 1959 
   */
  double uncorrectedUncertaintyVariance(const AlignmentColumn& col) const;
  
  /** Returns information of single column.
   * uses approximate correction term a la Basharin
   */
  Physical computeApproximateInformation(const AlignmentColumn& col) const;

  /** Returns information of single column.
   * Only use P log P  like Tom Schneider,
   * EXACT correction due to random sampling or approxamate correction depending on number of sequences
   */
  Physical computeInformation(const AlignmentColumn& col) const;

  /** returns (the error corrected) mutual information and an estimate of the small sample noise as a standard deviation.
   * Assumes that rows which contain at least one gap have been removed already from both alignment columns. 
   */
  Physical computeInformationClean(const AlignmentColumn& col1, 
				   const AlignmentColumn& col2) const;

  /** returns two string without mutual gaps or individual gaps */
  pair<string, string> cleanAliColumns(const string& s1Orig, const string& s2Orig) const;

  /** returns two string without mutual gaps or individual gaps */
  string cleanAliColumn(const string& s1Orig) const;


  void calehnb(long n, long gna, long gnc, long gng, long gnt, 
	  double* hg, double* ehnb, double* varhnb) const;


  /** generates random sequence. Does not changes size of string! */
  void generateRandomSequence(string& result,
			      Random& rnd) const;
  
  /** Computes uncorrected uncertainty of an alignment column with respect to an alphabet.
   * Assumes that columns are already cleaned from gaps! 
   */
  double uncorrectedUncertaintyVariance(const AlignmentColumn& col1,
					const AlignmentColumn& col2) const;

  /* MODIFIERS */

  virtual void copy(const CorrectedMutualInformationScorer& other) {
    alphabet = other.alphabet;
    correctionMode = other.correctionMode;
  }

  /* ATTRIBUTES */

  CharacterAlphabet alphabet;

  bool correctionMode;
  
};

#endif
