// --*- C++ -*------x---------------------------------------------------------
// $Id: Stem.h,v 1.7 2005/09/20 20:21:53 bindewae Exp $
//
// Class:           Stem
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            06/2004
//
// Description:     concept of RNA secondary structure stem
// 
// Reviewed by:     -
// -----------------x-------------------x-------------------x-----------------

#ifndef __STEM_H__
#define __STEM_H__

#include <iostream>
#include <debug.h>
#include <string>
#include <math.h>

// Includes


/** contains info about RNA secondary structure stem
    
@author Eckart Bindewald
@see    GA software of Shapiro group
@review - */

class Stem {
public:

  typedef int size_type;
  
  /* CONSTRUCTORS */
  
  Stem() : start(0), stop(0), length(0), energy(0.0) { }

  Stem(unsigned int _start, 
       unsigned int _stop, 
       unsigned int _length,
       double _energy, 
       const string& s1, 
       const string& s2) 
    : start(_start), 
      stop(_stop), length(_length),
      energy(_energy), sequence1(s1), sequence2(s2) {
    // PRECOND(isRnaMatch(s1, s2, true), exception);
 }

  Stem(unsigned int _start, 
       unsigned int _stop, 
       unsigned int _length)
    : start(_start), 
      stop(_stop), length(_length),
      energy(0.0) {
  }
  
  Stem(const Stem& orig) { copy(orig); }

  virtual ~Stem() { }

  /* OPERATORS */

  /** Assigment operator. */
  Stem& operator = (const Stem& orig) {
    if (&orig != this) {
      copy(orig);
    }
    return *this;
  }

  friend ostream& operator << (ostream& os, const Stem& rval);

  friend istream& operator >> (istream& is, Stem& rval);

  /* PREDICATES */

  /** Is current state valid? */
  virtual bool isValid() const {
    return (length > 0);
  }

  /** How big is object? */
  virtual unsigned int size() const {
    return length;
  }

  virtual int getStart() const { return start; }

  virtual int getStop() const { return stop; }

  virtual int getLength() const { return length; }

  virtual double getEnergy() const { return energy; }

  virtual const string& getSequence1() const { return sequence1; }

  virtual const string& getSequence2() const { return sequence2; }

  virtual bool isSimilar(const Stem& other, int slack) const;

  /** orders start and stop position */
  virtual int orderInt() const {
    return orderInt(start, stop);
  }
  
  /* MODIFIERS */
  
  virtual void clear() {
    *this = Stem();
  }
  
  virtual void setStart( int n)  {  start = n; }
  
  virtual void setStop( int n)  {  stop = n; }
  
  virtual void setLength( int n)  {  length = n; }
  
  virtual void setEnergy(double e)  {  energy = e; }
  
  virtual void setSequence1(const string& s) { sequence1 = s; }
  
  virtual void setSequence2(const string& s) { sequence2 = s; }
  
  /* STATIC */
  static int orderInt(int istart, int istop) {
    return istart * istart + istop + istop;
  }
   
  /** returns distance in matrix space between two 2d points */
  static double pointDistance(int x1, int y1,
			      int x2, int y2)
  {
    return sqrt(static_cast<double>((x1-x2) * (x1-x2) + (y1-y2) * (y1-y2)));
  }

protected:
  /* OPERATORS  */
  
  /* PREDICATES */
  
  /* MODIFIERS  */
  
  void copy(const Stem& other) {
    start = other.start;
    stop = other.stop;
    length = other.length;
    energy = other.energy;
    sequence1 = other.sequence1;
    sequence2 = other.sequence2;
  }

private:

  /* OPERATORS  */

  /* PREDICATES */

  /* MODIFIERS  */

private:

  /* PRIVATE ATTRIBUTES */

  int start;
  
  int stop;
  
  int length;
  
  double energy;
  
  string sequence1; // sequence starting at position start
  
  string sequence2; // sequence ending at position stop
  
};

inline
ostream& operator << (ostream& os, const Stem& rval)
{
  os << rval.start + 1 << " " << rval.stop + 1<< " " 
     << rval.length << " " << rval.energy << " ";
  if (rval.sequence1.size() > 0) {
    os << rval.sequence1 << " ";
  }
  else {
    os << "- ";
  }
  if (rval.sequence2.size() > 0) {
    os << rval.sequence2 << " ";
  }
  else {
    os << "- ";
  }
  return os;
}

inline
istream& operator >> (istream& is, Stem& rval)
{
  rval.clear();
  string seq1, seq2;
  is >> rval.start >> rval.stop >> rval.length >> rval.energy 
     >> seq1 >> seq2;
  if (seq1.compare(string("-")) != 0) {
    rval.setSequence1(seq1);
  }
  if (seq2.compare(string("-")) != 0) {
    rval.setSequence2(seq2);
  }
  return is;
}

inline 
bool operator < (const Stem& stemA, const Stem& stemB)
{
  if (stemA.getStart() == stemB.getStart()) {
    if (stemA.getStop() == stemB.getStop()) {
      return (stemA.getLength() < stemB.getLength());
    }
    else {
      return (stemA.getStop() < stemB.getStop());
    }
  }
  return (stemA.getStart() < stemB.getStart());
}

inline
bool 
Stem::isSimilar(const Stem& other, int slack) const
{
  return ((abs(getStart() - other.getStart()) <= slack)
	  && (abs(getStop() - other.getStop()) <= slack) );
}


#endif /* __STEM_H__ */

