#include <readPdbHetero.h>
#include <iomanip>
#include <Vector3D.h>
#include <StringTools.h>
#include <ctype.h>
#include <vectornumerics.h>
#include <Bond.h>
#include <PdbParser.h>

// #define DEBUG_HETERO

/*
   n: atom number offset, molnum: "residue number"
   chain : chain id
*/
void
writeHeteroRecord(ostream& os, const Atom& a, const string& molNameOrig,
		  unsigned int n, unsigned int molNum, char chain,
		  double val1, double val2)
{
  string aName = a.getName();
  if (aName.size() > 3) {
    aName = aName.substr(0,3);
  }
  string molName = PdbParser::pdbAdjustName(molNameOrig); // adjust to length 3
  ERROR_IF((a.getName().size() == 0) || (a.getName().size() > 100),
	   "Atom with invalid name found!", exception);
  ERROR_IF(molName.size() > 100, "Too long molecule name encountered!",
	   excecption);
  const int precision = 3;
  Vector3D v = a.getRelativePosition();
  //  os << "HETATM" << setw(5) << a.getNumber() + n << "  " << aName;
  string breakString = "  "; // two spaces
  string breakString2;
  if (isdigit(aName[0])) {
    breakString = " "; // only one space
  }
  os << "HETATM" << setw(5) << n << breakString << aName;
  if (isdigit(aName[0]) && (aName.size() < 4)) {
    printChar(os,' ',5-aName.size());
  }
  else {
    printChar(os,' ',4-aName.size());
  }
  unsigned old_prec = os.precision();
  ios::fmtflags old_flags = os.flags();
  os.setf(ios::fixed, ios::floatfield);

  os << molName << " " << chain << setw(4) << molNum
     << setw(9+precision) << setprecision(precision) << v.x()
     << setw(5+precision) << setprecision(precision) << v.y()
     << setw(5+precision) << setprecision(precision) << v.z() 
     << setw(4+2) << setprecision(2) << val1
     << setw(4+2) << setprecision(2) << val2
     << "           " << a.getChemicalElementChar();
  // << "  1.00  1.00";

  os.precision(old_prec);
  os.flags(old_flags);

//    os << "HETATM " << n << " " << a.getAtomName() << " " 
//         << molName << " " << molNum << " " 
//         << v.x() << " " << v.y() << " " << v.z() << " 1.0 1.0";
}

/*
   n: atom number offset, molnum: "residue number"
   chain : chain id
*/
void
writeHeteroDirRecord(ostream& os, const Atom& a, const string& molNameOrig,
		     unsigned int n, unsigned int molNum, char chain)
{
  PRECOND((n > 0) && (molNum > 0) , exception);
  string molName = PdbParser::pdbAdjustName(molNameOrig); // adjust to length 3
  string aName = a.getName();
  if (aName.size() > 3) {
    aName = aName.substr(0,3);
  }
  ERROR_IF((a.getName().size() == 0) || (a.getName().size() > 100),
	   "Atom with invalid name found!", exception);
  ERROR_IF(molName.size() > 100, "Too long molecule name encountered!",
	   excecption);
  const int precision = 3;
  Vector3D v = a.getDir1();

  string breakString = "  "; // two spaces
  string breakString2;
  if (isdigit(aName[0])) {
    breakString = " "; // only one space
  }
  os << "DIR1  " << setw(5) << n << breakString << aName;
  if (isdigit(aName[0]) && (aName.size() < 4)) {
    printChar(os,' ',5-aName.size());
  }
  else {
    printChar(os,' ',4-aName.size());
  }

  unsigned old_prec = os.precision();
  ios::fmtflags old_flags = os.flags();
  os.setf(ios::fixed, ios::floatfield);

  os << molName << " " << chain << setw(4) << molNum
     << setw(9+precision) << setprecision(precision) << v.x()
     << setw(5+precision) << setprecision(precision) << v.y()
     << setw(5+precision) << setprecision(precision) << v.z() 
     << "  1.00  1.00"
     << "           " << a.getChemicalElementChar();
  os.precision(old_prec);
  os.flags(old_flags);
  os << endl; 
  v = a.getDir2();
  os << "DIR2  " << setw(5) << n << "  " << aName;
  if (aName.size() < 4) {
    printChar(os,' ',4-aName.size());
  }
  old_prec = os.precision();
  old_flags = os.flags();
  os.setf(ios::fixed, ios::floatfield);

  os << molName << " " << chain << setw(4) << molNum
     << setw(9+precision) << setprecision(precision) << v.x()
     << setw(5+precision) << setprecision(precision) << v.y()
     << setw(5+precision) << setprecision(precision) << v.z() 
     << "  1.00  1.00";
  os.precision(old_prec);
  os.flags(old_flags);

//    os << "HETATM " << n << " " << a.getAtomName() << " " 
//         << molName << " " << molNum << " " 
//         << v.x() << " " << v.y() << " " << v.z() << " 1.0 1.0";
}

/** read HETATM records of PDB file */
void
readPdbHetero(istream& is, Mol& m, bool readHydrogens)
{
  const unsigned int precision = 3;
  bool validFlag = true;
  vector<string> words;
  unsigned int heteroCounter = 0;
  m.clear();
  while(validFlag && is) {
    string line = getLine(is);
    //    cout << "line: " << linectr++ << " " << line << endl;
    if ( (line.size() > 20) && (line.substr(17,3).find("HOH") == 0) ) {
      // cout << "readPdbHetero: Ignoring water molecule!" << endl;
      continue; // skip water molecule
    }
    // ignore hydrogens if asked too
    if ( (!readHydrogens) && (line[13] =='H') ) {
      continue;
    }
    //    cout << "getting tokens... " << endl;
    words = getTokens(line);
    //    cout << words.size() << " tokens found!" << endl;
    if (words.size() >= 8) {
      if (((words[0].find("HETATM") == 0) 
	   || (words[0].find("ATOM") == 0))
	  && (line.size() > 45) ){
	Atom a;
	double x = stod(line.substr(30, 5+precision));
	double y = stod(line.substr(35+precision, 5+precision));
	double z = stod(line.substr(40+(2*precision), 5+precision));
	Vector3D v(x,y,z);
	a.setPosition(v);
	string name = line.substr(12,4);
	name = removeFromString(name, ' '); // remove space
        char chainName = line[21];
        a.setChainName(chainName);
	a.setName(name);
	a.setAtomName(name);
	a.setChemicalElementChar(line[13]);
	m.push_back(a);
	//  	cout << "name: " << words[2] << " "
	//  	     << m[m.size()-1].getAtomName() << endl;
	// ASSERT(m[m.size()-1].getAtomName().find(words[2])==0, exception);
	++heteroCounter;
      }      
    }
    else if (words.size() > 0) {
      if ((words[0].find("END") == 0)
	  /* || (words[0].find("TER") == 0) */ // overread chain breaks
	  || (words[0].find("MASTER") == 0) ) {
	validFlag = false;
      }
	  
    }
    
  }
  // cout << heteroCounter << " atoms read!" << endl;
  POSTCOND(heteroCounter == m.size(),exception);
}

/** read HETATM records of PDB file */
void
readPdbMol(istream& is, Mol& m)
{
  const unsigned int precision = 3;
  bool validFlag = true;
  vector<string> words;
  unsigned int heteroCounter = 0;
  m.clear();
  while(validFlag && is) {
    string line = getLine(is);
    // cout << "line: " << line << endl;
    if ( (line.size() > 20) 
	 && (line.substr(17,3).find("HOH") == 0) ) {
      // cout << "readPdbHetero: Ignoring water molecule!" << endl;
      continue; // skip water molecule
    }
    //    cout << "getting tokens... " << endl;
    words = getTokens(line);
    //    cout << words.size() << " tokens found!" << endl;
    if (words.size() >= 8) {
      if (((words[0].find("ATOM") == 0) || (words[0].find("HETATM") == 0))
      && (line.size() > 45) ){
	Atom a;
	double x = stod(line.substr(30, 5+precision));
	double y = stod(line.substr(35+precision, 5+precision));
	double z = stod(line.substr(40+(2*precision), 5+precision));
	Vector3D v(x,y,z);
	a.setPosition(v);
	string name = line.substr(12,4);
	name = removeFromString(name, ' '); // remove space
        char chainName = line[21];
        a.setChainName(chainName);
	a.setName(name);
	a.setAtomName(name);
	a.setChemicalElementChar(line[13]);
	m.push_back(a);
	//  	cout << "name: " << words[2] << " "
	//  	     << m[m.size()-1].getAtomName() << endl;
	// ASSERT(m[m.size()-1].getAtomName().find(words[2])==0, exception);
	++heteroCounter;
      }      
    }
    else if (words.size() > 0) {
      if ((words[0].find("END") == 0)
	  || (words[0].find("TER") == 0)
	  || (words[0].find("CONECT") == 0)
	  /* || (words[0].find("TER") == 0) */ // overread chain breaks
	  || (words[0].find("MASTER") == 0) ) {
	validFlag = false;
      }
	  
    }
    
  }
  // cout << heteroCounter << " atoms read!" << endl;
  POSTCOND(heteroCounter == m.size(),exception);
}

/** read molecule ignoring hydrogen atoms */
void
readPdbHeteroNoH(istream& is, Vec<Atom>& m, char chain)
{
  const unsigned int precision = 3;
  bool validFlag = true;
  vector<string> words;
  unsigned int heteroCounter = 0;
  m.clear();
  while(validFlag && is) {
    string line = getLine(is);
    //    cout << "line: " << linectr++ << " " << line << endl;
    if ( (line.size() > 20) && ( (line[13] =='H')
	 || (line.substr(17,3).find("HOH") == 0) )
	 || ((chain != ' ') && (line[21] != chain) ) ) {
      // cout << "readPdbHetero: Ignoring water or hydrogen molecule!" << endl;
      continue; // skip water molecule or hydrogen
    }
    //    cout << "getting tokens... " << endl;
    words = getTokens(line);
    //    cout << "reading line" << line << endl;
    //    cout << words.size() << " tokens found!" << endl;
    if ((words.size() >= 8) && (line.size() > 45)) {
       if ((words[0].find("HETATM") == 0)
	   || (words[0].find("HETATM") == 0)) {
	 Atom a;
	 double x = stod(line.substr(30, 5+precision));
	 double y = stod(line.substr(35+precision, 5+precision));
	 double z = stod(line.substr(40+(2*precision), 5+precision));
	 Vector3D v(x,y,z);
	 a.setPosition(v);
	 string name = line.substr(12, 4);
	 name = removeFromString(name, ' '); // remove space
         char chainName = line[21];
         a.setChainName(chainName);
	 a.setName(name);
	 a.setAtomName(name);
	 a.setChemicalElementChar(line[13]);
	 string numString = line.substr(6,5);
	 numString = removeFromString(numString, ' ');
	 a.setNumber(stoi(numString));
	 m.push_back(a);
//  	 cout << "name: " << words[2] << " "
//  	      << m[m.size()-1].getAtomName() << endl;
	 // ASSERT(m[m.size()-1].getAtomName().find(words[2])==0, exception);
	 ++heteroCounter;
       }
       else if (    (words[0].find("DIR1") == 0)
		 || (words[0].find("DIR2") == 0) ) {
	 // cout << "found Dir keyword!" <<  endl;
	 // look for matching atom:
	 string numString = line.substr(6,5);
	 numString = removeFromString(numString, ' ');
	 int num = stoi(numString);
	 unsigned int found = m.size();
	 for (unsigned int i = 0; i < m.size(); ++i) {
	   if (m[i].getNumber() == num) {
	     found = i;
	     break;
	   }
	 }
	 if (found < m.size()) {
	   Vector3D dir;
	   dir.x(stod(line.substr(30, 5+precision)));
	   dir.y(stod(line.substr(35+precision, 5+precision)));
	   dir.z(stod(line.substr(40+(2*precision), 5+precision)));
	   if (words[0].find("DIR1") == 0) {
//  	     cout << "Setting Direction 1 of atom with number " 
//  		  << num << endl;
	     m[found].setDir1(dir);
	   }
	   else {
//  	     cout << "Setting Direction 2 of atom with number " 
//  		  << num << endl;
	     m[found].setDir2(dir);
	   }
	 }
	 else {
	   cout << "WARNING: Could not find atom with number " << num << endl;
	 }
      }
    }
    else if (words.size() > 0) {
      if ((words[0].find("END") == 0)
	  /* || (words[0].find("TER") == 0) */ // overread chain breaks!
	  || (words[0].find("MASTER") == 0) ) {
	validFlag = false;
      }
	  
    }
    
  }
  // cout << heteroCounter << " atoms read!" << endl;
//    POSTCOND((heteroCounter == m.size())
//  	   && (m.size() > 0),exception);
}


/** read molecule ignoring hydrogen atoms */
void
readPdbHeteroNoH(istream& is, Vec<Atom>& m, char chain,
		 Vec<double>& v1, Vec<double>& v2)
{
  const unsigned int precision = 3;
  bool validFlag = true;
  vector<string> words;
  unsigned int heteroCounter = 0;
  m.clear();
  while(validFlag && is) {
    string line = getLine(is);
    //    cout << "line: " << linectr++ << " " << line << endl;
    if ( (line.size() > 20) && ( (line[13] =='H')
	 || (line.substr(17,3).find("HOH") == 0) )
	 || ((chain != ' ') && (line[21] != chain) ) ) {
      // cout << "readPdbHetero: Ignoring water or hydrogen molecule!" << endl;
      continue; // skip water molecule or hydrogen
    }
    //    cout << "getting tokens... " << endl;
    words = getTokens(line);
    //    cout << "reading line" << line << endl;
    //    cout << words.size() << " tokens found!" << endl;
    if ((words.size() >= 8) && (line.size() > 45)) {
       if ((words[0].find("HETATM") == 0)
	 || (words[0].find("ATOM") == 0)) {
	 Atom a;
	 double x = stod(line.substr(30, 5+precision));
	 double y = stod(line.substr(35+precision, 5+precision));
	 double z = stod(line.substr(40+(2*precision), 5+precision));
	 double w1 = stod(line.substr(55,5));
	 double w2 = stod(line.substr(61,5));
	 ERROR_IF(!(isReasonable(w1) && isReasonable(w2)),
		  "Error parsing occupancy and B-factor column!", exception);
	 Vector3D v(x,y,z);
	 a.setPosition(v);
	 string name = line.substr(12, 4);
	 name = removeFromString(name, ' '); // remove space
         char chainName = line[21];
         a.setChainName(chainName);
	 a.setName(name);
	 a.setAtomName(name);
	 a.setChemicalElementChar(line[13]);
	 string numString = line.substr(6,5);
	 numString = removeFromString(numString, ' ');
	 a.setNumber(stoi(numString));
	 m.push_back(a);
	 v1.push_back(w1);
	 v2.push_back(w2);
//  	 cout << "name: " << words[2] << " "
//  	      << m[m.size()-1].getAtomName() << endl;
	 // ASSERT(m[m.size()-1].getAtomName().find(words[2])==0, exception);
	 ++heteroCounter;
       }
       else if (    (words[0].find("DIR1") == 0)
		 || (words[0].find("DIR2") == 0) ) {
	 // cout << "found Dir keyword!" <<  endl;
	 // look for matching atom:
	 string numString = line.substr(6,5);
	 numString = removeFromString(numString, ' ');
	 int num = stoi(numString);
	 unsigned int found = m.size();
	 for (unsigned int i = 0; i < m.size(); ++i) {
	   if (m[i].getNumber() == num) {
	     found = i;
	     break;
	   }
	 }
	 if (found < m.size()) {
	   Vector3D dir;
	   dir.x(stod(line.substr(30, 5+precision)));
	   dir.y(stod(line.substr(35+precision, 5+precision)));
	   dir.z(stod(line.substr(40+(2*precision), 5+precision)));
	   if (words[0].find("DIR1") == 0) {
//  	     cout << "Setting Direction 1 of atom with number " 
//  		  << num << endl;
	     m[found].setDir1(dir);
	   }
	   else {
//  	     cout << "Setting Direction 2 of atom with number " 
//  		  << num << endl;
	     m[found].setDir2(dir);
	   }
	 }
	 else {
	   cout << "WARNING: Could not find atom with number " << num << endl;
	 }
      }
    }
    else if (words.size() > 0) {
      if ((words[0].find("END") == 0)
	  /* || (words[0].find("TER") == 0) */ // overread chain breaks!
	  || (words[0].find("MASTER") == 0) ) {
	validFlag = false;
      }
	  
    }
    
  }
  // cout << heteroCounter << " atoms read!" << endl;
//    POSTCOND((heteroCounter == m.size())
//  	   && (m.size() > 0),exception);
}


/** read molecule ignoring hydrogen atoms */
void
readPdbHetero(istream& is, Vec<Atom>& m, char chain,
	      Vec<double>& v1, Vec<double>& v2, bool readHydrogens)
{
  const unsigned int precision = 3;
  bool validFlag = true;
  vector<string> words;
  unsigned int heteroCounter = 0;
  m.clear();
  while(validFlag && is) {
    string line = getLine(is);
    //    cout << "line: " << linectr++ << " " << line << endl;
    if ( (line.size() > 20)
	 || (line.substr(17,3).find("HOH") == 0) 
	 || ((chain != ' ') && (line[21] != chain) ) ) {
      // cout << "readPdbHetero: Ignoring water or hydrogen molecule!" << endl;
      continue; // skip water molecule or hydrogen
    }
    // ignore hydrogens if asked too
    if ( (!readHydrogens) && (line[13] =='H') ) {
      continue;
    }
    //    cout << "getting tokens... " << endl;
    words = getTokens(line);
    //    cout << "reading line" << line << endl;
    //    cout << words.size() << " tokens found!" << endl;
    if ((words.size() >= 8) && (line.size() > 45)) {
       if ((words[0].find("HETATM") == 0) 
	 || (words[0].find("ATOM") == 0)) {
	 Atom a;
	 double x = stod(line.substr(30, 5+precision));
	 double y = stod(line.substr(35+precision, 5+precision));
	 double z = stod(line.substr(40+(2*precision), 5+precision));
	 double w1 = stod(line.substr(55,5));
	 double w2 = stod(line.substr(61,5));
	 ERROR_IF(!(isReasonable(w1) && isReasonable(w2)),
		  "Error parsing occupancy and B-factor column!", exception);
	 Vector3D v(x,y,z);
	 a.setPosition(v);
	 string name = line.substr(12, 4);
	 name = removeFromString(name, ' '); // remove space
         char chainName = line[21];
         a.setChainName(chainName);       
	 a.setName(name);
	 a.setAtomName(name);
	 a.setChemicalElementChar(line[13]);
	 string numString = line.substr(6,5);
	 numString = removeFromString(numString, ' ');
	 a.setNumber(stoi(numString));
	 m.push_back(a);
	 v1.push_back(w1);
	 v2.push_back(w2);
//  	 cout << "name: " << words[2] << " "
//  	      << m[m.size()-1].getAtomName() << endl;
	 // ASSERT(m[m.size()-1].getAtomName().find(words[2])==0, exception);
	 ++heteroCounter;
       }
       else if (    (words[0].find("DIR1") == 0)
		 || (words[0].find("DIR2") == 0) ) {
	 // cout << "found Dir keyword!" <<  endl;
	 // look for matching atom:
	 string numString = line.substr(6,5);
	 numString = removeFromString(numString, ' ');
	 int num = stoi(numString);
	 unsigned int found = m.size();
	 for (unsigned int i = 0; i < m.size(); ++i) {
	   if (m[i].getNumber() == num) {
	     found = i;
	     break;
	   }
	 }
	 if (found < m.size()) {
	   Vector3D dir;
	   dir.x(stod(line.substr(30, 5+precision)));
	   dir.y(stod(line.substr(35+precision, 5+precision)));
	   dir.z(stod(line.substr(40+(2*precision), 5+precision)));
	   if (words[0].find("DIR1") == 0) {
//  	     cout << "Setting Direction 1 of atom with number " 
//  		  << num << endl;
	     m[found].setDir1(dir);
	   }
	   else {
//  	     cout << "Setting Direction 2 of atom with number " 
//  		  << num << endl;
	     m[found].setDir2(dir);
	   }
	 }
	 else {
	   cout << "WARNING: Could not find atom with number " << num << endl;
	 }
      }
    }
    else if (words.size() > 0) {
      if ((words[0].find("END") == 0)
	  /* || (words[0].find("TER") == 0) */ // overread chain breaks!
	  || (words[0].find("MASTER") == 0) ) {
	validFlag = false;
      }
	  
    }
    
  }
  // cout << heteroCounter << " atoms read!" << endl;
//    POSTCOND((heteroCounter == m.size())
//  	   && (m.size() > 0),exception);
}

/** looks which atom type name matches the namelist. Returns index. */
unsigned int
findTypeId(string name, 
	   const Vec<string>& nameList)
{
  upperCase(name);
  for (unsigned int i = 0; i < nameList.size(); ++i) {
    string tmpName = nameList[i];
    upperCase(tmpName);
    if (isEqual(name, tmpName)) {
      return i;
    }
  }
  return 0; // nameList.size(); // nothing found
}

/** reads Sybyl mol2 file, and defines atoms, charges, bonds, bond-orders, ligandTypes of molecule */
void
readMol2Hetero(istream& is, Vec<Atom>& m, 
	       Vec<double>& charges, 
	       Vec<Bond>& bonds,
	       Vec<unsigned int>& bondOrders,
               const Vec<string>& ligandTypeNames,
               Vec<unsigned int>& ligandTypes,
               string& molName)
{
  bool validFlag = true;
  Bond bond;
  // unsigned int linectr = 0;
  string line;
  vector<string> words;
  unsigned int heteroCounter = 0;
  m.clear();
  bonds.clear();
  bondOrders.clear();
  charges.clear();
  ligandTypes.clear();
  molName.clear();
  bool molNameFlag = false;
  while(is) {
    line = getLine(is);
#ifdef DEBUG_HETERO
     cout << "readMol2Hetero(header): line: " << line << endl;
#endif
    if (molNameFlag) {
      molName = trimWhiteSpaceFromString(line); // trim leading and trailing white space
      molNameFlag = false;
    }
    else if ((line.size() > 0) && (line[0] == '@')) {
      if (line.find("MOLECULE") < line.size()) {
        molNameFlag = true; // next line will be name of molecule
      }
      else if (line.find("ATOM") < line.size()) {
        break;
      }
    }
  }
  if (!is) {
    cout << "Could not find start of ATOM section in mol2 file!" << endl;
    return; // reached end of file before any molecule data was read
  }
  double x, y, z;
  while(validFlag && is) {
    line = getLine(is);
#ifdef DEBUG_HETERO
    cout << "readMol2Hetero: line: " << line << endl;
#endif
    if ( (line.size() > 0) 
	 && (line[0] == '@') ) {
      break;
    }
    words = getTokens(line);
#ifdef DEBUG_HETERO
    cout << words.size() << " tokens found!" << endl;
#endif
    if (words.size() >= 6) {
	Atom a;
	a.setName(words[1]);
        // workaround for weird cases like atom names "N A" as in certein heme molecules:
        if (!(isdigit(words[2][0]) || (words[2][0] == '-')
              || (words[2][0] == '.') ) ) {
          a.setName(words[1] + words[2]);
          for (unsigned int i = 3; i < words.size(); ++i) {
            words[i-1] = words[i];
          }
        }
	x = stod(words[2]);
	y = stod(words[3]);
	z = stod(words[4]);
	Vector3D v(x,y,z);
	a.setPosition(v);
	if (words[1][0] != '*') {
	  a.setName(words[1]);
	  a.setAtomName(words[1]);
	}
	else {
	  a.setName(words[5]);
	  a.setAtomName(words[5]);
	}
	a.setChemicalElementChar(words[5][0]);
	m.push_back(a);
        ligandTypes.push_back(findTypeId(words[5], ligandTypeNames));
	if (words.size() >= 9) {
	  double charge = stod(words[8]);
	  // check for plausibility:
	  if ((charge <= 3.0) && (charge >= -3.0)) {
	    charges.push_back(charge);
	  }
	  else {
	    charges.push_back(0.0);
	  }
	}
	else {
	  charges.push_back(0.0);
	}
#ifdef DEBUG_HETERO
        cout << "name: " << words[2] << " "
	     << m[m.size()-1].getAtomName() << endl;
#endif
	// ASSERT(m[m.size()-1].getAtomName().find(words[2])==0, exception);
	++heteroCounter;
    }
    if (line[0] == '@') {
      validFlag = false;
      break;
    }
  }
  while(is) {
    if ((line.size() > 0) && (line[0] == '@')
	&& (line.find("BOND") < line.size())) {
      validFlag = true;
      break;
    }
    line = getLine(is);
  }
  while (validFlag && is) {
    line = getLine(is);
#ifdef DEBUG_HETERO
    cout << "reading line: " << line << endl;
#endif
    if (line.size() == 0) {
      validFlag = false;
      break;
    }
    words = getTokens(line);
    if ((words.size() >= 3)
        && isdigit(words[1][0]) && isdigit(words[2][0])) {
      bond.first = stoui(words[1]);
      bond.second = stoui(words[2]);
      ERROR_IF(bond.first < 1,
	       "Illegal bond index 1 found!", exception);
      ERROR_IF(bond.second < 1,
	       "Illegal bond index 2 found!", exception);
      (bond.first)--; // internal counting starts from zero!
      (bond.second)--; // internal counting starts from zero!
      bonds.push_back(bond);
      if (words.size() >= 4) {
        bondOrders.push_back(stringToBondOrder(words[3]));
	/* 
           if ((words[3].size() == 1) && isdigit(words[3][0]) )  {
           bondOrders.push_back(stoui(words[3]));
           }
           // most likely "ar" type
           else if ((words[3].compare(string("ar")) == 0)
           || (words[3].compare(string("am")) == 0)){
           bondOrders.push_back(4); // special flag for aromatic
           }
           else {
           bondOrders.push_back(4); // unknown type
           }
        */
      }
      else {
	bondOrders.push_back(1);  // default: single bond
      }
    }
    else { 
      // if (line[0] == '@') {
      // as soon as there is a bad line in the bond block, the block is considered finished
      validFlag = false;
      break;
    }
    /* 
       else {
       ERROR("Could not interpret mol2 Bond line: "
       + line, exception);
       }
    */
  }
#ifdef DEBUG_HETERO
  cout << heteroCounter << " atoms read!" << endl;
#endif
  ERROR_IF(heteroCounter != m.size(),
           "Hetero counter not equal molecule size!", exception);
  ERROR_IF(bondOrders.size() != bonds.size(), 
           "Number of bond orders not equal to number of bonds!", exception);
  ERROR_IF(ligandTypes.size() != m.size(), 
           "Number of ligand types not equal number of atoms!", exception);
  ERROR_IF(charges.size() != m.size(), 
           "Number of ligand charges not equal number of atoms!", exception);
}

void
readMol2Hetero(istream& is, Vec<Atom>& m, 
	       Vec<double>& charges, 
	       Vec<Bond>& bonds,
	       Vec<unsigned int>& bondOrders,
               string& molName)
{
  Vec<string> ligandTypeNames;
  Vec<unsigned int> ligandTypes;
  readMol2Hetero(is, m, charges, bonds, bondOrders, ligandTypeNames, ligandTypes, molName);
}

/** ignore reading of bond information, otherwise like above readMol2Hetero */
void
readMol2Hetero(istream& is, Vec<Atom>& m)
{
  Vec<double> charges;
  Vec<Bond> bonds;
  Vec<unsigned int> bondOrders;
  string molName;
  readMol2Hetero(is, m, charges, bonds, bondOrders, molName);
}

/**
 * reads complete molecule information
 */
MolAnnotation
readMol2Hetero(istream& is, const Vec<string>& atomTypeNames) {
  MolAnnotation molecule;
  Mol m;
  Vec<Bond> bonds;
  Vec<double> charges;
  Vec<unsigned int> bondOrders;
  Vec<unsigned int> atomTypes;
  string molName;
  readMol2Hetero(is, m, charges, bonds, bondOrders, atomTypeNames, atomTypes, molName);
  molecule.setMol(m);
  molecule.setCharges(charges);
  molecule.setBonds(bonds);
  molecule.setBondOrders(bondOrders);
  molecule.setAtomTypes(atomTypes);
  molecule.setName(molName);
  return molecule;
}

/**
 * writes Sybyl mol2 file for defined molecule. List of atomTypeNames has to be specified
 */
void
writeMol2Hetero(ostream& os, 
		const Vec<Atom>& m, 
                const Vec<double>& charges,
                const Vec<Bond>& bonds,
                const Vec<unsigned int>& bondOrders,
                const Vec<string>& ligandTypeNames,
                const Vec<unsigned int>& ligandTypes,
		const string& name)
{
  PRECOND((bonds.size() == bondOrders.size()), exception);
  unsigned int precision = 4;
  unsigned old_prec = os.precision();
  ios::fmtflags old_flags = os.flags();
  os.setf(ios::fixed, ios::floatfield);
  os << "@<TRIPOS>MOLECULE" << endl;
  os << name << endl;
  os << "   " << m.size() << "   " << bonds.size() << "     0     0     0" 
     << endl;
  os << "SMALL" << endl;
  os << "SCREENDOCK" << endl << endl << endl;
  os << "@<TRIPOS>ATOM" << endl;
  for (unsigned int i = 0; i < m.size(); ++i) {
    const Vector3D& v = m[i].getPosition();
    string s1 = uitos(i+1);
    ERROR_IF(s1.size() > 7, "Illegal size of number.", exception);
    printChar(os, ' ', 7 - s1.size());
    string s2 = m[i].getAtomName();
    if (s2.size() > 7) {
      s2 = s2.substr(0,7);
    }
    os << s1 << " " << s2; // << " "
    printChar(os, ' ' , 10 - m[i].getAtomName().size());
    string atTypeName;
    if ((ligandTypeNames.size() > 0) && (i < ligandTypes.size())) {
      ASSERT(ligandTypes[i] < ligandTypeNames.size(), exception);
      atTypeName = ligandTypeNames[ligandTypes[i]];
    }
    else {
      atTypeName = m[i].getAtomName(); // .getChemicalElementChar(); 
    }
    double charge = 0.0;
    if (i < charges.size()) {
      charge = charges[i];
    }
    os << setw(4 + precision) << setprecision(precision) << v.x() << "  "
       << setw(4 + precision) << setprecision(precision) << v.y() << "  "
       << setw(4 + precision) << setprecision(precision) << v.z() << " "
       << atTypeName << "         1" << " MOL   "
       << setw(4 + precision) << setprecision(precision) << charge << endl;
  }
  os << "@<TRIPOS>BOND" << endl;
  for (unsigned int i = 0; i < bonds.size(); ++i) {
    os << "    " << i + 1 << "   " << bonds[i].first + 1
       << "    " << bonds[i].second + 1 << "  " << bondOrderToString(bondOrders[i]) << endl;
  }
  os << endl; // empty line after bonds info
  os.precision(old_prec);
  os.flags(old_flags);

}

/**
 * writes Sybyl mol2 file for defined molecule. List of atomTypeNames has to be specified
 */
void
writeMol2Hetero(ostream& os, 
                const MolAnnotation& molecule,
                const Vec<string>& atomTypeNames) 
{
  writeMol2Hetero(os, molecule.getMol(), molecule.getCharges(),
                  molecule.getBonds(), molecule.getBondOrders(),
                  atomTypeNames, molecule.getAtomTypes(),
                  molecule.getName());
}

static
string
alignRight(const string& s, unsigned int n) {
  string result;
  if (s.size() > n) {
    result = s.substr(s.size() - n, n); // not great solution, cutting off digits
  }
  else if (s.size() < n) {
    result = string(n - s.size(), ' ') + s;
  }
  else {
    result = s;
  }
  POSTCOND(result.size() == n, exception);
  return result;
}

static
string
alignLeft(const string& s, unsigned int n) {
  string result;
  if (s.size() > n) {
    result = s.substr(s.size() - n, n); // not great solution, cutting off digits
  }
  else if (s.size() < n) {
    result = s + string(n - s.size(), ' ');     
  }
  else {
    result = s;
  }
  POSTCOND(result.size() == n, exception);
  return result;
}

/**
 * writes appropriate CONECT entries for pdb file and molecule 
 */
void
writePdbBonds(ostream& os,
              unsigned int startIndex, 
              const Vec<Bond>& bonds) {
  Vec<unsigned int> bondedAtoms = getUniqBondedList(bonds);
  for (unsigned int i = 0; i < bondedAtoms.size(); ++i) {
    string result = "CONECT";
    unsigned int n = bondedAtoms[i]; // this atom
    Vec<unsigned int> neighbors = getNeighbours(n, bonds);
    if (neighbors.size() == 0) {
      continue;
    }
    result = result + alignRight(uitos(n + startIndex), 5);
    for (unsigned int j = 0; j < neighbors.size(); ++j) {
      result = result + alignRight(uitos(neighbors[j] + startIndex), 5);
    }
    result = PdbParser::pdbAdjustedLine(result); // add spaces if necessary
    os << result << endl;
  }
}

/* write PDB file */
void
writePdbHetero(ostream& os,
	       const Vec<Atom>& m,
	       const string& molName,
	       unsigned int startCount,
	       unsigned int molNum, char chain)
{
  ERROR_IF(m.size() > 1000000, "Unlikely molecule size encountered!",
	   exception);
  //  os << " size of molecule: " << m.size() << endl;
  for (unsigned int i = 0; i < m.size(); ++i) {
    writeHeteroRecord(os,m[i], molName, startCount + i, molNum, chain);
    os << endl;
  }

}

/** write PDB file, with arbitrary values in Occupancy and B-factor column */
void
writePdbHetero(ostream& os,
	       const Vec<Atom>& m,
	       const string& molName,
	       unsigned int startCount,
	       unsigned int molNum, char chain,
	       const Vec<double>& v1,
	       const Vec<double>& v2)
{
  PRECOND((v1.size() == m.size()) && (v2.size() == m.size()),
	  exception);
  ERROR_IF(m.size() > 10000, "Unlikely molecule size encountered!",
	   exception);
  //  os << " size of molecule: " << m.size() << endl;
  for (unsigned int i = 0; i < m.size(); ++i) {
    writeHeteroRecord(os,m[i], molName, startCount + i, molNum, chain,
		      v1[i], v2[i]);
    os << endl;
  }
}

/* write non-amino acid molecule to pdb file */
void
writePdbHeteroOrientations(ostream& os,
			   const Vec<Atom>& m,
			   const string& molName,
			   unsigned int startCount,
			   unsigned int molNum, char chain)
{
  ERROR_IF(m.size() > 10000, "Unlikely molecule size encountered!",
	   exception);
  //  os << " size of molecule: " << m.size() << endl;
  for (unsigned int i = 0; i < m.size(); ++i) {
    writeHeteroRecord(os,m[i], molName, startCount + i, molNum, chain);
    os << endl;
    if (m[i].size() > 1) {
      writeHeteroDirRecord(os,m[i], molName, startCount + i, molNum, chain);
    }
    os << endl;
  }

}


/* write non-amino acid molecule to pdb file */
void
writePdbHeteroOrientations(ostream& os,
			   const Vec<Atom>& mol,
			   const string& molName,
			   unsigned int startCount,
			   unsigned int molNum, char chain,
			   const Vec<Vec<int> >& match)
{
  ERROR_IF(mol.size() > 10000, "Unlikely molecule size encountered!",
	   exception);
  //  os << " size of molecule: " << m.size() << endl;
  for (unsigned int i = 0; i < match.size(); ++i) {
    Atom a = mol[match[i][0]];
    if (match[i].size() > 1) {
      const Atom& b = mol[match[i][1]];
      const Atom& c = mol[match[i][2]];
      a.setDir1(b.getPosition() - a.getPosition());
      a.setDir2(c.getPosition() - a.getPosition());
    }
    writeHeteroRecord(os,a, molName, startCount, molNum, chain);
    os << endl;
    if ((a.getDir1().length() > 0.0)
	&& (a.getDir1().length() > 0.0)) {
      writeHeteroDirRecord(os,a, molName, startCount, molNum, chain);
      os << endl;
    }
    ++startCount;
    for (unsigned int j = 1; j < match[i].size(); ++j) {
      writeHeteroRecord(os,mol[match[i][j]], molName, 
			      startCount, molNum, chain);
      os << endl;
      ++startCount;
    }
    ++molNum;
  }

}


/* write PDB file with COLOR records defining a user color scheme */
void
writeMolColor(ostream& os, const Mol& mol, 
	      const Vec<Vector3D>& colorlist,
	      const string& molName,
	      unsigned int atNumStart,
	      unsigned int aanumber,
	      char chain)
{
  string aaname = molName;
  for (unsigned int i = 0; i < mol.size(); ++i) {
    const Atom& atom = mol[i];
    string atname = atom.getName();
    // Handle codes like "1HG3" properly.
    if (atname[0] != '1' && atname[0] != '2' && atname[0] != '3')
      {
	atname = string(" ") + atname;
      }
    while (atname.size() < 4)
      {
	atname += " ";
      }
    Vector3D pos = atom.getPosition();
    const Vector3D& colors = colorlist[i];
    const unsigned int precision = 3;
    unsigned old_prec = os.precision();
      ios::fmtflags old_flags = os.flags();
      os.setf(ios::fixed, ios::floatfield);
      os << "COLO" << setw(7) << atNumStart + i << " " 
	 << atname << " " << aaname << " " << chain << setw(4) << aanumber
	 << setw(9+precision) << setprecision(precision) << colors.x()
	 << setw(5+precision) << setprecision(precision) << colors.y()
	 << setw(5+precision) << setprecision(precision) << colors.z()
	 << "  1.00  1.00" << endl;
      os << "ATOM" << setw(7) << atNumStart + i << " " 
	 << atname << " " << aaname << " " << chain << setw(4) << aanumber 
	 << setw(9+precision) << setprecision(precision) << pos.x()
	 << setw(5+precision) << setprecision(precision) << pos.y()
	 << setw(5+precision) << setprecision(precision) << pos.z() 
	 << "  1.00  1.00" << endl;      
      os.precision(old_prec);
      os.flags(old_flags);
  }
}

/* write PDB file with COLOR records defining a user color scheme */
void
writeMolBFactors(ostream& os, const Mol& mol, 
	      const Vec<double>&  bfactors,
	      const string& molName,
	      unsigned int atNumStart,
	      unsigned int aanumber, char chain)
{
  string aaname = molName;
  for (unsigned int i = 0; i < mol.size(); ++i) {
    const Atom& atom = mol[i];
    string atname = atom.getName();
    // Handle codes like "1HG3" properly.
    if (atname[0] != '1' && atname[0] != '2' && atname[0] != '3') {
		atname = string(" ") + atname;
    }
    while (atname.size() < 4) {
		atname += " ";
    }
    Vector3D pos = atom.getPosition();
    double bfac = bfactors[i];
	 if (bfac < 0.0) {
		bfac = 0.0;
		}
    const unsigned int precision = 3;
    unsigned old_prec = os.precision();
    ios::fmtflags old_flags = os.flags();
    os.setf(ios::fixed, ios::floatfield);
    os << "ATOM" << setw(7) << atNumStart + i << " " 
	 	<< atname << " " << aaname << " " << chain
		<< setw(4) << aanumber 
	 	<< setw(9+precision) << setprecision(precision) << pos.x()
	 	<< setw(5+precision) << setprecision(precision) << pos.y()
	 	<< setw(5+precision) << setprecision(precision) << pos.z() 
	 	<< "  1.00" << setw(6) << setprecision(2) << bfac << endl;      
    os.precision(old_prec);
    os.flags(old_flags);
  }
}



/** Read CONNECT table of PDB file */
Vec<Vec<int> >
readConnections(istream& is)
{ 
  Vec<Vec<int> > result;
  string line = getLine(is);
  line = line.substr(0,40);
  Vec<string> words = getTokens(line);
//    cout << "Processing line: " << line << endl 
//         << words << endl;
  while (is && (words.size() > 0)
    && (words[0].compare("CONECT") == 0)) {
//      cout << "Processing line: " << line << endl 
//  	 << words << endl;
    Vec<int> tmpInts;
    tmpInts.clear();
    for (unsigned int i = 1; i < words.size(); ++i) {
      tmpInts.push_back(stoi(words[i]) -1);
    }
    result.push_back(tmpInts);   
    line = getLine(is);
    line = line.substr(0,40);
    words = getTokens(line);
  }
  return result;
}

/* write single line of mol block in MDL mol file (V2000) */
void
writeMolfileLine(ostream& os, const Atom& a)
{
  Vector3D v = a.getPosition();
  unsigned old_prec = os.precision();
  ios::fmtflags old_flags = os.flags();
  os.setf(ios::fixed, ios::floatfield);
  cout << setw(10) << setprecision(4) << v.x()
       << setw(10) << setprecision(4) << v.y()
       << setw(10) << setprecision(4) << v.z();
  cout << " ";
  cout << getChemicalElement(a);
  cout << "   0  0  0  0  0  0  0  0  0  0  0  0";
  os.precision(old_prec);
  os.flags(old_flags);
}

/* write MDL mol file (V2000) */
void
writeMolfile(ostream& os, const Mol& m, string idName)
{
  cout << endl;
  cout << "  -ISIS-  " << idName << endl;
  cout << endl;
  cout << " " << m.size() << "  0  0  0  0  0  0  0  0  0999 V2000" << endl;  
  for (unsigned int i = 0; i < m.size(); ++i) {
    writeMolfileLine(os, m[i]);
    cout << endl;
  }

  cout << "M  END" << endl;
} 

/* interpret mol file line (V2000) */
Atom
interpretMolfileLine(const string& line)
{
  Vec<string> words = getTokens(line);
  ERROR_IF(words.size() < 4, "At least 4 words expected in Mol file line.", 
	   exception);
  Atom a;
  Vector3D pos;
  char elementChar = words[3][0];
  a.setChemicalElementChar(elementChar);
  a.setAtomName(words[3]);
  pos.x(stod(words[0]));
  pos.y(stod(words[1]));
  pos.z(stod(words[2]));
  a.setPosition(pos);
  return a;
}

/* interpret mol file line (V2000) */
Atom
interpretSdfMolfileLine(istream& is)
{
  double xs, ys, zs;
  string atomNameString;
  
  is >> xs >> ys >> zs >> atomNameString;
  ERROR_IF(!is, "Error reading atom coordinates!", exception);
  Atom a;
  Vector3D pos;
  a.setAtomName(atomNameString);
  a.setName(atomNameString);
  pos.x(xs);
  pos.y(ys);
  pos.z(zs);
  a.setPosition(pos);
  // read until line of zeros is over:
  string word;
  do {
    is >> word;
    // cout << "trail word: " << word << " ";
  }
  while (word.compare(string("0")) == 0);
  return a;
}

/* interpret mol file line (V2000) */
Atom
readSdfAtom(istream& is)
{
  double xs, ys, zs;
  string atomNameString;  
  string dummy;
  is >> xs >> ys >> zs >> atomNameString;
  ERROR_IF(!is, "Error reading atom coordinates!", exception);
  Atom a;
  Vector3D pos;
  a.setAtomName(atomNameString);
  a.setName(atomNameString);
  pos.x(xs);
  pos.y(ys);
  pos.z(zs);
  a.setPosition(pos);
  // read until line of zeros is over:
  string word;
  for (unsigned int i = 0; i < 10; ++i) {
    is >> word;
  }
  return a;
}

void
readSdfBond(istream& is,
            Bond& bond,
            unsigned int& bondOrder) 
{
  string dummy;
  is >> bond.first >> bond.second >> bondOrder >> dummy >> dummy >> dummy;
  ERROR_IF((bond.first == 0) || (bond.second == 0), 
           "Invalid atom index found in readSdfBond!", exception);
  --bond.first; // convert to internal counting
  --bond.second;
}

/* read MDL mol file (V2000) */
void
readMolfile(istream& is, Mol& m)
{
  // read lines until it reaches keyword V2000:
  Vec<string> words;
  string line;
  do {
    line = getLine(is);
    VERBOSE3 << "reading line: " << line << endl;
    ERROR_IF(!is, 
	 "Error reading head of Mol file! Could not find keyword V2000" ,
	     exception);
    words = getTokens(line);
  } while ((words.size() == 0)
	   || (words[words.size() -1].compare(string("V2000")) != 0));
  ASSERT(words.size() > 0, exception);
  VERBOSE3 << "Found header line: " << line << endl;
  unsigned int nAtoms = stoui(words[0]);
  m = Mol(nAtoms);
  for (unsigned int i = 0; i < m.size(); ++i) {
    m[i] = interpretMolfileLine(getLine(is));
  }
  // read until END record found
  do {
    line = getLine(is);
    VERBOSE3 << "reading line(2): " << line << endl;
    ERROR_IF(!is, 
	 "Error reading head of Mol file! Could not find keyword V2000" ,
	     exception);
    words = getTokens(line);
  } while ((words.size() < 2)
	   || (words[0].compare(string("M")) != 0)
	   || (words[1].compare(string("END")) != 0) );
  ERROR_IF(m.size() < 1, "No atoms read!", exception);
} 


/* read mol header from SDF file */
bool
readSdfMolHeader(istream& is,
		 unsigned int& numEntries,
                 unsigned int& numBonds,
		 string& versionString)
{
  //  cout << "reading sdf header!" << endl;
  Vec<string> words;
  string newWord;
  string versionStart = "V2"; 
  do {
    is >> newWord;
    // cout << "read new word: " << newWord << " ";
    words.push_back(newWord);
    if (!is) {
      return false;
    }
  }
  while (newWord.find(versionStart) > 0);
  versionString = newWord;
  if (words.size() >= 11) {
    numEntries = stoui(words[words.size() - 11]);
    numBonds = stoui(words[words.size() - 10]);
  }
  else {
    return false;
  }
  return true;
}

/* read MDL mol file (V2000) as part of SDF file 
   (using example from Shoichet) */
bool
readSdfMolfile(istream& is, Mol& m)
{
  // read lines until it reaches keyword V2000:
  unsigned int nAtoms = 0;
  unsigned int nBonds = 0;
  string word;
  string line, versionString;
  string atomNameString;
  bool success = readSdfMolHeader(is, nAtoms, nBonds, versionString);
  if (!success) {
    return success;
  }
  m = Mol(nAtoms);
  for (unsigned int i = 0; i < m.size(); ++i) {
    m[i] = interpretSdfMolfileLine(is);
  }
  // read until END record found
  do {
    is >> word;
    // cout << "read word: " << word << " ";
    ERROR_IF(!is, 
	     "Error reading end of Mol file! Could not find END keyword",
	     exception);
  } while (word.compare(string("END")) != 0);
  if (m.size() < 1) {
    return false;
  }
  return true;
} 

/* read MDL mol file (V2000) as part of SDF file 
   (using example from Shoichet) */
bool
readSdfMolfile(istream& is, MolAnnotation& molAnno)
{
  cout << "Starting readSdfMolfile(istream&, MolAnnotation&)" << endl;
  Mol m;
  Vec<Bond> bonds;
  Vec<unsigned int> bondOrders;
  // read lines until it reaches keyword V2000:
  molAnno.clear();
  unsigned int nAtoms = 0;
  unsigned int nBonds = 0;
  string word;
  string line, versionString;
  string atomNameString;
  bool success = readSdfMolHeader(is, nAtoms, nBonds, versionString);
  if (!success) {
    return success;
  }
  m = Mol(nAtoms);
  if (nBonds > 0) {
    bonds = Vec<Bond>(nBonds);
    bondOrders = Vec<unsigned int>(nBonds);
  }
  for (unsigned int i = 0; i < m.size(); ++i) {
    m[i] = readSdfAtom(is);
  }
  for (unsigned int i = 0; i < nBonds; ++i) {
    readSdfBond(is, bonds[i], bondOrders[i]);
  }
  molAnno.setMol(m);
  molAnno.setBonds(bonds);
  molAnno.setBondOrders(bondOrders);
  // read until END record found
  do {
    is >> word;
    // cout << "read word: " << word << " ";
    ERROR_IF(!is, 
	     "Error reading end of Mol file! Could not find END keyword",
	     exception);
  } while (word.compare(string("END")) != 0);
  if (m.size() < 1) {
    return false;
  }
  return true;
} 





