#include <colorFunctions.h>

// compute euclidian norm of vector
inline
double
absNorm(const Vec<double>& v) {
  double result = 0.0;
  for (unsigned int i = 0; i < v.size(); ++i) {
    result += fabs(v[i]);
  }
  return result;
}

// transform such that sum of SQUARED values is 1.0
static
void
normalizeAbs(Vec<double>& v)
{
  double norm = absNorm(v);
  if (norm <= 0.0) {
    return;
  }
  norm = 1.0 / norm;
  for (unsigned int i = 0; i < v.size(); ++i) {
    v[i] *= norm;
  }
}

/* return rainbow like mapping from value between 0.0 and 1.0 */
Vector3D 
rainbow(double xOrig, double minVal, double maxVal)
{
  PRECOND(maxVal > minVal, exception);
  Vector3D result(0.0,0.0,0.0);
  double delta = maxVal - minVal;
  double x  = (xOrig - minVal) / delta; 
  if (x <= 0.0) {
    result = Vector3D(0.0,0.0,1.0);
  }
  else if (x >= 1.0) { 
    result = Vector3D(1.0,0.0,0.0);
  }
  else if (x < 0.5) {
    result = Vector3D(0.0,2.0*x, 1.0 - (2.0 * x));
  }
  else {
    // x between 0.5 and 1:
    result = Vector3D(2.0 * (x-0.5), 2.0 * (1.0-x), 0.0);
  }
  // result.normalize();
  ASSERT(result.length() > 0.0, exception);
  result = result * (1.0 / result.length());
  POSTCOND((result.x() >= 0.0)
	   && (result.y() >= 0.0) 
	   && (result.z() >= 0.0)
	   && isSimilar(result.length(),1.0), exception);
  return result;
}

/** returns x1 for a <= 0, x2 for a >= 1,
    interpolation for values a ]0,1[
*/
double
interpolate(double x1, double x2, double a) {
  if (a < 0) {
    return x1;
  }
  else if (a > 1.0) {
    return x2;
  }
  return ((1.0 - a) * x1) + (a * x2);
}

/** returns interopolated color a = [0,1] */
Vector3D
colInterpolate(const Vector3D& c1, const Vector3D& c2, double a)
{
  return Vector3D(interpolate(c1.x(), c2.x(), a),
		  interpolate(c1.y(), c2.y(), a),
		  interpolate(c1.z(), c2.z(), a));
}

/* return rainbow like mapping from value between 0.0 and 1.0 */
Vector3D 
rainbow2(double xOrig, double minVal, double maxVal)
{
  PRECOND(maxVal > minVal, exception);
  unsigned int numCol = 10;
  double colDelta = 1.0 / static_cast<double>(numCol-1);
  Vec<Vector3D> colors(numCol);
  colors[0] = Vector3D(0.9, 0.0, 0.0); // red
  colors[1] = Vector3D(1.0, 0.65, 0.0); // orange
  colors[2] = Vector3D(1.0, 1.0, 0.0); // yellow
  colors[3] = Vector3D(0.7, 1.0, 0.0); // yellow-green
  colors[4] = Vector3D(0.0, 0.92, 0.0); // green
  colors[5] = Vector3D(0.0, 0.9, 1.0); // cyan
  colors[6] = Vector3D(0.0, 0.6, 0.8); // sky-blye
  colors[7] = Vector3D(0.0, 0.0, 1.0); // dk-blue
  colors[8] = Vector3D(0.6, 0.0, 1.0); // dk-violet
  colors[9] = Vector3D(1.0, 0.0, 1.0); // magenta
  // colors[10] = Vector3D(1.0, 0.0, 1.0); // magenta
  Vector3D result(0.0,0.0,0.0);
  double delta = maxVal - minVal;
  double x  = (xOrig - minVal) / delta; 
  if (x <= 0.0) {
    result = colors[0];
  }
  else if (x >= 1.0) { 
    result = colors[numCol];
  }
  else {
    double idxD = (numCol-1) * x;
    int idx = static_cast<int>(idxD);
    if ((idx + 1) < static_cast<int>(numCol)) {
      double y = (idxD - idx) / colDelta;
      result = colInterpolate(colors[idx], colors[idx+1], y);
    }
  }
  return result;
}

/* 
   return rainbow like mapping from value between 0.0 and 1.0
   use discretization 
*/
Vector3D 
rainbow(double xOrig, double minVal, double maxVal,
	unsigned int discrete)
{
  PRECOND((maxVal > minVal)&& (discrete > 0), exception);
  double delta = maxVal - minVal;
  double x  = (xOrig - minVal) / delta; 
  if (x < 0.0) {
    x = 0.0;
  }
  if (x > 1.0) {
    x = 1.0;
  }
  double discFrac = 1.0 / discrete;
  unsigned int mult = static_cast<unsigned int>(x / discFrac);
  x = mult * discFrac;
  ASSERT((0.0 <= x) && (x <= 1.0), exception);
  return rainbow(x, 0.0, 1.0);
}

Vec<Vector3D>
generateRainbowColors(unsigned int n)
{
  PRECOND(n > 0, exception);
  Vec<Vector3D> result(n);
  double step = 1.0 / static_cast<double>(n);
  for (unsigned int i = 0; i < n; ++i) {
    double val = i * step;
    result[i] = rainbow(val, 0.0, 1.0, n);
  }
  return result;
}

bool
isRgbColor(const Vector3D& c)
{
  return (c.x() >= 0.0) && (c.x() <= 1.0)
    && (c.y() >= 0.0) && (c.y() <= 1.0)
    && (c.z() >= 0.0) && (c.z() <= 1.0);
}

/** return center of mass. Do not take weights into account! */
Vector3D
computeCenterOfMass(const Vec<Vector3D>& v)
{
  PRECOND(v.size() > 0, exception);
  Vector3D sum(0.0,0.0,0.0);
  for (unsigned int i = 0; i < v.size(); ++i) {
    sum += v[i];
  }
  sum *= (1.0/static_cast<double>(v.size()));
  return sum;
}

Vector3D
computeColor(const Vec<double>& vOrig,
	     const Vec<Vector3D>& colors)
{
  PRECOND(colors.size() == vOrig.size(), exception);
  Vec<double> v = vOrig;
  Vector3D meanColor = computeCenterOfMass(colors);
  normalizeAbs(v);
  Vector3D result = meanColor;
  for (unsigned int i = 0; i < colors.size(); ++i) {
    result += (colors[i] - meanColor) * v[i];
  }
  POSTCOND(isRgbColor(result), exception);
  return result;
}


Vector3D
computeRainbowColor(const Vec<double>& vOrig)
{
  Vec<Vector3D> colors = generateRainbowColors(vOrig.size());
  return computeColor(vOrig, colors);
}
