// --*- C++ -*------x---------------------------------------------------------
#ifndef _HOPS_
#define _HOPS_

#include <debug.h>
#include <Vec.h>
#include <RnaSecondaryStructure.h>

class Hops {

public:

  typedef RnaSecondaryStructure::index_type index_type;

  enum { UNKNOWN_ID = -2, UNPAIRED_ID = RnaSecondaryStructure::UNPAIRED_ID, LOCKED_ID = -3 };

private:

  RnaSecondaryStructure structure;

  index_type verbose;

  mutable Vec<Vec<index_type> > matrix;

public:

  Hops(const RnaSecondaryStructure& _structure) : structure(_structure), verbose(1) {
    matrix = Vec<Vec<index_type> >(_structure.size(), Vec<index_type>(_structure.size(), UNKNOWN_ID));
  }

  virtual ~Hops() { }

  index_type hops(index_type id1, index_type id2) const {
    ASSERT(id1 >= 0);
    ASSERT(id2 >= 0);
    ASSERT(id1 < static_cast<index_type>(structure.size()));
    ASSERT(id2 < static_cast<index_type>(structure.size()));
    ASSERT(matrix[id1][id2] != LOCKED_ID);
    if (verbose > 0) {
      cout << "Starting hops " << id1 << " " << id2 << endl;
    }
    if ((matrix[id1][id2] != UNKNOWN_ID) && (matrix[id1][id2] != LOCKED_ID)) {
      return matrix[id1][id2];
    } 
    index_type result = 0;
    index_type n = static_cast<index_type>(matrix.size());
    if (id1 == id2) {
      result = 0;
    } else if (id1 > id2) {
      result = hops(id2, id1);
    } else {
      matrix[id1][id2] = LOCKED_ID;
      matrix[id2][id1] = LOCKED_ID;
      if (((id1 + 1) < n) && (matrix[id1 + 1][id2] != LOCKED_ID)) {
	result = 1 + hops(id1 + 1, id2);
      }
      index_type bp = structure.getBasePair(id1);
      if ((bp != UNPAIRED_ID) && (matrix[bp][id2] != LOCKED_ID)) {
	index_type result2 = 1 + hops(bp, id2);
	if (result2 < result) {
	  result = result2; // hopping accross base pair was shorter than walking along sequence
	}
      }
      if ((id1 > 0) && (matrix[id1-1][id2] != LOCKED_ID)) {
	index_type result3 = 1 + hops(id1 - 1, id2);
	if (result3 < result) {
	  result = result3; 
	}
      }
    }
    if (verbose > 0) {
      cout << "Finished hops " << id1 << " " << id2 << " with result " << result << endl;
    }
    matrix[id1][id2] = result;
    matrix[id2][id1] = result;
    return result;
  }

  index_type hops2(index_type id1, index_type id2) const {
    ASSERT(id1 >= 0);
    ASSERT(id2 >= 0);
    ASSERT(id1 < static_cast<index_type>(structure.size()));
    ASSERT(id2 < static_cast<index_type>(structure.size()));
    ASSERT(matrix[id1][id2] != LOCKED_ID);
    if (verbose > 0) {
      cout << "Starting hops" << id1 << id2 << endl;
    }
    if (matrix[id1][id2] != UNKNOWN_ID) {
      return matrix[id1][id2];
    } else {
      matrix[id1][id2] = LOCKED_ID;
    }
    index_type result = 0;
    if (id1 > id2) {
      result = hops(id2, id1); // reverse order
    } else if (id1 < id2) {
      result = 1 + hops(id1 + 1, id2);
      index_type bp = structure.getBasePair(id1);
      if ((bp != UNPAIRED_ID) && (bp > id1) && (abs(bp - id2) < (id2-id1))) {
	index_type result2 = 1 + hops(bp, id2);
	if (result2 < result) {
	  result = result2; // hopping accross base pair was shorter than walking along sequence
	}
      }
    } else {
      ASSERT(id1 == id2);
      result = 0;
    }
    if (verbose > 0) {
      cout << "Finished hops" << id1 << id2 << "with result" << result << endl;
    }
    matrix[id1][id2] = result;
    return result;
  }

  index_type getVerbose() const { return verbose; }

  void setVerbose(index_type _verbose) { verbose = _verbose; }
};

#endif
