// --*- C++ -*------x---------------------------------------------------------
// $Id: compass.cc,v 1.4 2011/02/19 20:55:46 eckart Exp $
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2011/02/19 20:55:46 $
//
// Description:     
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <debug.h>
#include <GetArg.h>
#include <FileName.h>
#include <SequenceAlignment.h>
#include <SimpleSequenceAlignment.h>
#include <CompensationScorer.h>
#include <vectornumerics.h>
#include <Stem.h>
#include <Random.h>
#include <Gaussian1D.h>
// #include <mynrnumerics.h>
// #include <SequenceMutator.h>
#include <sequencestat.h>
#include <stemhelp.h>
#include <generalNumerics.h>
#include <ClassifierBase.h>
#include <KnnNet2.h>
#include <KnnNetTrainer.h>
#include <compass_help.h>
// #include <Timer.h>

#define __STL_NO_DRAND48 

const string NUC_IGNORE = "XN"; // ignore X and N characters when comparing sequences

void
helpOutput(ostream& os)
{
  os << "find compensatory base changes in nucleotide alignments.\n" << endl;
  os << "usage: compass -i inputfilename" << endl
     << " other options: " << endl
     << "--algorithm n " << endl
     << "--ali-length n : specify maximum sequence length of alignment " << endl
     << "--ali-out filename : output filename for alignment " << endl
     << "--ali-ref filename : input filename for reference alignment " << endl
     << "-C filename  : output filename for alignment with constrain information" << endl
     << "--char-min n : minimum number of characters. Default: 15. Combine with --collapse-mode 3." << endl
     << "--char-min-filter n : minimum number of non-gap characters in alignment columns, set prediction to zero otherwise. Default: 0(inactive). " << endl
     << "--cleave-length n" << endl
     << "--cleave-pos n" << endl
     << "--clip-after n  : remove all sequences with index higher than n" << endl
     << "--collapse id   : collapsing with respect to this sequence number"
     << endl
     << "--collapse-mode id   : collapsing using this algorithm" 
     << endl
     << "--constraint-gap-mode 0|1 : if zero (default), no gaps in constraint output alignment (see option --C)" << endl
     << "--corr-file filename : output of correlation data between external probability and compensatory score matrix" << endl
     << "--distance-mean value   : mean contact distance for polynomial rescaling" << endl
     << "--distance-weight value : weight for polynomial rescaling of contacts" << endl
     << "--dmax value    : maximum difference of nucleotide position" << endl
     << "--dmin value    : minimum difference of nucleotide position" << endl
     << "--energy value  : weight of energy term" << endl
     << "--energy-min value  : energy cutoff" << endl
     << "--entropy-rescale value : exponential correction term to give entroyp of small number of sequences a higher weight" << endl
     << "--entropy value : weight of entropy term" << endl
     << "--entropy value" << endl
     << "--entropy-matrix-out filename  : filename for output of entropy data." << endl
     << "--entropy-read filename  : filename for input of entropy data." << endl
     << "--exit code : specify exit code for premature exiting of program."
     << endl
     << "--filter 0|1    : if set to one, filter out impossible contacts with respect to first sequence. Default: 0\n"
     << "--filter-bad n  : set all columns with more than this many positions higher than threshold to zero. n=2,3,4 are probably good choices (0=off is default)." << endl
     << endl
     << "--forced-stems filename" << endl
     << "--forced-stems-format 1|2" << endl
     << "--forced-weight value " << endl
     << "-g gausskernel" << endl
     << "--gap-frac-alpha value : defines exponent for rescaling of result matrix according to fraction of gaps. 1.0: linear" << endl
     << "--grow-limit value   : probability limit for simple growing procedure of matrix" << endl
     << "--grow-match value   : fraction matching minimum for simple growing procedure of matrix" << endl
     << "--knn-cutoff2 value  : used for speed optimized data structure of knn nodes." << endl
     << "--knn-cutoff22 value  : used for speed optimized data structure of knn nodes." << endl
     << "--knn-data filename  : filename of training data for k-nearest neighbor algorithm" << endl
     << "--knn-data2 filenames  : filenames of training data for k-nearest neighbor algorithm used as postprocessing" << endl
     << "--knn-data-border filenames : filenames of training data foor k-nearest neigbor algorithm used for first, second ... n'th column/row near border" << endl
     << "--knn-dir filename  : directory of training data for k-nearest neighbor algorithm" << endl
     << "--knn-dir2 filename  : directory of training data for k-nearest neighbor algorithm used for postprocessing" << endl
     << "--knn-gauss value   : if greater zero, use as deviation of gaussian weighting in knn method" << endl
     << "--knn-optimize id   : id of knn node to optimize scaling of" << endl
     << "--knn-optimize-steps steps  : number of optimization steps" << endl
     << "--knn-rounds2 n     : number of postprocessing rounds" << endl
     << "--knn-save-file filename    : filename for saving knn definition. Use with --knn-optimize 0" << endl
     << "--knn-train-file filename    : filename for knn training data. Use with --knn-write-file" << endl
     << "--knn-write-level n : write to training file after this many knn levels. Use with --knn-write-file" << endl
     << "--knn-write-file filename     : write training data to this file" << endl
     << "--knn-write-max n   : maximum number of lines in file defined by --knn-write-file" << endl
     << "--length-mean value   : mean length for polynomial rescaling" << endl
     << "--length-weight value : weight for polynomial rescaling" << endl
     << "--list filename  : output filename for list of high scoring points" << endl
     << "--log filename" << endl
     << "--loop filename   : name of loop file." << endl
     << "--if id   : input format id. 1: FASTA, 4: Danforth; 6: Blast (mode 6); 7: CE" << endl
     << "--match-thresh fraction : if greater zero, set all pairing probabilities to zero that have a fraction of complementary base pairs smaller than thresh"
     << endl
     << "--matrix filename : read matrix externally instead of computing it oneself" << endl
     << "--matrix-format 1|2  : format of matrix file (see option --matrix) 1: standard, 2: list" << endl
     << "--matrix-normal 0|1 : if non zero, write row 1 of matrix first, otherwise write row n first." << endl
     << "--matrix-ids ids    : define which feature matrices to write to file (combine with option -o)" << endl
     << "--neural filename  : filename for neural network output" << endl
     << "--neural-all 0|1   : if one (default), write feature vector even for borders." << endl
     << "--neural-both 0|1  : if one (not default), write feature vector for both sides of diagonal." << endl
     << "--neural-conserve 0|1  : if one, compute single column entropies"
     << endl
     << "--neural-dist dist : distance bias of neural network output" << endl
     << "--neural-energy-mode 0|1 : output of matching energy in feature vector output file" << endl
     << "--neural-head filename  : filename for neural header file" << endl
     << "--neural-mode 1..10     : format of feature vectors" << endl
     << "--neural-mode2 1..10     : format of second level feature vectors" << endl
     << "--neural-mode3 0|1|2     : 1: regular fingerprint, 2: use pseudocounts, 0: no fingerprints"
     << endl
     << "--neural-patch 1...n  : patch size for neural network will be 2n+1" << endl
     << "--neural-prob-mode 0|1 : output of folding probability in feature vector output file" << endl
     << "--neural-rename-mode 0|1 : switching alphabet renaming score on or off" << endl
     << "--neural-self-mode 0|1 : if true, ignore closest hit of knn search and in this way avoid problems with overlap of training and testing set." << endl
     << "-o outputfile" << endl
     << "--optimize mode : 0: no optimization 1: complete search optimization" << endl
     << "--optimize-parameters filename : filename with parameters for optimization mode 2"
     << endl
     << "--optimize-rounds number : number of rounds of optimization" << endl
     << "--pfile filename : filename with mean and stddev of scores" << endl
     << "--prediction-confidence value" << endl
     << "--prob-consensus mode  : 1: average; 2: average but zero if one is zero, 3: non-linear (standard)" << endl
     << "--prob-matrix filename   : read complete alignment probability matrix from file" << endl
     << "--prob-matrix-format 1|2|3  : read probability matrix in native/list/R format" << endl
     << "--prob-matrix-out filename   : write complete alignment probability matrix to file" << endl
     << "--prob-dir filename   : directory with probability files" << endl
     << "--prob-format id      : format of probability file 1: RNAfold postscript | 2: own format" << endl
     << "--prob-file filename  : list with probability files" << endl
     << "--prune-length n" << endl
     << "--prune-pos n" << endl
     << "--randomize n" << endl
     << "--rescale filename  : filename with rescale info" << endl
     << "--secondary-alignment" << endl
     << "--seed number : random number generator seed" << endl
     << "--sequence-weight-file filename : filename with weighting tree" << endl
     << "--sequence-weight-mode n        : 0: uniform weighting, 1: simple weighting, 2: read file" << endl
     << "--stem-file filename" << endl
     << "--stem-format format" << endl
     << "--stem-min n    : define minimum stem length" << endl
     << "--stem-min-anyways n    : include contacts even if smaller minimum stem length" << endl
     << "--stem-outfile filename : output file for reference stems" << endl
     << "--stem-outfile2 filename : output file for output of all possible predicted stems." << endl
     << "--stem-outfile2-limit filename : stem score limit for output file for output of all possible predicted stems." << endl
     << "--stem-outfile2-weight filename : stem score weight for output file for output of all possible predicted stems." << endl
     << "--stem-outfile-format 1|2|3|4 : format of output file for reference stems" << endl
     << "--stem-weight value" << endl
     << "--subset-optimize positions" << endl
     << "--subset-start parsestring    : define subset as in unix program cut :  -3,4-8,12 -> 1,2,3,4,5,6,7,8,12"
     << endl
     << "--subset-stop parsestring    : define subset as in unix program cut :  -3,4-8,12 -> 1,2,3,4,5,6,7,8,12"
     << endl
     << "--sum-max value  : maximum sum of indices" << endl
     << "--sum-min value  : minimum sum of indices" << endl
     << "--thresh value" << endl
     << "--thresh-high value" << endl
     << "--thresh-mediumhigh value" << endl
     << "--thresh-individual value" << endl
     << "--tree-stat filename : output of mutation statistic" << endl
     << "--of id   : output format id. 1: FASTA, 2: ClustalW; 3: Mannheim; 4: Danforth" << endl
     << "--verbose n" << endl
     << "--winner 0|1  : if one, apply winner-takes-all filter to entropy matrix (default: zero)" << endl
     << endl;
}

/** returns false if an unknown option was found, true otherwise. */
bool
checkArg(int argc, char ** argv)
{
  return testArg(argc, argv, 127,
		 "-algorithm",
		 "-ali-length",
		 "-ali-out",
		 "-ali-ref",
		 "-add-left",
		 "-add-right",
		 "C",
		 "-char-min",
		 "-char-min-filter",
		 "-collapse",
		 "-collapse-mode",
		 "-constraint-gap-mode",
		 "-corr-file",
		 "-distance-mean",
		 "-distance-weight",
		 "-dmax",
		 "-dmin",
		 "-energy",
		 "-energy-min",
		 "-entropy",
		 "-entropy-matrix-out",
		 "-entropy-read",
		 "-entropy-rescale",
		 "-exit",
		 "-filter",
		 "-filter-bad",
		 "-forced-stems",
		 "-forced-stems-format",
		 "-forced-weight",
		 "g",
		 "-gap-frac-alpha",
		 "-grow-limit",
		 "-grow-match",
		 "i",
		 "-if",
		 "-knn-cutoff2",
		 "-knn-cutoff22",
		 "-knn-data",
		 "-knn-data2",
		 "-knn-data-border",
		 "-knn-dir",
		 "-knn-dir2",
		 "-knn-gauss",
		 "-knn-optimize",
		 "-knn-optimize-steps",
		 "-knn-rounds2",
		 "-knn-save-file",
		 "-knn-train-file",
		 "-knn-write-max",
		 "-knn-write-level",
		 "-knn-write-file",
		 "-length-mean",
		 "-length-weight",
		 "-list",
		 "-log",
		 "-loop",
		 "-match-thresh",
		 "-matrix",
		 "-matrix-format",
		 "-matrix-ids",
		 "-matrix-normal",
		 "-md-limit",
		 "-neural",
		 "-neural-all",
		 "-neural-both",
		 "-neural-conserve",
		 "-neural-dist",
		 "-neural-energy-mode",
		 "-neural-head",
		 "-neural-max",
		 "-neural-mode",
		 "-neural-mode2",
		 "-neural-mode3",
		 "-neural-patch",
		 "-neural-prob-mode",
		 "-neural-rename-mode",
		 "-neural-self-mode",
		 "-norm",
		 "o",
		 "-of",
		 "-optimize",
		 "-optimize-entropy",
		 "-optimize-match",
		 "-optimize-parameters",
		 "-optimize-ratio",
		 "-optimize-rounds",
		 "-overwrite",
		 "p",
		 "-pfile",
		 "-prediction-confidence",
		 "-prob-consensus",
		 "-prob-dir",
		 "-prob-file",
		 "-prob-format",
		 "-prob-matrix",
		 "-prob-matrix-format",
		 "-prob-matrix-out",
		 "-secondary-alignment",
		 "-seed",
		 "-sequence-weight-file",
		 "-sequence-weight-mode",
		 "-stem-min",
		 "-stem-min-anyways",
		 "-stem-file",
		 "-stem-format",
		 "-stem-outfile",
		 "-stem-outfile2",
		 "-stem-outfile2-limit",
		 "-stem-outfile2-weight",
		 "-stem-outfile-format",
		 "-stem-weight",
		 "-subset-optimize",
		 "-subset-start",
		 "-subset-stop",
		 "-randomize",
		 "-reference",
		 "-rescale",
		 "-root",
		 "-sum-max",
		 "-sum-min",
		 "-thresh",
		 "-thresh-high",
		 "-thresh-mediumhigh",
		 "-thresh-individual",
		 "-tree-stat",
		 "-verbose",
		 "-winner");
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

string
charvec2string(const vector<char>& v)
{
  string result;
  for (unsigned int i = 0; i < v.size(); ++i) {
    result = result + v[i];
  }
  return result;
}

/** returns id of i'th and j'th character pair */
unsigned int
getPairId(unsigned int i, unsigned int j, unsigned int size)
{
  return (i * size) + j;
}

unsigned int
countPairsLeaveOneOut(const Vec<string>& sequences,
		      unsigned int leaveOut,
		      char c1, char c2,
		      unsigned int p1, unsigned int p2)
{
  unsigned int counter = 0;
  for (unsigned int i = 0; i < sequences.size(); ++i) {
    if (i == leaveOut) {
      continue;
    }
    if ((sequences[i][p1] == c1)
	&& (sequences[i][p2] == c2)) {
      ++counter;
    }
  }
  return counter;
}

unsigned int
countLeaveOneOut(const Vec<string>& sequences,
		 unsigned int leaveOut,
		 char c,
		 unsigned int p)
{
  unsigned int counter = 0;
  for (unsigned int i = 0; i < sequences.size(); ++i) {
    if (i == leaveOut) {
      continue;
    }
    if (sequences[i][p] == c) {
      ++counter;
    }
  }
  return counter;
}

Vec<Vec<double> >
generateLeaveOneOutCountMatrix(const Vec<string>& sequences,
			       unsigned int leaveOut,
			       char c1, char c2)
{
  unsigned int sLen = sequences[0].size();
  Vec<Vec<double> > result(sLen,Vec<double>(sLen, 0.0));
  for (unsigned int i = 0; i < sLen; ++i) {
    for (unsigned int j = 0; j < sLen; ++j) {
      result[i][j] = countPairsLeaveOneOut(sequences, leaveOut, c1, c2, i, j);
      result[j][i] = result[i][j];
    }
  }
  return result;
}

Vec<Vec<Vec<double> > >
generateLeaveOneOutCountMatrix(const Vec<string>& sequences,
			       unsigned int leaveOut,
			       const string& alphabet)
{
  unsigned int numPairs = alphabet.size() * alphabet.size();
  Vec<Vec<Vec<double> > > result(numPairs);
  for (unsigned int i = 0; i < alphabet.size(); ++i) {
    for (unsigned int j = 0; j < alphabet.size(); ++j) {
      unsigned int pairId = getPairId(i, j, alphabet.size());
      result[pairId] = generateLeaveOneOutCountMatrix(sequences, leaveOut, 
						 alphabet[i], alphabet[j]);      
    }
  }
  return result;
}

Vec<double >
generateLeaveOneOutCountVector(const Vec<string>& sequences,
			       unsigned int leaveOut,
			       char c1)
{
  unsigned int sLen = sequences[0].size();
  Vec<double> result(sLen, 0.0);
  for (unsigned int i = 0; i < sLen; ++i) {
    result[i] = countLeaveOneOut(sequences, leaveOut, c1, i);
  }
  POSTCOND(result.size() == sLen);
  return result;
}

Vec<Vec<double> >
generateLeaveOneOutCountVector(const Vec<string>& sequences,
			       unsigned int leaveOut,
			       const string& alphabet)
{
  Vec<Vec<double> > result(alphabet.size());
  for (unsigned int i = 0; i < alphabet.size(); ++i) {
    result[i] = generateLeaveOneOutCountVector(sequences, leaveOut, 
						 alphabet[i]);      
  }
  return result;
}

/** returns compensatory score between positions p1 and p2 of sequence alignment
    NEGATIVE entropy!
    mi: leaveOneOutCount,
    mij: leaveOneOutcountMatrix */
double
computeCompensatoryScore(unsigned int p1, unsigned int p2,
			 const string& leftOutSequence,
			 const string& alphabet,
			 unsigned int numSequences,
			 const Vec<Vec<double> >& mi,
			 const Vec<Vec<Vec<double> > >& mij)
{
  // cout << "Starting computeCompensatoryScore(p1, p2...) " << endl;
  if (numSequences == 0U) {
    return 0.0;
  }
  unsigned int aSize = alphabet.size();
  unsigned int nxi = 0;
  unsigned int nxj = 0;
  unsigned int nxij = 0;
  unsigned int pairId = 0;
  double sum = 0.0;
  double snum = static_cast<double>(numSequences);
  for (unsigned int k1 = 0; k1 < aSize; ++k1) {
    ASSERT(p1 < leftOutSequence.size());
    ASSERT(k1 < alphabet.size());
    if (leftOutSequence[p1] == alphabet[k1]) {
      nxi = 1;
    }
    else {
      nxi = 0;
    }
    for (unsigned int k2 = 0; k2 < aSize; ++k2) {
      ASSERT(p2 < leftOutSequence.size());
      if (leftOutSequence[p2] == alphabet[k2]) {
	nxj = 1;
      }
      else {
	nxj = 0;
      }
      pairId = getPairId(k1, k2, aSize);
      nxij = nxi * nxj; // joint count
      ASSERT(pairId < mij.size());
      ASSERT(p1 < mij[pairId].size());
      ASSERT(p2 < mij[pairId][p1].size());
      ASSERT(k1 < mi.size());
      ASSERT(p1 < mi[k1].size());
      ASSERT(k2 < mi.size());
      ASSERT(p2 < mi[k2].size());
      sum += (mij[pairId][p1][p2] + nxij)
	* (lg2(snum) + lg2(mij[pairId][p1][p2]+nxij)
	   - lg2(mi[k1][p1] + nxi) - lg2(mi[k2][p2] + nxj) );
    }
  }  
  return sum / snum;
}


double
computeCompensatoryScore(const SequenceAlignment& ali, 
			 unsigned int m , unsigned int n,
			 const CompensationScorer& scorer)
{
  string slice1 = ali.getColumn(m);
  string slice2 = ali.getColumn(n);
//   upperCase(slice1);
//   upperCase(slice2);
  double result = scorer.compensationScore(slice1, slice2, 
	   Vec<double>(ali.size(), 1.0)); // TODO : unnecessary generation of weights !
//   if (result >= scoreLimit) {
//     cout << "result of " << m + 1 << " " << n + 1 << " " 
// 	 << result << " " << slice1 << " " << slice2 << endl;
//   }
  return result;
}

Vec<Vec<double> > 
computeCompensatoryScores(const SequenceAlignment& ali, 
			  const CompensationScorer& scorer)
{
  unsigned int nn = ali.getLength();
  Vec<Vec<double> > result(nn, Vec<double>(nn, 0.0));
  cout << "Computing compensatory scores with scorer algorithm "
       << scorer.algorithm << endl;
  for (unsigned int i = 0; i < result.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      result[i][j] = computeCompensatoryScore(ali, i, j, scorer);
      result[j][i] = result[i][j];
      if (!isReasonable(result[i][j])) {
	cout << i + 1 << " " << j + 1 << " "
	     << result[i][j] << " "
	     << ali.getColumn(i) << " " << ali.getColumn(j) << endl;
	ERROR("Unreasonable matrix element found!");
      }
    }
  }
  return result;
}


/** returns compensatory scores with precomputed leave-one out counting tables */
Vec<Vec<double> > 
computeCompensatoryScores(const string& leftOutSequence,
			  const string& alphabet,
			  unsigned int numSequences,
			  const Vec<Vec<double> >& leaveOneOutCount,
			  const Vec<Vec<Vec<double> > >& leaveOneOutCountMatrix,
			  double& minVal,
			  double& maxVal)
{
  // cout << "starting compute compensation scores..." << endl;
  unsigned int nn = leftOutSequence.size();
  Vec<Vec<double> > result(nn, Vec<double>(nn, 0.0));
  minVal = 1e30;
  maxVal = -1e30;
  for (unsigned int i = 0; i < result.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      result[i][j] = computeCompensatoryScore(i, j, leftOutSequence, alphabet,
		  numSequences, leaveOneOutCount, leaveOneOutCountMatrix);
      result[j][i] = result[i][j];
      if (result[i][j] > maxVal) {
	maxVal = result[i][j];
      }
      if (result[i][j] < minVal) {
	minVal = result[i][j];
      }
    }
  }
  return result;
}

/** returns compensatory scores with precomputed leave-one out counting tables.
 This version uses only windows with length winLen, starting at startA and startB*/
Vec<Vec<double> > 
computeCompensatoryScores(const string& leftOutSequence,
			  const string& alphabet,
			  unsigned int numSequences,
			  const Vec<Vec<double> >& leaveOneOutCount,
			  const Vec<Vec<Vec<double> > >& leaveOneOutCountMatrix,
			  double& minVal,
			  double& maxVal,
			  int startA,
			  int startB,
			  int winLen)
{
  // cout << "starting compute compensation scores..." << endl;
  unsigned int nn = winLen; // leftOutSequence.size();
  Vec<Vec<double> > result(nn, Vec<double>(nn, 0.0));
  minVal = 1e30;
  maxVal = -1e30;
  for (unsigned int i = 0; i < result.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
//       ASSERT((i+startA) < leaveOneOutCount.size());
//       ASSERT((j+startB) < leaveOneOutCount.size());
      result[i][j] = computeCompensatoryScore(i+startA, j+startB, leftOutSequence, alphabet,
		  numSequences, leaveOneOutCount, leaveOneOutCountMatrix);
      result[j][i] = result[i][j];
      if (result[i][j] > maxVal) {
	maxVal = result[i][j];
      }
      if (result[i][j] < minVal) {
	minVal = result[i][j];
      }
    }
  }
  return result;
}


/** returns compensatory scores with precomputed leave-one out counting tables */
Vec<Vec<double> > 
computeCompensatoryScores(const Vec<string>& sequences,
			  const string& alphabet)
{
  unsigned int leaveOut = 0; // leave out this sequence in counting
  // unsigned int numSequences = sequences.size();
  double minVal = 0.0;
  double maxVal = 0.0;
  Vec<Vec<double> > leaveOneOutCountVec = generateLeaveOneOutCountVector(sequences, leaveOut, alphabet);
  Vec<Vec<Vec<double> > > leaveOneOutCountMatrix
    = generateLeaveOneOutCountMatrix(sequences, leaveOut, alphabet);
  return computeCompensatoryScores(sequences[leaveOut], alphabet, sequences.size(),
				   leaveOneOutCountVec, leaveOneOutCountMatrix, minVal, maxVal);				   
}



void
reportCompensatoryScores(ostream& os, 
			 const Vec<Vec<double> >& compMatrix, 
			 const SequenceAlignment& ali, 
			 double scoreLimit,
			 int verboseLevel)
{
  unsigned int counter = 0;
  for (unsigned int i = 0; i < compMatrix.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      if (compMatrix[i][j] >= scoreLimit) {
	cout << i + 1 << " " << j + 1 << " " << compMatrix[i][j] << endl;
	++counter;
      }
    }
  }
  cout << "Number of base pairs above threshold: " << counter << endl;
}

Vec<double>
getCrossVector(const Vec<Vec<double> >& mat, 
	       int m, int n)
{
  if (mat.size() < 1) {
    return Vec<double>();
  }
  Vec<double> cross(2*mat.size() - 1);
  unsigned int pc = 0;
  for (unsigned int i = 0; i < mat.size(); ++i) {
    cross[pc++] = mat[i][n];
  }
  for (unsigned int i = 0; i < mat[m].size(); ++i) {
    if (static_cast<int>(i) == n) {
      continue;
    }
    ASSERT(pc < cross.size());
    cross[pc++] = mat[m][i];
  }
  return cross;
}

// ignore elements too close to diagonal and border 
Vec<double>
getCrossVector(const Vec<Vec<double> >& mat, 
	       int m, int n,
	       int borderLim, int diagLim)
{
  if (mat.size() < 1) {
    return Vec<double>();
  }
  //   int size1 = 1 * 
  //     (static_cast<int>(mat.size()) - 1 - 2 * diagLim - 2 * borderLim);
  //   int size1 = static_cast<int>(mat.size()) - (2 * diagLim) - (2 * borderLim) + 2;
  //   ERROR_IF(size1 <= 1, "Cannot compute z-score, border values are too large.",
  // 	   exception);
  Vec<double> cross; // (size1, 0.0);
  // unsigned int pc = 0;
  // n - diagLim - borderLim elemenets
  for (int i = borderLim; i < n - diagLim; ++i) {
    //     ASSERT(pc < cross.size());
    //     cross[pc++] = mat[i][n];
    cross.push_back(mat[i][n]);
  }
  // mat.size() - borderLim - n - diagLim - 1 
  for (int i = n + diagLim + 1; i < static_cast<int>(mat.size()) - borderLim; 
       ++i) {
    // ASSERT(pc < cross.size());
    // cross[pc++] = mat[i][n];
    cross.push_back(mat[i][n]);
  }
  //   for (int i = borderLim; (i + borderLim) < static_cast<int>(mat[m].size()); 
  //        ++i) {
  //     if ( abs(i - m) <= diagLim) {
  //       continue;
  //     }
  //     ASSERT(pc < cross.size());
  //     cross[pc++] = mat[m][i];
  //   }
  //   for (unsigned int i = 0; i < mat[m].size(); ++i) {
  //     if (static_cast<int>(i) == n) {
  //       continue;
  //     }
  //     ASSERT(pc < cross.size());
  //     cross[pc++] = mat[m][i];
  //   }
  return cross;
}

/** substracts row and column mean for each element */
Vec<Vec<double> >
renormalizeMatrix(const Vec<Vec<double> >& m)
{
  unsigned int nn = m.size();
  Vec<Vec<double> > result = m;
  if (nn == 0) {
    return result;
  }
  Vec<double> cross(2*nn - 1);
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      cross = getCrossVector(m, i, j);
      double divi = sqrt(vecVariance(cross));
      result[i][j] = (m[i][j] - (vecMean(cross)));
      if (divi > 0.0) {
	result[i][j] /= divi;
      }
    }
  }
  return result;
}

/** substracts row and column mean for each element */
Vec<Vec<double> >
renormalizeMatrix(const Vec<Vec<double> >& m,
		  int borderLim,
		  int diagLim)
{
  int nn = m.size();
  Vec<Vec<double> > result = m;
  if (nn == 0) {
    return result;
  }
  Vec<double> cross(2*nn - 1);
  for (int i = 0; i < nn; ++i) {
    for (int j = 0; j < static_cast<int>(m[i].size()); ++j) {
      cross = getCrossVector(m, i, j, borderLim, diagLim);
      // ignore border and diagonal
      double divi = sqrt(vecVariance(cross));
      result[i][j] = (m[i][j] - (vecMean(cross)));
      if (divi > 0.0) {
	result[i][j] /= divi;
      }
    }
  }
  return result;
}

/** substracts row and column mean for each element */
void
applyLogisticFunction(Vec<Vec<double> >& m, double scale)
{
  PRECOND(scale != 0.0);
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      m[i][j] = logistic(m[i][j] / scale);
    }
  }
}

double
convolution(const Vec<Vec<double> >& mat,
	    const Vec<Vec<double> >& mask,
	    unsigned int m, // row
	    unsigned int n // column
	    )
{
  int nn = static_cast<int>(mask.size());
  int nnh = (static_cast<int>(nn) - 1) / 2;
  ASSERT((nnh + 1) * 2 == nn);
  double sum = 0.0;
  int irow, icol;
  for (unsigned int i = 0; i < mask.size(); ++i) {
    irow = m + i - nnh;
    for (unsigned int j = 0; j < mask[i].size(); ++j) {
      icol = n + j - nnh;
      if ((irow >= 0) && (irow < static_cast<int>(mat.size())) 
	  && (icol >= 0) && (icol < static_cast<int>(mat[i].size()))) {
	sum += mat[irow][icol];
      }
    }
  }
  return sum;
}

Vec<Vec<double> >
filterMatrix(const Vec<Vec<double> >& m,
	     const Vec<Vec<double> >& mask)
{
  Vec<Vec<double> > result = m;
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      result[i][j] = convolution(m, mask, i, j);
    }
  } 
  return result;
}

Vec<Vec<double> >
filterMatrix(const Vec<Vec<double> >& m, 
	     double fm0,
	     double fmv,
	     double fmh,
	     double fmd,
	     double fma)
{
  Vec<Vec<double> > mask(3, Vec<double>(3));
  mask[0][0] = fmd;
  mask[0][1] = fmv;
  mask[0][2] = fma;
  mask[1][0] = fmh;
  mask[1][1] = fm0;
  mask[1][2] = fmh;
  mask[2][0] = fma;
  mask[2][1] = fmv;
  mask[2][2] = fmd;
  return filterMatrix(m, mask);
}

Vec<Stem>
matrix2Stems(const Vec<Vec<double> >& m, 
	     unsigned int minStemLength,
	     double thresh)
{
  bool stemOpen = false;

  unsigned int stemStart = 0;
  unsigned int stemStop = 0;
  unsigned int stemLength = 0;
  double stemEnergy = 0.0;
  Vec<Stem> stems;
  string seq1, seq2;
  for (unsigned int i = 1; i < m.size(); ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      if (m[i][j] >= thresh) {
	if ((i > 0) && (m[i-1][j+1] >= thresh)) {
	  continue; // this is not the start of the stem, so skip
	}
	stemOpen = true;
	stemLength = 1;
	stemStart = i;
	stemStop = j;
	stemEnergy = m[i][j]; // dummy energy
	++stemLength;
	unsigned int k = 1;
	for (k = 1; (i+k < m.size()) && (k <= j); ++k) {
	  if (m[i+k][j-k] < thresh) {
	    break;
	  }
	  stemEnergy += m[i+k][j-k];
	} 
	stemLength = k;
// 	if (stemLength >= histogram.size()) {
// 	  stemLength = histogram.size() - 1;
// 	}
	// 	string seq1= sequence.substr(stemStart, stemLength);
	// 	string seq2 = sequence.substr(stemStop - stemLength + 1, stemLength);
	if (stemLength >= minStemLength) {     
	  stems.push_back(Stem(stemStart, stemStop, stemLength, stemEnergy, seq1, seq2));
	}
      }
    }
  }
  return stems;
}

Vec<Vec<double> >
stems2Matrix(const Vec<Stem>& stems, 
	     unsigned int totalLength,
	     int minStemLength)
{
  Vec<Vec<double> > result(totalLength, Vec<double>(totalLength, 0.0));
  for (unsigned int i = 0; i < stems.size(); ++i) {
    if (stems[i].getLength() < minStemLength) {
      continue;
    }
    double avgEnergy = stems[i].getEnergy() / stems[i].getLength();
    for (int j = 0; j < stems[i].getLength(); ++j) {
      int row = stems[i].getStart() + j;
      int col = stems[i].getStop() - j;
      ASSERT((row >= 0) && (row < static_cast<int>(result.size())));
      ASSERT((col >= 0) && (col < static_cast<int>(result[row].size())));
      result[row][col] = avgEnergy;
      result[col][row] = result[row][col];
    }
  }
  return result;
}

Vec<Vec<double> >
filterMatrixWithStems(const Vec<Vec<double> >& m, 
		      unsigned int minStemLength,
		      double thresh)
{
  unsigned int nn = m.size();
  Vec<Stem> stems = matrix2Stems(m, minStemLength, thresh);
  cout << "generated stems: " << stems << endl;
  return stems2Matrix(stems, nn, minStemLength);
}

string
randomizeString(const string & s, int maxShift)
{
  string result;
  Random& rnd = Random::getInstance();
  int num = rnd.getRand(maxShift);
  int leftFlag = rnd.getRand(2);
  if (leftFlag) {
    result = s.substr(num) + string(num, '-');
  }
  else {
    result = string(num, '-') + s.substr(0, s.size() - num);
  }
  POSTCOND(result.size() == s.size());
  return result;
}

void
randomizeAlignment(SequenceAlignment& ali, int maxShift)
{
  for (unsigned int i = 0; i < ali.size(); ++i) {
    ali.setSequence(randomizeString(ali.getSequence(i), maxShift),
		    ali.getName(i), i);
  }
}

/** returns Shannon information entropy of vector */
double
vectorEntropy(const Vec<double>& data,
	      double min, double delta, unsigned int numBins)
{
  if (data.size() == 0) {
    return 0.0;
  }
  const double LOG2 = log(2.0);
  Vec<unsigned int> hist = computeHistogram(data, min, delta, numBins);
  cout << "Histogram related to entropy: " << hist << endl;
  double result = 0.0;
  for (unsigned int i = 0; i < hist.size(); ++i) {
    if (hist[i] > 0) {
      double f = static_cast<double>(hist[i]) / data.size();
      ASSERT(f > 0.0);
      result -= f * log(f) / LOG2;
    }
  }
  return result;
}

/** returns Shannon information entropy of matrix */
double
matrixEntropy(const Vec<Vec<double> >& data,
	      double min, double delta, unsigned int numBins)
{
  Vec<double> v = matrix2Vec(data);
  return vectorEntropy(v, min, delta, numBins);
}

/** returns Shannon entropy of alignment. 0: perfect matching alignment, >0 otherwise
    leaveoneOutCount: for given letter and position, how often is that letter found at a certain position in 
    the alignment
    letterId: given letter index in alphabet, seqPos: position in sequence (which column)
 */
double
sequenceAlignmentEntropy(const string& leftOutSequence,
			 const string& alphabet,
			 unsigned int numSequences,
			 double leaveOneOutCount,
			 unsigned int letterId,
			 unsigned int seqPos)
{
  PRECOND(letterId < alphabet.size());
  PRECOND(numSequences > 0);
  PRECOND(seqPos < leftOutSequence.size());
  double nia0 = 0.0;  
  if (alphabet[letterId] == leftOutSequence[seqPos]) {
    nia0 = 1.0;
  }
  double nc = leaveOneOutCount + nia0; // total number of that character at that column position
  double frac = nc / numSequences;
  double result = 0.0;
  ASSERT(frac <= 1.0);
  if (frac > 0.0) {
    result = - (nc/numSequences) * lg2(frac);
  }
  return result;
}

/** returns Shannon entropy of alignment. 0: perfect matching alignment, >0 otherwise
    leaveoneOutCount: for each letter, how often is that letter found at a certain position in 
    the alignment */
double
sequenceAlignmentEntropy(const string& leftOutSequence,
			 const string& alphabet,
			 unsigned int numSequences,
			 const Vec<Vec<double> >& leaveOneOutCount)
{
  double entropy = 0.0;
  for (unsigned int i = 0; i < alphabet.size(); ++i) {
    for (unsigned int j = 0; j < leftOutSequence.size(); ++j) {
      entropy += sequenceAlignmentEntropy(leftOutSequence, 
					  alphabet, numSequences,
					  leaveOneOutCount[i][j], i, j);
    }
  }
  return entropy;
}

/** returns Shannon entropy of alignment. 0: perfect matching alignment, >0 otherwise
    leaveoneOutCount: for each letter, how often is that letter found at a certain position in 
    the alignment */
double
sequenceAlignmentEntropy(const string& leftOutSequence,
			 const string& alphabet,
			 unsigned int numSequences,
			 const Vec<Vec<double> >& leaveOneOutCount,
			 int startA, int startB, int winLen)
{
  double entropy = 0.0;
  for (string::size_type i = 0; i < alphabet.size(); ++i) {
    for (int j = 0; j < winLen; ++j) {
      entropy += sequenceAlignmentEntropy(leftOutSequence, 
					  alphabet, numSequences,
					  leaveOneOutCount[i][j], i, j+startA);
      entropy += sequenceAlignmentEntropy(leftOutSequence, 
					  alphabet, numSequences,
					  leaveOneOutCount[i][j], i, j+startB);
    }
  }
  return entropy;
}


// double
// bimodalSeparation(const Vec<double>& v, double ratioWeight)
// {
//   // determine min and max values:
//   double mini = v[minElement(v)];
//   double maxi = v[maxElement(v)];
//   double sepa = 0.5 * (maxi + mini);
//   Gaussian1D lowerG, higherG;
//   for (unsigned int i = 0; i < v.size(); ++i) {
//     if (v[i] < sepa) {
//       lowerG.add(v[i]);
//     }
//     else {
//       higherG.add(v[i]);
//     }
//   }
//   cout << "two defined gaussians: " << lowerG << "  " << higherG << endl;
//   double t, p;
//   t_test(lowerG.getAverage(), lowerG.getVariance(), lowerG.getNumElements(),
// 	 higherG.getAverage(), higherG.getVariance(), higherG.getNumElements(),
// 	 t, p);
//   cout << "t and p-value for t-test: " << t << " " << p << endl;
//   double ratio = lowerG.getNumElements() 
//     / static_cast<double>(higherG.getNumElements());
//   cout << "ratio of population sizes: " 
//        << ratio << " " << 1.0/ratio << endl;
//   cout << "min max lowerG: " << lowerG.getLowest() << " " 
//        << lowerG.getHighest()
//        << endl;
//   cout << "min max higherG: " << higherG.getLowest() << " " 
//        << higherG.getHighest()
//        << endl;
//   double avgDiff = higherG.getAverage() - lowerG.getAverage();
//   double score = avgDiff - ratioWeight * (1.0/ratio);
//   cout << "difference of means minus population ratio: "
//        <<  score << endl;
//   return score;
// }


/** how much more likely it is to find "stems" in anti-diagonal direction vs diagonal */
double
localSkewness(const Vec<Vec<double> >& m, int i, int j)
{
  double diagSum = 0.0;
  double antiSum = 0.0;
  unsigned int diagCount = 0;
  unsigned int antiCount = 0;
  int sz = m.size(); // m must be quadratic!
//   if (abs(i-j) < 1) {
//     return 0.0; // around diagonal and left and reight neighbor of diagonal
//   }
  // currently borders are ignored!
  if (((j-1) >= 0)  && ((i+1) < sz)) {
    antiSum += m[i+1][j-1];
    ++antiCount;
  }
  if (((i-1) >= 0)  && ((j+1) < sz)) {
    antiSum += m[i-1][j+1];
    ++antiCount;
  }
  if (((i-1) >= 0)  && ((j-1) > 0)) {
    diagSum += m[i-1][j-1];
    ++diagCount;
  }
  if (((i+1) < sz)  && ((j+1) < sz)) {
    diagSum += m[i+1][j+1];
    ++diagCount;
  }
  if (antiCount > 0) {
    antiSum /= antiCount;
  }
  if (diagCount > 0) {
    diagSum /= diagCount;
  }

  double result = (antiSum - diagSum);
  return result;
}

/** how much more likely it is to find "stems" in anti-diagonal direction vs diagonal */
Vec<Vec<double> >
localSkewness(const Vec<Vec<double> >& m)
{
  Vec<Vec<double> > result = m;
  for (int i = 0; i < static_cast<int>(m.size()); ++i) {
    for (int j = 0; j < static_cast<int>(m.size()); ++j) {
      result[i][j] = localSkewness(m, i, j);
    }
  }
  return result;
}

/** add local skewness of positions that are highest in row */
double
bestSkewness(const Vec<Vec<double> >& m)
{
  double result = 0.0;
  for (int i = 0; i < static_cast<int>(m.size()); ++i) {
    int bestJ  = 0;
    double bestJVal = m[i][bestJ];
    for (int j = 0; j < static_cast<int>(m.size()); ++j) {
      if (abs(i-j) < 3) {
	continue; // skip if too near diagonal
      }
      if (m[i][j] > bestJVal) {
	bestJ = j;
	bestJVal = m[i][j];
      }
    }
    if (abs(i-bestJ) < 3) {
      continue; // skip if too near diagonal
    }
    result+= localSkewness(m, i, bestJ);
  }
  result /= m.size(); // divide by number of rows
  return result;
}

/** how much more likely it is to find "stems" in anti-diagonal direction vs diagonal */
double
skewness(const Vec<Vec<double> >& m, double thresh)
{
  unsigned int countThresh = 0;
  // unsigned int countDiag = 0;
  // unsigned int countAntiDiag = 0;
  unsigned int countBelowThresh = 0;
  double diagSum = 0.0;
  double antiSum = 0.0;
  double diagSumLow = 0.0;
  double antiSumLow = 0.0;
  // currently borders are ignored!
  for (unsigned int i = 1; i < m.size()-1; ++i) {
    for (unsigned int j = 1; j < i; ++j) {
      if (m[i][j] >= thresh) {
	++countThresh;
	antiSum += m[i+1][j-1];
	antiSum += m[i-1][j+1];
	diagSum += m[i-1][j-1];
	diagSum += m[i+1][j+1];
// 	if (m[i-1][j-1] >= thresh) {
// 	  ++countDiag;
// 	}
// 	if (m[i+1][j+1] >= thresh) {
// 	  ++countDiag;
// 	}
// 	if (m[i-1][j+1] >= thresh) {
// 	  ++countAntiDiag;
// 	}
// 	if (m[i+1][j-1] >= thresh) {
// 	  ++countAntiDiag;
// 	}
      }
//       else {
// 	++countBelowThresh;
// 	antiSumLow += m[i+1][j-1];
// 	antiSumLow += m[i-1][j+1];
// 	diagSumLow += m[i-1][j-1];
// 	diagSumLow += m[i+1][j+1];
//       }
    }
  }
  if (countThresh == 0) {
    return 0.0;
  }
//   double all = countDiag + countAntiDiag;
//   if (all <= 0.0) {
//     return 0.0;
//   }
//   double result = countAntiDiag / all;
  double result = (antiSum - diagSum); //  / countThresh;
  double result2 = 0.0;
  // count the cells below threshold the other way around
  if (countBelowThresh > 0) {
    result2 = (diagSumLow - antiSumLow) / countBelowThresh;
  }
  // fraction of cells above threshold
  // double result3 = countThresh / (m.size() * m.size());
  result = result; //  + result2 - result3;
  return result;
}


double
alignmentQualityScore(const string& leftOutSequence,
		      const string& alphabet,
		      unsigned int numSequences,
		      const Vec<Vec<double> >& leaveOneOutCount,
		      const Vec<Vec<Vec<double> > >& leaveOneOutCountMatrix,
		      double entropyWeight,
		      double matchWeight,
		      double ratioWeight,
		      double skewThreshold)
{ 
  // cout << "Starting quality score... " << endl;
  // entropy term:
  double minVal, maxVal;
  Vec<Vec<double> > compMatrix 
    = computeCompensatoryScores(leftOutSequence, alphabet, numSequences,
				leaveOneOutCount, leaveOneOutCountMatrix, minVal, maxVal);
  double entropyScore = skewness(compMatrix, skewThreshold); // bimodalSeparation(matrix2Vec(compMatrix), ratioWeight);
  // sequence alignment term: NEGATIVE entropy if high order -> is "good" -> is high score
  double aliScore = - sequenceAlignmentEntropy(leftOutSequence, alphabet,
	       numSequences, leaveOneOutCount);
  return ((entropyWeight * entropyScore) + (matchWeight * aliScore));
}

double
alignmentQualityScore(const string& leftOutSequence,
		      const string& alphabet,
		      unsigned int numSequences,
		      const Vec<Vec<double> >& leaveOneOutCount,
		      const Vec<Vec<Vec<double> > >& leaveOneOutCountMatrix,
		      double entropyWeight,
		      double matchWeight,
		      double ratioWeight,
		      double skewThreshold,
		      int startA, 
		      int startB,
		      int winLen)
{ 
  // cout << "Starting quality score... " << endl;
  // entropy term:
  double minVal, maxVal;
  Vec<Vec<double> > compMatrix 
    = computeCompensatoryScores(leftOutSequence, alphabet, numSequences,
	leaveOneOutCount, leaveOneOutCountMatrix, minVal, maxVal,
				startA, startB, winLen);
  double entropyScore = skewness(compMatrix, skewThreshold); // bimodalSeparation(matrix2Vec(compMatrix), ratioWeight);
  // sequence alignment term: NEGATIVE entropy if high order -> is "good" -> is high score
  double aliScore = - sequenceAlignmentEntropy(leftOutSequence, alphabet,
	       numSequences, leaveOneOutCount, startA, startB, winLen);
  return ((entropyWeight * entropyScore) + (matchWeight * aliScore));
}


double
alignmentQualityScore(const SequenceAlignment& ali, const CompensationScorer& scorer,
		      double ratioWeight)
{ 
  ERROR("Deprecated!");
//   Vec<Vec<double> > compMatrix 
//     = computeCompensatoryScores(ali, scorer);
  // return bimodalSeparation(matrix2Vec(compMatrix), ratioWeight);
  return 0.0;
}

/** skewthreshold is used if it is not smaller or equal -999.0, otherwise 
    it is set to the average of minimum and maximum value of the scoring matrix */
double
alignmentQualityScore(const SequenceAlignment& ali,
		      const string& alphabet,
		      double entropyWeight,
		      double matchWeight,
		      double ratioWeight,
		      double skewThreshold)

{ 
  unsigned int leaveOut = 0; // leave out this sequence in counting
  const Vec<string>& sequences = ali.getSequences();
  unsigned int numSequences = sequences.size();
  Vec<Vec<double> > leaveOneOutCountVec = generateLeaveOneOutCountVector(sequences, leaveOut, alphabet);
  Vec<Vec<Vec<double> > > leaveOneOutCountMatrix
    = generateLeaveOneOutCountMatrix(sequences, leaveOut, alphabet);
  // entropy term:
  double minVal = 0.0;
  double maxVal = 0.0;
  Vec<Vec<double> > compMatrix 
    = computeCompensatoryScores(sequences[leaveOut], alphabet, numSequences,
				leaveOneOutCountVec, leaveOneOutCountMatrix,
				minVal, maxVal);
  if (skewThreshold <= -999) {
    skewThreshold = 0.5 * (maxVal + minVal);
  }
  double entropyScore = skewness(compMatrix, skewThreshold); // bimodalSeparation(matrix2Vec(compMatrix), ratioWeight);
  // sequence alignment term: NEGATIVE entropy if high order -> is "good" -> is high score
  double aliScore = - sequenceAlignmentEntropy(sequences[leaveOut], alphabet,
	       numSequences, leaveOneOutCountVec);
  return ((entropyWeight * entropyScore) + (matchWeight * aliScore));
}

/** optimize n'th sequence of alignmnent */
// 			  unsigned int shiftMax,
// 			  unsigned int insertMax,
// 			  unsigned int delMax)
double
optimizeAlignmentSequenceFast(SequenceAlignment& ali, 
			      const string& alphabet,
			      unsigned int n,
			      double entropyWeight,
			      double matchWeight,
			      double ratioWeight,
			      double skewThreshold)
{
  ERROR("Sorry, fast sequence optimization not supported anymore!");
//   string sequenceOrig = ali.getSequence(n);
//   // SequenceMutator mutator(sequenceOrig); 
//   Vec<string> sequences = ali.getSequences();
//   // cout << "Generating count matrix... " << endl;
//   Vec<Vec<Vec<double > > > countMatrix = generateLeaveOneOutCountMatrix(
// 				sequences, n, alphabet);
//   // cout << "Generating count vector... " << endl;
//   Vec<Vec<double> > countVec = generateLeaveOneOutCountVector(sequences,
// 							      n, alphabet);
//   // cout << "Result of countVector: " << countVec << endl;
//   string bestSequence = sequenceOrig;
//   double bestScore = alignmentQualityScore(sequenceOrig, alphabet, 
// 					   sequences.size(),
// 					   countVec, countMatrix, 
// 					   entropyWeight, matchWeight, 
// 					   ratioWeight, skewThreshold);  
//   // cout << "Initial score: " << bestScore << endl;
//   double score;
//   while (mutator.hasNext()) {
//     mutator.nextSequence();
//     string seq = mutator.getSequenceCurr();
//     // ali.setTemplate(seq, n);
//     // score = alignmentQualityScore(ali, scorer);
//     score = alignmentQualityScore(seq, alphabet, sequences.size(),
// 	  countVec, countMatrix, entropyWeight, matchWeight, ratioWeight,
// 				  skewThreshold);
//     // cout << "modified sequence: " << score << " " << seq << endl;
//     if (score > bestScore) {
//       bestSequence = seq;
//       bestScore = score;
//     }
//   }
//   ali.setTemplate(bestSequence, ali.getName(n),  n);
//   return bestScore;
  return 0.0;
}

/** optimize n'th sequence of alignmnent
 @todo: use smaller window*/
double
optimizeAlignmentSequenceFastSubset(SequenceAlignment& ali, 
				    SequenceAlignment& secondaryAli,
				    const string& alphabet,
				    unsigned int n,
				    double entropyWeight,
				    double matchWeight,
				    double ratioWeight,
				    double skewThreshold,
				    int startA,
				    int startB,
				    int totLen,
				    int borderLen,
				    char gapChar)
{
  PRECOND(n < ali.size());
  ERROR_IF(!((secondaryAli.size() == ali.size())
	  || (secondaryAli.size() == 0)),
	   "Secondary structure alignment must have same size or be empty!");
  cout << "Starting optimizeAlignmentSequenceFastSubset!" << endl;
  string sequenceOrig = ali.getSequence(n);
  Vec<string> sequences = ali.getSequences();
  Vec<string> secondaries = secondaryAli.getSequences();
  cout << "Generating count matrix... " << endl;
  Vec<Vec<Vec<double > > > countMatrix = generateLeaveOneOutCountMatrix(
				sequences, n, alphabet);
  // cout << "Generating count vector... " << endl;
  Vec<Vec<double> > countVec = generateLeaveOneOutCountVector(sequences,
					      n, alphabet);
  // cout << "Result of countVector: " << countVec << endl;
  string bestSequence = sequenceOrig;
  cout << "Starting sequence: " 
       << bestSequence.substr(startA, totLen)
       << "\t" << bestSequence.substr(startB, totLen) << endl;
  double bestScore = alignmentQualityScore(sequenceOrig, alphabet, 
					   sequences.size(),
					   countVec, countMatrix, 
					   entropyWeight, matchWeight, 
					   ratioWeight, skewThreshold,
					   startA, startB, totLen);
  PRECOND(totLen > 0);
  cout << "Initial score: " << bestScore << endl;
  double score;
  int helpA1 = startA - borderLen;
  int helpA2 = startA + totLen + borderLen;
  int helpB1 = startB - borderLen;
  int helpB2 = startB + totLen + borderLen;
  ERROR_IF((helpA1 < 0) || (helpA2 >= static_cast<int>(sequenceOrig.size())), 
	   "Illegal first intervall parameters found!");
  ERROR_IF((helpB1 < 0) || (helpB2 >= static_cast<int>(sequenceOrig.size())), 
	   "Illegal second intervall parameters found!");
  cout << "Original sequence intervalls: " 
       << bestSequence.substr(helpA1, borderLen) << " " << bestSequence.substr(startA, totLen) << " "
       << bestSequence.substr(startA + totLen, borderLen) << "\t"
       << bestSequence.substr(startB-borderLen, borderLen) << " " << bestSequence.substr(startB, totLen) << " "
       << bestSequence.substr(startB+totLen, borderLen) << endl;
  for (int i = -borderLen; i <= borderLen; ++i) { // shift of 5' region 
    // cout << "Working on shift A: " << i << endl;
    string seq = sequenceOrig;
    if (i < 0) {
      // insert i gaps at last position
      seq.insert(helpA2,
		 static_cast<string::size_type>(abs(i)), gapChar);
      // delete i characters at first position
      seq.erase(helpA1, abs(i));
    }
    else if (i > 0) {
      // delete i characters at last position
      seq.erase(helpA2, i);
      // insert i gaps at position startPos
      seq.insert(helpA1, static_cast<string::size_type>(i), gapChar);
    }
    for (int j = -borderLen; j <= borderLen; ++j) { // shift of 3' region 
      string seq2 = seq; // reset to result of outer loop
      // cout << "Working on shift B: " << j << endl;
      if (j < 0) {
	// insert j gaps at last position
	seq2.insert(helpB2,
		   static_cast<string::size_type>(abs(j)), gapChar);
	// delete j characters at first position
	seq2.erase(helpB1, abs(j));
      }
      else if (j > 0) {
	// delete j characters at last position
	seq2.erase(helpB2, j);
	// insert j gaps at position startPos
	seq2.insert(helpB1, static_cast<string::size_type>(j), gapChar);
      }
      // ali.setTemplate(seq, n);
      // score = alignmentQualityScore(ali, scorer);
      // @todo: use smaller inteverall
      score = alignmentQualityScore(seq2, alphabet, sequences.size(),
				    countVec, countMatrix, entropyWeight, matchWeight, ratioWeight,
				    skewThreshold,
				    startA, startB, totLen);
      // cout << "modified sequence: " << score << " " << seq2 << endl;
      if (score > bestScore) {
	bestSequence = seq2;
	bestScore = score;
	cout << "New best intervall found: " 
	     << bestSequence.substr(helpA1, borderLen) << " " << bestSequence.substr(startA, totLen) << " "
	     << bestSequence.substr(startA + totLen, borderLen) << "\t"
	     << bestSequence.substr(startB-borderLen, borderLen) << " " << bestSequence.substr(startB, totLen) << " "
	     << bestSequence.substr(startB+totLen, borderLen) << endl;
	cout << "Best new sequence found: " << bestScore << " Original and modified sequence: " << endl 
	     << sequenceOrig << endl << seq << endl;
      }
    }
  }
  cout << "Best found sequence intervalls: " 
       << bestSequence.substr(helpA1, borderLen) << " " << bestSequence.substr(startA, totLen) << " "
       << bestSequence.substr(startA + totLen, borderLen) << "\t"
       << bestSequence.substr(startB-borderLen, borderLen) << " " << bestSequence.substr(startB, totLen) << " "
       << bestSequence.substr(startB+totLen, borderLen) << endl;
  cout << "Best sequence found: " << bestScore << " Original and modified sequence: " << endl 
       << sequenceOrig << endl << bestSequence << endl;
  ali.setSequence(bestSequence, ali.getName(n), n);
  return bestScore;
}

void
optimizeAlignmentRound(SequenceAlignment& ali,
		       const string& alphabet,
		       double entropyWeight,
		       double matchWeight,
		       double ratioWeight,
		       double skewThreshold)
		       // const CompensationScorer& scorer)
{
  Vec<unsigned int> stair = generateStair(ali.size(), 0);
  Random& rnd = Random::getInstance();
  random_shuffle(stair.begin(), stair.end(), rnd);
  for (unsigned int i = 0; i < stair.size(); ++i) {
    // optimize sequence with index stair[i]
    double result = optimizeAlignmentSequenceFast(ali, alphabet, stair[i], entropyWeight, matchWeight, ratioWeight,
				  skewThreshold);
    cout << "Optimized sequence " << i + 1 << " " 
	 << stair[i] + 1 << " " << result << endl;
  }
}

void
optimizeAlignmentRoundSubset(SequenceAlignment& aliOrig,
			     SequenceAlignment& secondaryAli,
			     const string& alphabet,
			     double entropyWeight,
			     double matchWeight,
			     double ratioWeight,
			     double skewThreshold,
			     int startA,
			     int startB,
			     int totLen,
			     int borderLen)
		       // const CompensationScorer& scorer)
{
  Vec<unsigned int> stair = generateStair(aliOrig.size(), 0);
  SimpleSequenceAlignment ali;
  Random& rnd = Random::getInstance();
  random_shuffle(stair.begin(), stair.end(), rnd);
  cout << "Step 1 : adding sequence " << stair[0]+1 << endl;
  ali.addSequence(aliOrig.getSequence(stair[0]),
		  aliOrig.getName(stair[0]));
  for (unsigned int i = 1; i < stair.size(); ++i) {
    cout << "Step " << i + 1 << " adding sequence " << stair[i]+1
	 << endl;
    ali.addSequence(aliOrig.getSequence(stair[i]),
		    aliOrig.getName(stair[i]));
    // optimize newest sequence:
    double result = optimizeAlignmentSequenceFastSubset(ali, secondaryAli, alphabet, 
							ali.size()-1, 
		entropyWeight, matchWeight, ratioWeight,
		skewThreshold, startA, startB, totLen, borderLen, GAP_CHAR);
    cout << "Optimized sequence " << i + 1 << " " 
	 << stair[i] + 1 << " " << result << endl;

  }
}


SimpleSequenceAlignment
optimizeAlignment(const SimpleSequenceAlignment& aliOrig,
		  const string& alphabet,
		  // const CompensationScorer& scorer,
		  unsigned int numRounds,
		  double entropyWeight,
		  double matchWeight,
		  double ratioWeight,
		  double skewThreshold)
{
  SimpleSequenceAlignment ali = aliOrig;
  for (unsigned int i = 0; i < numRounds; ++i) {
    cout << "Starting round " << i + 1 << " of alignment optimization " << endl;
    optimizeAlignmentRound(ali, alphabet, entropyWeight, matchWeight, ratioWeight, skewThreshold);
  }
  return ali;
}


SimpleSequenceAlignment
optimizeAlignmentSubset(const SimpleSequenceAlignment& aliOrig,
			const SimpleSequenceAlignment& secondaryAliOrig,
			const string& alphabet,
			// const CompensationScorer& scorer,
			unsigned int numRounds,
			double entropyWeight,
			double matchWeight,
			double ratioWeight,
			double skewThreshold,
			const Vec<int>& startA,
			const Vec<int>& startB,
			const Vec<int>& totLen,
			const Vec<int>& borderLen)
{
  PRECOND((startA.size()== startB.size())
	  && (startA.size() == totLen.size())
	  && (startA.size() == borderLen.size()));
  SimpleSequenceAlignment ali = aliOrig;
  SimpleSequenceAlignment secondaryAli = secondaryAliOrig;
  for (unsigned int k = 0; k < startA.size(); ++k) {
    cout << "Optimizing intervalls " << startA[k] + 1 << " - " << startA[k]+totLen[k]+1
	 << "   " << startB[k] + 1 << " - " << startB[k] + totLen[k] + 1 << " with border "
	 << borderLen[k] << endl;
    for (unsigned int i = 0; i < numRounds; ++i) {
      cout << "Starting round " << i + 1 << " of alignment optimization " << endl;
      optimizeAlignmentRoundSubset(ali, secondaryAli, alphabet, entropyWeight, 
				   matchWeight, ratioWeight, skewThreshold,
				   startA[k], startB[k], totLen[k], borderLen[k]);
    }
  }
  return ali;
}

Vec<Vec<double> >
smoothAntiDiagGauss(const Vec<Vec<double> >&m, double g)
{
  cout << "Starting smoothAntiDiagGaussian with : " << g << endl;
  Vec<Vec<double> > result = m;
  int numDiag = 2 * m.size() - 1;
  for (int i = 0; i < numDiag; ++i) {
    Vec<double> antiDiag = getAntiDiagonal(m, i);
    // cout << "diagonal " << i << " " << antiDiag << endl;
    antiDiag = smoothGaussian(antiDiag, g);
    // cout << "diagonal " << i << " smoothed: " << antiDiag << endl;
    setAntiDiagonal(result, antiDiag, i);
  }
  return result;
}


/** returns indices: which is index of probability matrix of i'th sequence in ali? */
Vec<unsigned int>
generateAliNameIndices(const SequenceAlignment& ali,
		       const Vec<string>& names,
		       int maxChar)
{
  if (ali.size() == 0) {
    return Vec<unsigned int>();
  }
  cout << "Starting generateAliNameIndices" << endl;
  Vec<unsigned int> result(ali.size(), names.size());
  for (unsigned int i = 0; i < ali.size(); ++i ) {
    string name = ali.getName(i);
    if ((maxChar > 0) && (static_cast<int>(name.size()) > maxChar)) {
      name = name.substr(0, maxChar);
    }
    for (unsigned int j = 0; j < names.size(); ++j) {
      string aliName = names[j];
      if ((maxChar > 0) && (static_cast<int>(aliName.size()) > maxChar)) {
	aliName = aliName.substr(0, maxChar);
      }
      if (aliName.find(name) == 0) {
	result[i] = j;
	break;
      }
    }
  }
  cout << "Ending generateAliNameIndices" << endl;
  return result;
}

/** returns correlation coefficient between predicted contacts and entropy score */
double
probabilityCorrelation(const SequenceAlignment& ali, 
		       const Vec<Vec<double> >& compMatrix, 
		       const Vec<Vec<RankedSolution5<unsigned int, unsigned int> > >& probVectors,
		       const Vec<unsigned int>& probIndices,
		       const string& outFileName)
{
  unsigned int numProbEntries = countElements(probVectors);
  Vec<double> probValues(numProbEntries);
  Vec<double> matrixValues(numProbEntries);
  int pc = 0;
  unsigned int nx, ny;
  for (unsigned int i = 0; i < ali.size(); ++i) {
    unsigned int probId = probIndices[i];
    if (probId < probVectors.size()) {
      for (unsigned int j = 0; j < probVectors[probId].size(); ++j) {
	probValues[pc] = probVectors[probId][j].first;
	nx = probVectors[probId][j].second;
	ny = probVectors[probId][j].third;
	matrixValues[pc++] = compMatrix[nx][ny];
      }
    }
    else {
      cout << "Warning: Could not find probability file for : " <<
	ali.getName(i) << endl;
    }
  }
  if (pc < static_cast<int>(matrixValues.size())) {
    cout << "Warning: not all probability values used!" << endl;
  }
  double result = correlationCoefficient(probValues, matrixValues);
  if (outFileName.size() > 0) {
    ofstream outfile(outFileName.c_str());
    for (unsigned int i = 0; i < probValues.size(); ++i) {
      outfile << i + 1 << " " << probValues[i] << " " << matrixValues[i] << endl;
    }
    outfile.close();
  }
  return result;
}



void
readNeuralFileElement(istream& is,
		      Vec<Vec<Vec<double> > >& mtx,
		      Vec<double>& globalFeatures,
		      double& resultClass,
		      int numGlobalFeatures,
		      int numMatrices,
		      int patchSize,
		      int& nx,
		      int& ny)
{
  // os << nx + 1 << " " << ny + 1 << " ";
  int totSize = 0; // number of feature vector elements not counting result column or first three columns
  double val;
  mtx.clear();
  globalFeatures.clear();
  resultClass = 0.0;
  nx = 0;
  ny = 0;
  is >> nx;
  if (!is) {
    return; 
  }
  is >> ny;
  if (!is) {
    return; 
  }
  is >> totSize;
  if (!is) {
    return; 
  }
  int numRead = 0;
  for (int i = 0; i < numGlobalFeatures; ++i) {
    is >> val;
    globalFeatures.push_back(val);
    ++numRead;
  }
  int len = (2*patchSize+1);
  // os << (globalFeatures.size() + (mtx.size() * len * len)) << " ";
  mtx = Vec<Vec<Vec<double> > >(numMatrices, Vec<Vec<double> >(len, Vec<double>(len, 0.0)));
  int mx = patchSize;
  int my = patchSize; // indices of middle point
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    is >> mtx[k][mx][my];
    ++numRead;
    // organize in circles around center point
    for (int i = 1; i <= patchSize; ++i) {
      // upper side
      for (int x = mx - i + 1; x <= mx+i; ++x) {
	is >> mtx[k][x][my-i];
	++numRead;
      }
      // right side go down
      for (int y = my - i + 1; y <= my + i; ++y) {
	is >> mtx[k][mx+i][y];
	++numRead;
      }
      // lower side
      for (int x = mx - i; x < mx+i; ++x) {
	is >> mtx[k][x][my+i];
	++numRead;
      }
      // left side
      for (int y = my - i; y < my + i; ++y) {
	is >> mtx[k][mx-i][y];
	++numRead;
      }
    }
  }
  ERROR_IF(numRead != totSize, "Inconsistent feature vector size!");
  is >> resultClass;
  ++numRead;
}


void
writeNeuralFileElement1(ostream& os,
		       const Vec<Vec<Vec<double> > >& mtx,
		       const Vec<double>& globalFeatures,
		       int patchSize,
		       int nx,
		       int ny)
{
  // int len = (2*patchSize+1);
  os << nx + 1 << " " << ny + 1 << " ";
//   os << (globalFeatures.size() + (mtx.size() * len * len)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    os << mtx[k][nx][ny] << " ";
    // organize in circles around center point
    for (int i = 1; i <= patchSize; ++i) {
      // upper side
      for (int x = nx - i + 1; x <= nx+i; ++x) {
	os << mtx[k][x][ny-i] << " ";
      }
      // right side go down
      for (int y = ny - i + 1; y <= ny + i; ++y) {
	os << mtx[k][nx+i][y] << " ";
      }
      // lower side
      for (int x = nx - i; x < nx+i; ++x) {
	os << mtx[k][x][ny+i] << " ";
      }
      // left side
      for (int y = ny - i; y < ny + i; ++y) {
	os << mtx[k][nx-i][y] << " ";
      }
    }
  }
}

/** only write "cross" with diagonal and antidiagonal !*/
void
writeNeuralFileElement2(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    for (int i = 1; i <= patchSize; ++i) {
      // normal diagonal:
      os << mtx[k][nx+i][ny+i] << " ";
      // anti-diagonal:
      os << mtx[k][nx-i][ny+i] << " ";
      // anti-diagonal of other end:
      os << mtx[k][nx+i][ny-i] << " ";
      // normal diagonal of other end:
      os << mtx[k][nx-i][ny-i] << " ";
    }
  }
}

/** only write "cross" with diagonal and antidiagonal !*/
void
writeNeuralFileElement3(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  // int len = (2*patchSize+1);
  // int numCross = len;
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    // organize in circles around center point
    for (int i = -patchSize; i <= patchSize; ++i) {
      if (i == 0) {
	continue;
      }
      double val = 0.5 + (0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny+i]));
      // normal diagonal:
      os << val << " ";
    }
  }
}

/** only write "cross" with diagonal and antidiagonal !*/
void
writeNeuralFileElement4(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
//   int len = (2*patchSize+1);
//   int numCross = len;
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    // organize in circles around center point
    for (int i = 1; i <= patchSize; ++i) {
      double val1 = 0.5 + ( 0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny+i]));
      double val2 = 0.5 + ( 0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx-i][ny+i]));
      double val = 0.5 * (val1 + val2);
      // normal diagonal:
      os << val << " ";
    }
  }
}

/** only write "cross" with diagonal and antidiagonal !*/
void
writeNeuralFileElement5(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  // int len = (2*patchSize+1);
  // int numCross = len;
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
  //   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
  //     os << globalFeatures[i] << " ";
  //   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    for (int i = 1; i <= patchSize; ++i) {
      double val1 = 0.5 + ( 0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny+i]));
      double val2 = 0.5 + ( 0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx-i][ny+i]));
      // normal diagonal:
      os << val1 << " " << val2 << " ";
    }
  }
}

/** only write "cross" with diagonal and antidiagonal, 
    complete for first neighbors, average for all subsequent neighbors !*/
void
writeNeuralFileElement6(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    // organize in circles around center point
    for (int i = 1; i <= 1; ++i) {
      // normal diagonal:
      os << mtx[k][nx+i][ny+i] << " ";
      // anti-diagonal:
      os << mtx[k][nx-i][ny+i] << " ";
      // anti-diagonal of other end:
      os << mtx[k][nx+i][ny-i] << " ";
      // normal diagonal of other end:
      os << mtx[k][nx-i][ny-i] << " ";
    }
    for (int i = 2; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      os << (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i]) << " ";
      // anti-diagonal - normal diagonal of other end:
      os << mtx[k][nx-i][ny-i]  - mtx[k][nx+i][ny-i] << " ";
    }
  }
}

/** only write "cross" with diagonal and antidiagonal, 
    complete for first neighbors (also average of non-diagonals), 
    average for all subsequent neighbors. */
void
writeNeuralFileElement7(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    // organize in circles around center point
    for (int i = 1; i <= 1; ++i) {
      // normal diagonal:
      os << mtx[k][nx+i][ny+i] << " ";
      // anti-diagonal:
      os << mtx[k][nx-i][ny+i] << " ";
      // anti-diagonal of other end:
      os << mtx[k][nx+i][ny-i] << " ";
      // normal diagonal of other end:
      os << mtx[k][nx-i][ny-i] << " ";

      // non-diagonal neighbors:
      os << 0.5 * (mtx[k][nx][ny+i] + mtx[k][nx][ny-i]) << " ";
      os << 0.5 * (mtx[k][nx+i][ny] + mtx[k][nx-i][ny]) << " ";

    }
    for (int i = 2; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      os << (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i]) << " ";
      // anti-diagonal - normal diagonal of other end:
      os << mtx[k][nx-i][ny-i]  - mtx[k][nx+i][ny-i] << " ";
    }
  }
}

/** only write "cross" with diagonal and antidiagonal, 
    complete for first neighbors (also all non-diagonals), 
    average for all subsequent neighbors. */
void
writeNeuralFileElement8(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    // organize in circles around center point
    for (int i = 1; i <= 1; ++i) {
      // normal diagonal:
      os << mtx[k][nx+i][ny+i] << " ";
      // anti-diagonal:
      os << mtx[k][nx-i][ny+i] << " ";
      // anti-diagonal of other end:
      os << mtx[k][nx+i][ny-i] << " ";
      // normal diagonal of other end:
      os << mtx[k][nx-i][ny-i] << " ";

      // non-diagonal neighbors:
      os << mtx[k][nx][ny+i] << " ";
      os << mtx[k][nx-i][ny] << " ";
      os << mtx[k][nx][ny+i] << " ";
      os << mtx[k][nx-i][ny] << " ";

    }
    for (int i = 2; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      os << (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i]) << " ";
      // anti-diagonal - normal diagonal of other end:
      os << mtx[k][nx-i][ny-i]  - mtx[k][nx+i][ny-i] << " ";
    }
  }
}


/** only write "cross" with diagonal and antidiagonal, 
    average for first neighbors (also average of non-diagonals), 
    average for all subsequent neighbors.
    Like writeNeuralFileElement9 but also writing border elements.
*/
void
writeNeuralFileElement9b(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  // cout << "Writing "  << nx + 1 << " " << ny + 1 << endl;
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    int nn = mtx[k].size();
    // write center element
    os << mtx[k][nx][ny] << " ";
    // non-diagonal neighbors:
    if (ny+1 < nn) {
      if (ny - 1 >= 0) {
	os << 0.5 * (mtx[k][nx][ny+1] + mtx[k][nx][ny-1]) << " ";
      }
      else {
	os << 0.5 * (mtx[k][nx][ny+1] + mtx[k][nx][ny+1]) << " ";
      }
    }
    else {
      os << 0.5 * (mtx[k][nx][ny-1] + mtx[k][nx][ny-1]) << " ";
    }
    if (nx+1 < nn) {
      if ((nx - 1)  >= 0) {
	os << 0.5 * (mtx[k][nx+1][ny] + mtx[k][nx-1][ny]) << " ";
      }
      else {
	os << 0.5 * (mtx[k][nx+1][ny] + mtx[k][nx+1][ny]) << " ";
      }
    }
    else {
      os << 0.5 * (mtx[k][nx-1][ny] + mtx[k][nx-1][ny]) << " ";
    }
    for (int i = 1; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      if ((nx - i) >= 0) {
	if ((ny + i) < nn) {
	  if ((nx + i) < nn) {
	    // this is the "normal" case, the other lines are special bordering cases
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i])) << " ";
	  }
	  else {
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx-i][ny+i])) << " ";
	  }
	}
	else {
	  if ((nx + i) < nn) {
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx+i][ny-i])) << " ";
	  }
	  else {
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx-i][ny-i])) << " ";
	  }
	}
      }
      else {  // nx - i < 0 !
	if ((ny + i) < nn) {
	  os << 0.5 + (0.5 * (mtx[k][nx+i][ny+i] - mtx[k][nx+i][ny+i])) << " ";
	}
	else {
	  os << 0.5 + (0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny-i])) << " ";
	}
      }
      // anti-diagonal of other end:
      if ((nx - i) >= 0) {
	if ((ny - i) >= 0) {
	  if (nx + i < nn) {
	    // this is the normal case, the other lines are workarounds for bordering cases
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx+i][ny-i])) << " ";
	  }
	  else {
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx-i][ny-i])) << " ";
	  }
	}
	else {
	  if (nx + i < nn) {
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i])) << " ";
	  }
	  else {
	    os << 0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx-i][ny+i])) << " ";
	  }
	}
      }
      else {
	if ((ny - i) >= 0) {
	  os << 0.5 + (0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny-i])) << " ";
	}
	else {
	  os << 0.5 + (0.5 * (mtx[k][nx+i][ny+i] - mtx[k][nx+i][ny+i])) << " ";
	}
      }
    }
  }
  // cout << "Ending writeneuralFileElement9b!" << endl;
}

/** only write "cross" with diagonal and antidiagonal, 
    average for first neighbors (also average of non-diagonals), 
    average for all subsequent neighbors.
    Like writeNeuralFileElement9 but also writing border elements.
*/
Vec<double>
generateNeuralFileElement9b(const Vec<Vec<Vec<double> > >& mtx,
			    const Vec<double>& globalFeatures,
			    int patchSize,
			    int nx,
			    int ny)
{
  // cout << "Writing "  << nx + 1 << " " << ny + 1 << endl;
  Vec<double> ov;
  ov.push_back(nx + 1);
  ov.push_back(ny + 1);
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    int nn = mtx[k].size();
    // write center element
    ov.push_back(mtx[k][nx][ny]);
    // non-diagonal neighbors:
    if (ny+1 < nn) {
      if (ny - 1 >= 0) {
	ov.push_back(0.5 * (mtx[k][nx][ny+1] + mtx[k][nx][ny-1]));
      }
      else {
	ov.push_back(0.5 * (mtx[k][nx][ny+1] + mtx[k][nx][ny+1]));
      }
    }
    else {
      ov.push_back(0.5 * (mtx[k][nx][ny-1] + mtx[k][nx][ny-1]));
    }
    if (nx+1 < nn) {
      if ((nx - 1)  >= 0) {
	ov.push_back(0.5 * (mtx[k][nx+1][ny] + mtx[k][nx-1][ny]));
      }
      else {
	ov.push_back(0.5 * (mtx[k][nx+1][ny] + mtx[k][nx+1][ny]));
      }
    }
    else {
      ov.push_back(0.5 * (mtx[k][nx-1][ny] + mtx[k][nx-1][ny]));
    }
    for (int i = 1; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      if ((nx - i) >= 0) {
	if ((ny + i) < nn) {
	  if ((nx + i) < nn) {
	    // this is the "normal" case, the other lines are special bordering cases
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i])));
	  }
	  else {
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx-i][ny+i])));
	  }
	}
	else {
	  if ((nx + i) < nn) {
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx+i][ny-i])));
	  }
	  else {
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx-i][ny-i])));
	  }
	}
      }
      else {  // nx - i < 0 !
	if ((ny + i) < nn) {
	  ov.push_back(0.5 + (0.5 * (mtx[k][nx+i][ny+i] - mtx[k][nx+i][ny+i])));
	}
	else {
	  ov.push_back(0.5 + (0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny-i])));
	}
      }
      // anti-diagonal of other end:
      if ((nx - i) >= 0) {
	if ((ny - i) >= 0) {
	  if (nx + i < nn) {
	    // this is the normal case, the other lines are workarounds for bordering cases
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx+i][ny-i])));
	  }
	  else {
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx-i][ny-i])));
	  }
	}
	else {
	  if (nx + i < nn) {
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i])));
	  }
	  else {
	    ov.push_back(0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx-i][ny+i])));
	  }
	}
      }
      else {
	if ((ny - i) >= 0) {
	  ov.push_back(0.5 + (0.5 * (mtx[k][nx+i][ny-i] - mtx[k][nx+i][ny-i])));
	}
	else {
	  ov.push_back(0.5 + (0.5 * (mtx[k][nx+i][ny+i] - mtx[k][nx+i][ny+i])));
	}
      }
    }
  }
  // cout << "Ending writeneuralFileElement9b!" << endl;
  return ov;
}


/** only write "cross" with diagonal and antidiagonal, 
    average for first neighbors (also average of non-diagonals), 
    average for all subsequent neighbors. */
void
writeNeuralFileElement9(ostream& os,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  os << nx + 1 << " " << ny + 1 << " ";
  // os << (globalFeatures.size() + (mtx.size() * numCross)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     os << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    // write center element
    os << mtx[k][nx][ny] << " ";
    // non-diagonal neighbors:
    os << 0.5 * (mtx[k][nx][ny+1] + mtx[k][nx][ny-1]) << " ";
    os << 0.5 * (mtx[k][nx+1][ny] + mtx[k][nx-1][ny]) << " ";
    for (int i = 1; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      os << 0.5 + (0.5 * (mtx[k][nx-i][ny+i] - mtx[k][nx+i][ny+i])) << " ";
      // anti-diagonal of other end:
      os << 0.5 + (0.5 * (mtx[k][nx-i][ny-i] - mtx[k][nx+i][ny-i])) << " ";
    }
  }
}

/** write diagonal elements and nearest non-diagonal elements
    average for first neighbors (also average of non-diagonals), 
    Like writeNeuralFileElement9 but also writing border elements.
*/
void
writeNeuralFileElement10b(ostream& ov,
			const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  // cout << "Writing "  << nx + 1 << " " << ny + 1 << endl;
  ov << nx + 1 << " " << ny + 1 << " ";
  // ov << (globalFeatures.size() + (mtx.size() * numCrovs)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     ov << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    int nn = mtx[k].size();
    // write center element
    ov << mtx[k][nx][ny] << " ";
    // non-diagonal neighbors:
    if (ny+1 < nn) {
      if (ny - 1 >= 0) {
	ov << mtx[k][nx][ny+1] << " " << mtx[k][nx][ny-1] << " ";
      }
      else {
	ov << mtx[k][nx][ny+1] << " " <<  mtx[k][nx][ny+1] << " ";
      }
    }
    else {
      ov << mtx[k][nx][ny-1] << " " << mtx[k][nx][ny-1] << " ";
    }
    if (nx+1 < nn) {
      if ((nx - 1)  >= 0) {
	ov << mtx[k][nx+1][ny] << " " << mtx[k][nx-1][ny] << " ";
      }
      else {
	ov << mtx[k][nx+1][ny] << " " << mtx[k][nx+1][ny] << " ";
      }
    }
    else {
      ov << mtx[k][nx-1][ny] << " " << mtx[k][nx-1][ny] << " ";
    }
    for (int i = 1; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      if ((nx - i) >= 0) {
	if ((ny + i) < nn) {
	  if ((nx + i) < nn) {
	    // this is the "normal" case, the other lines are special bordering cases
	    ov << mtx[k][nx-i][ny+i] << " " << mtx[k][nx+i][ny+i] << " ";
	  }
	  else {
	    ov << mtx[k][nx-i][ny+i] << " " << mtx[k][nx-i][ny+i] << " ";
	  }
	}
	else {
	  if ((nx + i) < nn) {
	    ov << mtx[k][nx-i][ny-i] << " " << mtx[k][nx+i][ny-i] << " ";
	  }
	  else {
	    ov << mtx[k][nx-i][ny-i] << " " << mtx[k][nx-i][ny-i] << " ";
	  }
	}
      }
      else {  // nx - i < 0 !
	if ((ny + i) < nn) {
	  ov << mtx[k][nx+i][ny+i] << " " << mtx[k][nx+i][ny+i] << " ";
	}
	else {
	  ov << mtx[k][nx+i][ny-i] << " " <<  mtx[k][nx+i][ny-i] << " ";
	}
      }
      // anti-diagonal of other end:
      if ((nx - i) >= 0) {
	if ((ny - i) >= 0) {
	  if (nx + i < nn) {
	    // this is the normal case, the other lines are workarounds for bordering cases
	    ov << mtx[k][nx-i][ny-i] << " " << mtx[k][nx+i][ny-i] << " ";
	  }
	  else {
	    ov << mtx[k][nx-i][ny-i] << " " << mtx[k][nx-i][ny-i] << " ";
	  }
	}
	else {
	  if (nx + i < nn) {
	    ov << mtx[k][nx-i][ny+i] << " " << mtx[k][nx+i][ny+i] << " ";
	  }
	  else {
	    ov << mtx[k][nx-i][ny+i] << " " << mtx[k][nx-i][ny+i] << " ";
	  }
	}
      }
      else {
	if ((ny - i) >= 0) {
	  ov << mtx[k][nx+i][ny-i] << " " << mtx[k][nx+i][ny-i] << " ";
	}
	else {
	  ov << mtx[k][nx+i][ny+i] << " " << mtx[k][nx+i][ny+i] << " ";
	}
      }
    }
  }
  // cout << "Ending writeneuralFileElement9b!" << endl;
}


/** write diagonal elements and nearest non-diagonal elements
    average for first neighbors (also average of non-diagonals), 
    Like writeNeuralFileElement9 but also writing border elements.
*/
Vec<double>
generateNeuralFileElement10b(const Vec<Vec<Vec<double> > >& mtx,
			const Vec<double>& globalFeatures,
			int patchSize,
			int nx,
			int ny)
{
  // cout << "Writing "  << nx + 1 << " " << ny + 1 << endl;
  Vec<double> ov;
  ov.push_back(nx + 1.0);
  ov.push_back(ny + 1.0);
  // ov << (globalFeatures.size() + (mtx.size() * numCrovs)) << " ";
//   for (unsigned int i = 0; i < globalFeatures.size(); ++i) {
//     ov << globalFeatures[i] << " ";
//   }
  for (unsigned int k = 0; k < mtx.size(); ++k) {
    int nn = mtx[k].size();
    // write center element
    ov.push_back(mtx[k][nx][ny]);
    // non-diagonal neighbors:
    if (ny+1 < nn) {
      if (ny - 1 >= 0) {
	ov.push_back(mtx[k][nx][ny+1]);
	ov.push_back(mtx[k][nx][ny-1]);
      }
      else {
	ov.push_back(mtx[k][nx][ny+1]);
	ov.push_back( mtx[k][nx][ny+1]);
      }
    }
    else {
      ov.push_back(mtx[k][nx][ny-1]);
      ov.push_back(mtx[k][nx][ny-1]);
    }
    if (nx+1 < nn) {
      if ((nx - 1)  >= 0) {
	ov.push_back(mtx[k][nx+1][ny]);
	ov.push_back(mtx[k][nx-1][ny]);
      }
      else {
	ov.push_back(mtx[k][nx+1][ny]);
	ov.push_back(mtx[k][nx+1][ny]);
      }
    }
    else {
      ov.push_back(mtx[k][nx-1][ny]);
      ov.push_back(mtx[k][nx-1][ny]);
    }
    for (int i = 1; i <= patchSize; ++i) {
      // anti-diagonal - normal diagonal:
      if ((nx - i) >= 0) {
	if ((ny + i) < nn) {
	  if ((nx + i) < nn) {
	    // this is the "normal" case, the other lines are special bordering cases
	    ov.push_back(mtx[k][nx-i][ny+i]); 
	    ov.push_back(mtx[k][nx+i][ny+i]);
	  }
	  else {
	    ov.push_back(mtx[k][nx-i][ny+i]); 
	    ov.push_back(mtx[k][nx-i][ny+i]);
	  }
	}
	else {
	  if ((nx + i) < nn) {
	    ov.push_back(mtx[k][nx-i][ny-i]);
	    ov.push_back(mtx[k][nx+i][ny-i]);
	  }
	  else {
	    ov.push_back(mtx[k][nx-i][ny-i]);
	    ov.push_back(mtx[k][nx-i][ny-i]);
	  }
	}
      }
      else {  // nx - i < 0 !
	if ((ny + i) < nn) {
	  ov.push_back(mtx[k][nx+i][ny+i]);
	  ov.push_back(mtx[k][nx+i][ny+i]);
	}
	else {
	  ov.push_back(mtx[k][nx+i][ny-i]);
	  ov.push_back( mtx[k][nx+i][ny-i]);
	}
      }
      // anti-diagonal of other end:
      if ((nx - i) >= 0) {
	if ((ny - i) >= 0) {
	  if (nx + i < nn) {
	    // this is the normal case, the other lines are workarounds for bordering cases
	    ov.push_back(mtx[k][nx-i][ny-i]);
	    ov.push_back(mtx[k][nx+i][ny-i]);
	  }
	  else {
	    ov.push_back(mtx[k][nx-i][ny-i]);
	    ov.push_back(mtx[k][nx-i][ny-i]);
	  }
	}
	else {
	  if (nx + i < nn) {
	    ov.push_back(mtx[k][nx-i][ny+i]);
	    ov.push_back(mtx[k][nx+i][ny+i]);
	  }
	  else {
	    ov.push_back(mtx[k][nx-i][ny+i]);
	    ov.push_back(mtx[k][nx-i][ny+i]);
	  }
	}
      }
      else {
	if ((ny - i) >= 0) {
	  ov.push_back(mtx[k][nx+i][ny-i]);
	  ov.push_back(mtx[k][nx+i][ny-i]);
	}
	else {
	  ov.push_back(mtx[k][nx+i][ny+i]);
	  ov.push_back(mtx[k][nx+i][ny+i]);
	}
      }
    }
  }
  // cout << "Ending writeneuralFileElement9b!" << endl;
  return ov;
}

/** writes line of training data according to fileMode */
void
writeNeuralFileElement(ostream& os,
		       const Vec<Vec<Vec<double> > >& mtx,
		       const Vec<double>& globalFeatures,
		       int patchSize,
		       int nx,
		       int ny,
		       int fileMode,
		       int neuralAllMode)
{
  switch (fileMode) {
  case 1: writeNeuralFileElement1(os, mtx, globalFeatures,
				 patchSize, nx, ny);
    break;
  case 2: writeNeuralFileElement2(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 3: writeNeuralFileElement3(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 4: writeNeuralFileElement4(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 5: writeNeuralFileElement5(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 6: writeNeuralFileElement6(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 7: writeNeuralFileElement7(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 8: writeNeuralFileElement8(os, mtx, globalFeatures,
				  patchSize, nx, ny);
    break;
  case 9: 
    if (neuralAllMode) {
      writeNeuralFileElement9b(os, mtx, globalFeatures,
			       patchSize, nx, ny);
    }
    else {
      writeNeuralFileElement9(os, mtx, globalFeatures,
			      patchSize, nx, ny);
    }
    break;
  case 10: 
    writeNeuralFileElement10b(os, mtx, globalFeatures,
			     patchSize, nx, ny);
    break;
  default:
    ERROR("Unkonwn neural network output file mode");
  }
}

void
writeNeuralFileLine(ostream& os,
		    int nx,
		    int ny,
		    const Vec<Vec<Vec<double> > >& mtx,
		    const Vec<Vec<double> >& refMatrix,
		    const Vec<double>& globalFeatures,
		    int patchSize,
		    int fileMode,
		    int allMode,
		    int bothSideMode)
{
  ERROR_IF((nx < 0) || (ny < 0) || (nx > static_cast<int>(mtx.size())) 
	   || (ny > static_cast<int>(mtx.size())),
	   "Illegal matrix index in line 2618!");
  // cout << "writing writeNeuralFile!" << nx << " " << ny << endl;
  writeNeuralFileElement(os, mtx, globalFeatures,
			 patchSize, nx, ny, fileMode, allMode);
  os << " " << refMatrix[nx][ny];
  os << endl;
  if (bothSideMode) {
    writeNeuralFileElement(os, mtx, globalFeatures,
			   patchSize, ny, nx, fileMode, allMode);
    os << " " << refMatrix[ny][nx];
    os << endl;
  }
}

int
randomSign()
{
  Random& rnd = Random::getInstance();
  if (rnd.getRandf() >= 0.5) {
    return 1;
  }
  return -1;
}

void
writeNeuralFile(ostream& os,
		const Vec<Vec<Vec<double> > >& mtx,
		const Vec<Vec<double> >& refMatrix,
		const Vec<double>& globalFeatures,
		int patchSize,
		int neuralMaxLines,
		int fileMode,
		int allMode,
		int bothSideMode,
		double distBias)
{
  Random& rnd = Random::getInstance();
  // cout << "Starting writeNeuralFile! " << neuralMaxLines << endl;
  const int NEURAL_MAX_DIFF = 50; // maximum long distance interaction distance for mode "-2"
  int patchSizeBorder = patchSize;
  if (allMode) {
    patchSizeBorder = 0;
  }
  // for (unsigned int i = 0; i < mtx.size(); ++i) {
    // cout << i << " " << mtx[i].size() << " ";
  //     if (mtx[i].size() > 0) {
  //       cout << mtx[i][0].size() << endl;
  //     }
  //     else {
  //       cout << "0" << endl;
  //     }
  //   }
  int count = 0;
  for (int i = patchSizeBorder; i < static_cast<int>(mtx[0].size()) - patchSizeBorder; 
       ++i) {
    for (int j = patchSizeBorder; j < (i-(2*patchSizeBorder)); ++j) {
      ++count;  // slow, replace by formula
    }
  }
  Vec<pair<int, int> > points(count);
  count = 0;
  for (int i = patchSizeBorder; i < static_cast<int>(mtx[0].size()) - patchSizeBorder; 
       ++i) {
    for (int j = patchSizeBorder; j < (i-(2*patchSizeBorder)); ++j) {
      ERROR_IF(count >= static_cast<int>(points.size()), "Internal error in line 1288!");
      points[count].first = i;
      points[count].second = j;
      ++count;  // slow, replace by formula
    }
  }
  if (neuralMaxLines == -2) {
    for (int i = 0; i < static_cast<int>(points.size()); ++i) {
      int nx = points[i].first;
      int ny = points[i].second;
      if ((abs(nx-ny) > NEURAL_MAX_DIFF)) {
	if (refMatrix[nx][ny] < 0.5) {
	  continue; // skip
	}
	else {
	  writeNeuralFileLine(os, nx, ny, mtx, refMatrix, globalFeatures, 
			      patchSize, fileMode, allMode, bothSideMode);
	  // chose one random neigbhor and three totally random points
	  nx += randomSign() * ((rnd.getRand() % 2) + 1);
	  ny += randomSign() * ((rnd.getRand() % 2) + 1);
	  if ((nx >= patchSizeBorder) && (ny >= patchSizeBorder) && (nx + patchSizeBorder < static_cast<int>(mtx.size())) 
	      && (ny + patchSizeBorder < static_cast<int>(mtx.size()))) {
	    writeNeuralFileLine(os, nx, ny, mtx, refMatrix, globalFeatures, 
				patchSize, fileMode, allMode, bothSideMode);
	  }
	  nx = (rnd.getRand() % (mtx.size()-(2*patchSizeBorder))) + patchSizeBorder;
	  ny = (rnd.getRand() % (mtx.size()-(2*patchSizeBorder))) + patchSizeBorder;
	  writeNeuralFileLine(os, nx, ny, mtx, refMatrix, globalFeatures, 
			      patchSize, fileMode, allMode, bothSideMode);
	  nx = (rnd.getRand() % (mtx.size()-(2*patchSizeBorder))) + patchSizeBorder;
	  ny = (rnd.getRand() % (mtx.size()-(2*patchSizeBorder))) + patchSizeBorder;
	  writeNeuralFileLine(os, nx, ny, mtx, refMatrix, globalFeatures, 
			      patchSize, fileMode, allMode, bothSideMode);
	  nx = (rnd.getRand() % (mtx.size()-(2*patchSizeBorder))) + patchSizeBorder;
	  ny = (rnd.getRand() % (mtx.size()-(2*patchSizeBorder))) + patchSizeBorder;
	  writeNeuralFileLine(os, nx, ny, mtx, refMatrix, globalFeatures, 
			      patchSize, fileMode, allMode, bothSideMode);
	}
      }
      else {
	// cout << "writing writeNeuralFile!" << nx << " " << ny << endl;
	writeNeuralFileElement(os, mtx, globalFeatures,
			       patchSize, nx, ny, fileMode, allMode);
	os << " " << refMatrix[nx][ny];
	os << endl;
	if (bothSideMode) {
	  writeNeuralFileElement(os, mtx, globalFeatures,
				 patchSize, ny, nx, fileMode, allMode);
	  os << " " << refMatrix[ny][nx];
	  os << endl;
	}
      }
    }
  }
  else if (neuralMaxLines < 0) {
    for (int i = 0; i < static_cast<int>(points.size()); ++i) {
      unsigned int nx = points[i].first;
      unsigned int ny = points[i].second;
      // cout << "writing writeNeuralFile!" << nx << " " << ny << endl;
      writeNeuralFileElement(os, mtx, globalFeatures,
			     patchSize, nx, ny, fileMode, allMode);
      os << " " << refMatrix[nx][ny];
      os << endl;
      if (bothSideMode) {
	writeNeuralFileElement(os, mtx, globalFeatures,
			       patchSize, ny, nx, fileMode, allMode);
	os << " " << refMatrix[ny][nx];
	os << endl;
      }
    }
  }
  else if (distBias > 0.0) { // only write random subset, with bias towards true contacts and close neighbors
    int countWritten = 0;
    random_shuffle(points.begin(), points.end(), rnd);    
    Vec<unsigned int> goodIndices;
    // first write all the real points
    for (int i = 0; i < count; ++i) {
      unsigned int nx = points[i].first;
      unsigned int ny = points[i].second;
      // determine distance to closes real point
      if (refMatrix[nx][ny] > 0.0) {
	goodIndices.push_back(i);
      }
    }
    for (unsigned int i = 0; i < points.size(); ++i) {
      int nx = points[i].first;
      int ny = points[i].second;
      // determine distance to closes real contact
      double distBest = 1e10;
      double dist;
      int nx2, ny2;
      for (unsigned int j = 0; j < goodIndices.size(); ++j) {
	nx2 = points[goodIndices[j]].first;
	ny2 = points[goodIndices[j]].second;	
	dist = static_cast<double>((nx-nx2)*(nx-nx2) + (ny-ny2)*(ny-ny2)); // use quadratic distance
	if (dist < distBest) {
	  distBest = dist;
	  if (dist == 0.0) {
	    break;
	  }
	}
      }
      distBest = sqrt(distBest);
      double prob = exp(-distBest/distBias);
      if (rnd.getRandf() < prob) {
	writeNeuralFileElement(os, mtx, globalFeatures,
			       patchSize, points[i].first, points[i].second, fileMode, allMode);
	os << " " << refMatrix[points[i].first][points[i].second];
	os << endl;
	++countWritten;
	if (bothSideMode) {
	  writeNeuralFileElement(os, mtx, globalFeatures,
				 patchSize, points[i].second, points[i].first, fileMode, allMode);
	  os << " " << refMatrix[points[i].second][points[i].first];
	  os << endl;
	  ++countWritten;
	}
	if (countWritten >= neuralMaxLines) {
	  break;
	}
      }
    }
  }
  else { // only write random subset
    if (neuralMaxLines > count) {
      neuralMaxLines = count;
    }
    int countWritten = 0;
    random_shuffle(points.begin(), points.end(), rnd);    
    // first write all the real points
    for (int i = 0; i < count; ++i) {
      // cout << "hi6!" << endl;
      unsigned int nx = points[i].first;
      unsigned int ny = points[i].second;
      if (refMatrix[nx][ny] > 0.0) {
	writeNeuralFileElement(os, mtx, globalFeatures,
          patchSize, points[i].first, points[i].second, fileMode, allMode);
	os << " " << refMatrix[points[i].first][points[i].second];
	os << endl;
	++countWritten;
	if (bothSideMode) {
	  writeNeuralFileElement(os, mtx, globalFeatures,
				 patchSize, points[i].second, points[i].first, fileMode, allMode);
	  os << " " << refMatrix[points[i].second][points[i].first];
	  os << endl;
	  ++countWritten;
	}
      }
    }
    for (int i = 0; i < (neuralMaxLines-countWritten); ++i) {
      writeNeuralFileElement(os, mtx, globalFeatures,
        patchSize, points[i].first, points[i].second, fileMode, allMode);
      os << " " << refMatrix[points[i].first][points[i].second];
      os << endl;
      if (bothSideMode) {
	writeNeuralFileElement(os, mtx, globalFeatures,
			       patchSize, points[i].second, points[i].first, fileMode, allMode);
	os << " " << refMatrix[points[i].second][points[i].first];
	os << endl;
      }
    }
  }
}

Vec<Vec<double> >
readListMatrix(istream& is, double defaultValue, bool mirrorMode)
{
  unsigned int x, y, numEntries;
  is >> x >> y >> numEntries; // dimensions
  double val;
  Vec<Vec<double> > result(x, Vec<double>(y, defaultValue));
  for (unsigned int i = 0; i < numEntries; ++i) {
    is >> x >> y >> val;
    ERROR_IF(is.eof(), "Unexpected end of file!");
    ERROR_IF((x < 1) || (y < 1),
	     "Too small indices found in matrix file!");
    --x;
    --y;
    ERROR_IF((x >= result.size()), "Too large first index found in matrix file!");
    ERROR_IF((y >= result[x].size()), "Too large second index found in matrix file!");
    // cout << "Setting " << x << " " << y << " to " << val << endl;
    result[x][y] = val;
    if (mirrorMode) {
      result[y][x] = val;
    }
  }
  return result;
}
	       
/** computes global features used by neural networks */
Vec<double> 
computeGlobalFeatures(const SequenceAlignment& ali)
{
  Vec<double> result;
  if (ali.size() > 0) {
    result.push_back(1.0/static_cast<double>(ali.size()));
  }
  else {
    result.push_back(0.0);
  }
  // result.push_back(static_cast<double>(ali.size()) / 1000.0);
  result.push_back(1.0/static_cast<double>(ali.getLength()));
  Vec<double> seqDist = flatten(SequenceAlignment::getSeqDistances(ali, NUC_IGNORE));
//   double maxi = seqDist[maxElement(seqDist)];
//   double mini = seqDist[minElement(seqDist)];
//   result.push_back(maxi);
//   result.push_back(mini);
  result.push_back(vecMean(seqDist)); // average sequence similarity
  SequenceAlignment::sequence_size_type numChars = 0;
  SequenceAlignment::sequence_size_type numGaps = 0;
  ali.countCharacters(numChars, numGaps);
  double gapFrac = static_cast<double>(numGaps) 
    / (ali.size() * ali.getLength()); // devide by total number of chars
  result.push_back(gapFrac);
  return result;
}


/** adds contant to all matrix elements */
void
addConstant(Vec<Vec<double> >& m,
	    double d)
{
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      m[i][j] += d;
    }    
  }  
}

/** adds contant to all matrix elements */
void
mulConstant(Vec<Vec<double> >& m,
	    double d)
{
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      m[i][j] *= d;
    }    
  }  
}

void
applyCutoff(double& x,
	    double upper, double lower)
{
  if (upper > x) {
    x = upper;
  }
  else if (lower < x) {
    x = lower;
  }
}


void
applyCutoff(Vec<Vec<double> >& m,
	    double upper, double lower)
{
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      applyCutoff(m[i][j],upper, lower);
    }
  }
}

double
findMinVal(const Vec<Vec<double> >& m) 
{
  double x = m[0][0];
  for (unsigned int i = 0; i < m.size(); ++i ) {
    for (unsigned int j = 0; j < m[i].size(); ++j ) {
      if (m[i][j] < x) {
	x = m[i][j];
      }
    }
  }
  return x;
}

double
findMaxVal(const Vec<Vec<double> >& m) 
{
  double x = m[0][0];
  for (unsigned int i = 0; i < m.size(); ++i ) {
    for (unsigned int j = 0; j < m[i].size(); ++j ) {
      if (m[i][j] > x) {
	x = m[i][j];
      }
    }
  }
  return x;
}

/** 
    returns renormalized matrix
*/
Vec<Vec<double> >
renormMatrix(const Vec<Vec<double> >& matrix,
	     const Vec<int>& renormMatrixMode,
	     const CompensationScorer& scorer,
	     int borderLim,
	     int diagLim,
	     unsigned int aliSize, // number of sequences in alignment
	     int verboseLevel)
{
  double minVal = 0.0;
  double maxVal = 1.0;
  double scaling = 10.0; // maybe change value in future!
  Vec<Vec<double> > compMatrix(matrix); // make copy
  if ((verboseLevel > 0) && (renormMatrixMode.size() == 0)) {
    cout << "No matrix renormalization specified!" << endl;
  }
  for (unsigned int i = 0; i < renormMatrixMode.size(); ++i) {
    if (verboseLevel > 0) {
      cout << "Renormalizing matrix (mode " << renormMatrixMode[i] << ")" << endl;
    }
    switch (renormMatrixMode[i]) {
    case 0:
      // do nothing
      break;
    case 1:
      // z-score with respect to rows and columns
      compMatrix = renormalizeMatrix(compMatrix);
      break;
    case 2:
      ERROR("Renormalization mode 2 defunct!");
      // compMatrix = filterMatrix(compMatrix, fm0, fmv, fmh, fmd, fma);
      break;
    case 3:
      ERROR("Renormalization mode 3 defunct!");
      // compMatrix = filterMatrixWithStems(compMatrix, minStemLength, scoreLimit );
      break;
    case 4:
      compMatrix = localSkewness(compMatrix);
      break;
    case 5:
      // transform to z-scores
      // compMatrix = renormalizeMatrix(compMatrix);
      // rescale to range (0..1) using e(x)/(1+e(x))
      applyLogisticFunction(compMatrix, 1.0);
      break;
    case 6: 
      compMatrix = renormalizeMatrix(compMatrix, borderLim, diagLim);
      // change scaling, such that z-score 0 becomes 0.5
      // z-score -5 becomes 0, z-score +5 becomes 1.0
      // add constant ("5.0") to elements:
      addConstant(compMatrix, 0.5 * scaling); 
      mulConstant(compMatrix, 1.0/scaling); 
      // applyCutoff(compMatrix, 1.0, 0.0); // apply upper and lower cutoff
      break;
    case 7:
      minVal = scorer.pairwiseEntropyMin(aliSize);
      maxVal = scorer.pairwiseEntropyMax(aliSize);
      scaling = maxVal - minVal; 
      if (scaling <= 0.0) {
	cout << "Warning: scaling factor was " << scaling << " ... settting to one." << endl;
	scaling = 1.0;
      }
      addConstant(compMatrix, - minVal); 
      mulConstant(compMatrix, 1.0/scaling);       

      break;
    case 8:
      minVal = findMinVal(compMatrix);
      maxVal = findMaxVal(compMatrix);
      scaling = maxVal - minVal; 
      if (scaling <= 0.0) {
	cout << "Warning: scaling factor was " << scaling << " ... settting to one." << endl;
	scaling = 1.0;
      }
      addConstant(compMatrix, - minVal); 
      mulConstant(compMatrix, 1.0/scaling);       

      break;
    default:
    ERROR("Unknown matrix mode!");
    }
  }
  return compMatrix;
}

/** generates compensatory matrix acording to scorer,
    algorithm and renormMatrixMode
*/
Vec<Vec<double> >
generateCompensatoryMatrix(const SequenceAlignment& ali,
			   const CompensationScorer& scorer,
			   int algorithm,
			   const Vec<int>& renormMatrixMode,
			   int borderLim,
			   int diagLim,
			   int verboseLevel)
{
  Vec<Vec<double> > compMatrix;
  if (algorithm < 0) {
    compMatrix = computeCompensatoryScores(ali.getSequences(), 
					   scorer.getAlphabet());
  }
  else {
    ERROR_IF(scorer.algorithm != algorithm,
	     "Internal error in line 1426!");
    compMatrix = computeCompensatoryScores(ali, scorer);
  }

  // cout << "Matrix after computeCompensatoryScores :" << endl << compMatrix << endl;
  compMatrix = renormMatrix(compMatrix, renormMatrixMode, scorer, borderLim,
			    diagLim, ali.size(), verboseLevel);
  return compMatrix;
}


double
rescaleValue(double x, const Vec<Vec<double> >& rescaleInfo)
{
  for (unsigned int i = 0; i < rescaleInfo.size(); ++i) {
    if (x <= rescaleInfo[i][0]) {
      return rescaleInfo[i][1];
    }
  }
  cout << x << endl << rescaleInfo << endl;
  ERROR("Incomplete rescale data!");
  return 0.0; // dummy
}

void
rescaleMatrix(Vec<Vec<double> >& matrix,
	      const Vec<Vec<double> >& rescaleInfo)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    for (unsigned int j = 0; j < matrix[i].size(); ++j) {
      matrix[i][j] = rescaleValue(matrix[i][j], rescaleInfo);
    }
  }
}

Vec<Vec<double> >
readRescaleData(istream& is)
{
  unsigned int numEntries;
  Vec<double> p(2);
  Vec<Vec<double> > result;
  is >> numEntries;
  for (unsigned int i = 0; i < numEntries; ++i) {
    is >> p[0] >> p[1];
    result.push_back(p);
    ERROR_IF(!is, "Error reading rescale info file!");
  }
  return result;
}

/** computes simple weighting scheme, downregulating groups of highly homologous sequences */
Vec<double>
computeSequenceWeights(const SequenceAlignment& ali)
{
  Vec<Vec<double> > distMatrix = SequenceAlignment::getSeqDistances(ali, NUC_IGNORE);
  Vec<double> result(distMatrix.size(), 0.0);
  for (unsigned int i = 0; i < result.size(); ++i) {
    for (unsigned int j = 0; j < result.size(); ++j) {
      result[i] += distMatrix[i][j]; // similarity with itself always adds "1"
    }
    if (result[i] > 0.0) {
      result[i] = 1/result[i];
    }
    else {
      result[i] = 1.0;
    }
  }
  normalizeEuclidian(result);
  Vec<double> tVec(result.size(), 1.0);
  double goalNorm = euclidianNorm(tVec);
  scalarMul(result, goalNorm); // should have norm result.size()
  return result;
}		       


/** removes nucleotide, adjusts stems accordingly */
/* moved to stemhelp.h
void
deletePositionInStems(Vec<Stem>& stems, 
		      string sequence, unsigned int delPos)
{
  PRECOND(sequence.size() > 1);
  Vec<int> connections = convertStemsToConnections(stems, sequence.size());
  for (unsigned int j = 0; j < connections.size(); ++j) {
    if (connections[j] == static_cast<int>(delPos)) {
      connections[j] = -1; // delete that contact
    }
    else if (connections[j] > static_cast<int>(delPos)) {
      --connections[j]; // reduce index by one, because shift due to delete
    }
  }
  connections.erase(connections.begin()+delPos); // remove that entry
  unsigned int minStemLength = 1;
  sequence.erase(delPos, 1);
  stems = convertConnectionsToStems(connections,
				    minStemLength, sequence);
}
*/

/** removes gap at position col of sequence with number collapseId 
    in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignment(SequenceAlignment& ali, 
		  unsigned int collapseId, 
		  unsigned int col,
		  Vec<Vec<double> >& consProbMatrix, 
		  Vec<Stem>& referenceStems,
		  Vec<Stem>& referenceStems2)
{
  if ((collapseId >= ali.size()) || (col >= ali.getLength())
      || (ali.getLength() < 2)) {
    ERROR("Internal error in line 2584!");
    return;
  }
  deletePositionInStems(referenceStems, ali.getSequence(collapseId), col); // order important!
  deletePositionInStems(referenceStems2, ali.getSequence(collapseId), col); // order important!
  ali.deleteColumn(col);
  deleteMatrixRow(consProbMatrix, col);
  deleteMatrixCol(consProbMatrix, col);
  ERROR_IF(consProbMatrix.size() != ali.getLength(),
	   "Internal error: Probability matrix and alignment have different lengths!");
}

/** removes all gaps of sequence with number collapseId in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignment(SequenceAlignment& ali, 
		  unsigned int collapseId, 
		  Vec<Vec<double> >& consProbMatrix, 
		  Vec<Stem>& referenceStems,
		  Vec<Stem>& referenceStems2,
		  char gapChar)
{
  if (collapseId >= ali.size()) {
    return;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    if (ali.getSequence(collapseId)[i] == gapChar) {
      collapseAlignment(ali, collapseId, 
	static_cast<unsigned int>(i), consProbMatrix, referenceStems,
			referenceStems2);
    }
  }
}

/** removes all gaps of all sequences
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignmentAllGaps(SequenceAlignment& ali, 
			 unsigned int collapseId, 
			 Vec<Vec<double> >& consProbMatrix, 
			 Vec<Stem>& referenceStems,
			 Vec<Stem>& referenceStems2,
			 char gapChar)
{
  if (collapseId >= ali.size()) {
    collapseId = 0;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    for (unsigned int j = 0; j < ali.size(); ++j) {
      if (ali.getSequence(j)[i] == gapChar) {
	collapseAlignment(ali, collapseId, 
			  static_cast<unsigned int>(i), consProbMatrix, referenceStems,
			  referenceStems2);
	break;
      }
    }
  }
}

/** removes all gaps of all sequences
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignmentTooManyGaps(SequenceAlignment& ali, 
			     unsigned int collapseId, 
			     Vec<Vec<double> >& consProbMatrix, 
			     Vec<Stem>& referenceStems,
			     Vec<Stem>& referenceStems2,
			     unsigned int charNumMin,
			     char gapChar)
{
  if (collapseId >= ali.size()) {
    collapseId = 0;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    string col = ali.getColumn(static_cast<unsigned int>(i));
    int counter = 0;
    for (unsigned int j = 0; j < col.size(); ++j) {
      if (col[j] == gapChar) {
	++counter;
      }
    }
    counter = ali.size() - counter; // number of characters = number of sequences minus number of gaps
    if (counter < static_cast<int>(charNumMin)) {
      ERROR_IF(ali.getLength() < 5,
	       "Alignment ridicolously small!");
      collapseAlignment(ali, collapseId, 
			static_cast<unsigned int>(i), consProbMatrix, referenceStems,
			referenceStems2);
    }
  }
}

/** removes all gaps of all sequences
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignmentTooManyGaps(SequenceAlignment& ali, 
			     SequenceAlignment& aliRef,
			     unsigned int collapseId, 
			     Vec<Vec<double> >& consProbMatrix, 
			     Vec<Stem>& referenceStems,
			     Vec<Stem>& referenceStems2,
			     unsigned int charNumMin,
			     char gapChar)
{
  if (aliRef.size() == 0) {
    collapseAlignmentTooManyGaps(ali, collapseId, consProbMatrix, referenceStems, referenceStems2, charNumMin, gapChar);
    return;
  }
  ERROR_IF(ali.getLength() != aliRef.getLength(), 
	   "Alignment and refernece alignments have to have same sequence lengths!");
  if (collapseId >= ali.size()) {
    collapseId = 0;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    string col = aliRef.getColumn(static_cast<unsigned int>(i));
    int counter = 0;
    for (unsigned int j = 0; j < col.size(); ++j) {
      if (col[j] == gapChar) {
	++counter;
      }
    }
    counter = aliRef.size() - counter; // number of characters = number of sequences minus number of gaps
    if (counter < static_cast<int>(charNumMin)) {
      ERROR_IF(ali.getLength() < 5,
	       "Alignment ridicolously small!");
      collapseAlignment(ali, collapseId, 
			static_cast<unsigned int>(i), consProbMatrix, referenceStems,
			referenceStems2);
    }
  }
}


/** removes gap of first sequences, remove columns with not enough gaps 
    in reference or main alignment.
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignmentGapOrTooManyGaps(SequenceAlignment& ali, 
				  SequenceAlignment& aliRef,
				  unsigned int collapseId, 
				  Vec<Vec<double> >& consProbMatrix, 
				  Vec<Stem>& referenceStems,
				  Vec<Stem>& referenceStems2,
				  unsigned int charNumMin,
				  char gapChar)
{
  if (collapseId >= ali.size()) {
    collapseId = 0;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    string col;
    if (aliRef.size() > 0) {
      col = aliRef.getColumn(static_cast<unsigned int>(i));
    }
    else {
      col = ali.getColumn(static_cast<unsigned int>(i));
    }
    int counter = 0;
    for (unsigned int j = 0; j < col.size(); ++j) {
      if (col[j] == gapChar) {
	++counter;
      }
    }
    if (aliRef.size() > 0) {
      counter = aliRef.size() - counter; // number of characters = number of sequences minus number of gaps
    }
    else {
      counter = ali.size() - counter; // number of characters = number of sequences minus number of gaps
    }
    if ((ali.getSequence(collapseId)[i] == gapChar) 
	|| (counter < static_cast<int>(charNumMin))) {
      ERROR_IF(ali.getLength() < 5,
	       "Alignment ridicolously small!");
      collapseAlignment(ali, collapseId, 
			static_cast<unsigned int>(i), consProbMatrix, referenceStems,
			referenceStems2);
    }
  }
}



/** Returns frequency of matching pairs.
    allowed pairs: do not distinguish between for example GC and CG ! */
Vec<double> 
computeAliSliceFrequencies(const Vec<char>& s1, 
			   const Vec<char>& s2,
			   unsigned int dim,
			   const Vec<string>& allowedPairs,
			   char gapChar)
			   
{
  PRECOND((s1.size() > 0) && (s1.size() == s2.size()));
  Vec<double> result(dim, 0.0);
  unsigned int gapCount = 0;
  for (unsigned int i = 0; i < s1.size(); ++i) {
    if ((s1[i] == gapChar) || (s2[i] == gapChar)) {
      ++gapCount;
      continue;
    }
    unsigned int pairId = allowedPairs.size();
    for (unsigned int j = 0; j < allowedPairs.size(); ++j) {
      if (((s1[i] == allowedPairs[j][0]) && (s2[i] == allowedPairs[j][1]))
       || ((s2[i] == allowedPairs[j][0]) && (s1[i] == allowedPairs[j][1]))) {
	pairId = j; 
	break;
      }
    }
    // ASSERT(pairId < result.size());
    if (pairId < result.size()) {
      result[pairId] += 1.0;
    }
  }
  if (gapCount < s1.size()) {
    double divi = s1.size() - gapCount;
    for (unsigned int i = 0; i < result.size(); ++i) {
      result[i] /= divi;
    }
  }
  return result;
}

/** Returns frequency of matching pairs.
    allowed pairs: do not distinguish between for example GC and CG ! */
Vec<double> 
computeAliSliceFrequencies2(const Vec<char>& s1, 
			    const Vec<char>& s2,
			    unsigned int dim,
			    const Vec<string>& allowedPairs, // precise pairs
			    const Vec<string>& allowedPairs2, // sloppy pairs (do not distinguish GU and UG
			    char gapChar,
			    int ignoreCount)
{
  PRECOND((s1.size() > 0) && (s1.size() == s2.size()));
  Vec<double> result(dim, 0.0);
  unsigned int gapCount = 0;
  for (unsigned int i = 0; i < s1.size(); ++i) {
    if ((s1[i] == gapChar) || (s2[i] == gapChar)) {
      ++gapCount;
      continue;
    }
    unsigned int pairId = dim - 1;
    bool found = false;
    for (unsigned int j = 0; j < allowedPairs.size(); ++j) {
      if ((s1[i] == allowedPairs[j][0]) && (s2[i] == allowedPairs[j][1])) {
	pairId = j; 
	found = true;
	break;
      }
    }
    if (!found) {
      for (unsigned int j = 0; j < allowedPairs2.size(); ++j) {
	if (((s1[i] == allowedPairs2[j][0]) && (s2[i] == allowedPairs2[j][1]))
	    || ((s2[i] == allowedPairs2[j][0]) && (s1[i] == allowedPairs2[j][1]))) {
	  pairId = j + allowedPairs.size(); 
	  found = true;
	  break;
	}
      }
    }
    // ASSERT(pairId < result.size());
    if (pairId < result.size()) {
      result[pairId] += 1.0;
    }
  }
  if (gapCount < s1.size()) {
    double divi = s1.size() - gapCount;
    for (unsigned int i = 0; i < result.size(); ++i) {
      result[i] /= divi;
    }
  }
  // sort matching part, leave out last element which is non matching part
  sort(result.begin(), result.begin()+(result.size()-ignoreCount));
  if (ignoreCount > 1) {
    // always ignore last column
    sort(result.begin()+(result.size()-ignoreCount), result.end()-1);
  }
  //   reverse(result.begin(), result.begin()+(result.size()-ignoreCount)); // highest scoring element first, ignore mismatches, GU and UG
  // cout << "Sorted result: " << result << endl;
  return result;
}


/** Returns frequency of matching pairs.
    allowed pairs: do not distinguish between for example GC and CG !
    like computeAliSliceFrequencies2, but also storing 1/(number of non-gap pairs) 
*/
Vec<double> 
computeAliSliceFrequencies3(const string& s1, 
			    const string& s2,
			    unsigned int dim,
			    const Vec<string>& allowedPairs, // precise pairs
			    const Vec<string>& allowedPairs2, // sloppy pairs (do not distinguish GU and UG
			    char gapChar,
			    int ignoreCount,
			    const Vec<double>& sequenceWeights,
			    double pseudoTop, /* for all matches (not GU is that is separate */
			    double pseudoTop2, /* for all GU matches */
			    double pseudoTop3, /* for all mismatches */
			    double pseudoBottom )
{
  PRECOND((s1.size() > 0) && (s1.size() == s2.size()));
  PRECOND(sequenceWeights.size() == s1.size());
  Vec<double> result(dim, 0.0);
  unsigned int gapCount = 0;
  double weightSum = pseudoBottom;
  unsigned int pc = 0;
  for (unsigned int i = 0; i < allowedPairs.size(); ++i, ++pc) {
    result[pc] = pseudoTop;
  }
  for (unsigned int i = 0; i < allowedPairs2.size(); ++i, ++pc) {
    result[pc] = pseudoTop2;
  }
  for (unsigned int i = pc; i < result.size(); ++i, ++pc) {
    result[pc] = pseudoTop3; // pseudocounts for mismatches
  }
  for (unsigned int i = 0; i < s1.size(); ++i) {
    if ((s1[i] == gapChar) || (s2[i] == gapChar)) {
      ++gapCount;
      continue;
    }
    unsigned int pairId = allowedPairs.size()+allowedPairs2.size();
    bool found = false;
    for (unsigned int j = 0; j < allowedPairs.size(); ++j) {
      if ((s1[i] == allowedPairs[j][0]) && (s2[i] == allowedPairs[j][1])) {
	pairId = j; 
	found = true;
	break;
      }
    }
    if (!found) {
      for (unsigned int j = 0; j < allowedPairs2.size(); ++j) {
	if (((s1[i] == allowedPairs2[j][0]) && (s2[i] == allowedPairs2[j][1]))
	    || ((s2[i] == allowedPairs2[j][0]) && (s1[i] == allowedPairs2[j][1]))) {
	  pairId = j + allowedPairs.size(); 
	  found = true;
	  break;
	}
      }
    }
    // ASSERT(pairId < result.size());
    if (pairId < result.size()) {
      result[pairId] += sequenceWeights[i];
      // result[pairId] += 1.0; // sequenceWeights[i];
      weightSum += sequenceWeights[i];
    }
  }
  if (weightSum <= 0.0) {
    weightSum = 1.0;
  }
  if (gapCount < s1.size()) {
    double divi = s1.size() - gapCount; // number of non-gap pairs
    for (unsigned int i = 0; i + 1< result.size(); ++i) {
      result[i] /= weightSum; // normalize using sequence weights
      // result[i] /= divi; // normalize using sequence weights
    }
    /*
    double testSum = 0.0;
    for (unsigned int i = 0; i + 1< result.size(); ++i) {
      testSum += result[i];
    }
    if (fabs(testSum-1.0) > 0.05) {
      cout << "Warning: Strange normalization: " << testSum
	   << " " << result << endl;
    }
    */
    result[result.size()-1] = 1.0 / divi; // last column: 1/number of non-gap pairs
  }
  // else : nothing to do, whole column just gaps

  // sort matching part, leave out last 4 elements which is UG, GA, non matching part, non-gap pairs
  sort(result.begin(), result.begin()+(result.size()-ignoreCount));
  if (ignoreCount > 2) {
    sort(result.begin()+(result.size()-ignoreCount), result.end()-2);
  }
  //   reverse(result.begin(), result.begin()+(result.size()-ignoreCount)); // highest scoring element first, ignore mismatches, GU and UG
  // cout << "Sorted result: " << result << endl;
  return result;
}

Vec<Vec<Vec<double> > >
computeAliMatrices(const SequenceAlignment& ali, 
		   const CompensationScorer& scorer,
		   char gapChar)
{
  unsigned int nn = ali.getLength();
  Vec<string> allowedPairs(6);
  allowedPairs[0] = "AU"; // order matters
  allowedPairs[1] = "CG";
  allowedPairs[2] = "GC";
  allowedPairs[3] = "UA";
  allowedPairs[4] = "GU";
  allowedPairs[5] = "UG";
  Vec<string> allowedPairs2; // (2); // = scorer.getUniqueAllowedPairs();
  int ignoreCount = 2; // do sort GU, UG, but not mismatches , obsolete: number of non-gap pairs
  // allowedPairs2[0] = "GU"; // order of this one does not matter
  unsigned int dim = allowedPairs.size() + allowedPairs2.size() + 1 + 1; // alphabet.size() * alphabet.size() - allowedPairs.size() + 1;
  Vec<Vec<Vec<double> > > result(dim, Vec<Vec<double> >(nn, Vec<double>(nn, 0.0)));
  Vec<double> sliceResult(dim);
  double pseudoTop1 = 0.0;
  double pseudoTop2 = 0.0;
  double pseudoTop3 = 0.0;
  double pseudoBottom = 0.0;
  Vec<double> weights(ali.size(), 1.0);
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      sliceResult = computeAliSliceFrequencies3(ali.getColumn(i), 
						ali.getColumn(j),
						dim, allowedPairs, 
						allowedPairs2, gapChar, ignoreCount,
						weights,
						pseudoTop1, pseudoTop2, pseudoTop3, pseudoBottom);
      for (unsigned int k = 0; k < dim; ++k) {
	result[k][i][j] = sliceResult[k];
      }
      sliceResult = computeAliSliceFrequencies3(ali.getColumn(j), 
						ali.getColumn(i),
			dim, allowedPairs, allowedPairs2, gapChar, 
			ignoreCount, weights,
			pseudoTop1, pseudoTop2, pseudoTop3, pseudoBottom);
      for (unsigned int k = 0; k < dim; ++k) {
	result[k][j][i] = sliceResult[k];
      }
    }
  }
  return result;
}

/** like computeAliMatrices, but use real Bayesian pseudocounts */
Vec<Vec<Vec<double> > >
computeAliMatrices2(const SequenceAlignment& ali, 
		    const CompensationScorer& scorer,
		   char gapChar)
{
  unsigned int nn = ali.getLength();
  Vec<string> allowedPairs(6);
  allowedPairs[0] = "AU"; // order matters
  allowedPairs[1] = "CG";
  allowedPairs[2] = "GC";
  allowedPairs[3] = "UA";
  allowedPairs[4] = "GU";
  allowedPairs[5] = "UG";
  Vec<string> allowedPairs2; // (2); // = scorer.getUniqueAllowedPairs();
  int ignoreCount = 2; // do sort GU, UG, but not mismatches , obsolete: number of non-gap pairs
  // allowedPairs2[0] = "GU"; // order of this one does not matter
  unsigned int dim = allowedPairs.size() + allowedPairs2.size() + 1 + 1; // alphabet.size() * alphabet.size() - allowedPairs.size() + 1;
  Vec<Vec<Vec<double> > > result(dim, Vec<Vec<double> >(nn, Vec<double>(nn, 0.0)));
  Vec<double> sliceResult(dim);
  double pseudoTop1 = 1.0;
  double pseudoTop2 = 1.0;
  double pseudoTop3 = (scorer.getAlphabet().size() * scorer.getAlphabet().size() 
		       - allowedPairs.size()) - allowedPairs2.size() ;
  double pseudoBottom = scorer.getAlphabet().size() * scorer.getAlphabet().size(); // number of cases
  Vec<double> weights(ali.size(), 1.0);
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < i; ++j) {
      sliceResult = computeAliSliceFrequencies3(ali.getColumn(i), 
						ali.getColumn(j),
						dim, allowedPairs, 
				allowedPairs2, gapChar, ignoreCount,
				weights,
				pseudoTop1, pseudoTop2, pseudoTop3, pseudoBottom);
      for (unsigned int k = 0; k < dim; ++k) {
	result[k][i][j] = sliceResult[k];
      }
      sliceResult = computeAliSliceFrequencies3(ali.getColumn(j), 
						ali.getColumn(i),
			dim, allowedPairs, allowedPairs2, gapChar, 
			ignoreCount, weights,
			pseudoTop1, pseudoTop2, pseudoTop3, pseudoBottom);
      for (unsigned int k = 0; k < dim; ++k) {
	result[k][j][i] = sliceResult[k];
      }
    }
  }
  return result;
}

double
computeKnnPrediction(int nx, int ny,
		     const Vec<Vec<Vec<double> > >& compMatrices,
		     const Vec<double>& globalFeatures,
		     const ClassifierBase& knnNet,
		     int neuralPatchSize,
		     int neuralFileMode,
		     int verboseLevel)
{
  Vec<double> featureVector;
  switch (neuralFileMode) {
  case 9: featureVector = generateNeuralFileElement9b(compMatrices,
						      globalFeatures, neuralPatchSize, nx, ny);
    break;
  case 10: featureVector = generateNeuralFileElement10b(compMatrices,
							globalFeatures, neuralPatchSize, nx, ny);
    break;
  default: ERROR("Undefined neural file mode!");
  }
  double result = knnNet.predictClassProb(featureVector)[0];
  if (verboseLevel > 1) {
    cout << result << " is result of " << featureVector << endl;
  }
  POSTCOND((result >= 0.0) && (result <= 1.0));
  return result;
}

/** compute n'th level of knn hierarchy */
Vec<double>
computeKnnPredictionLevel(int nx, int ny,
			  const Vec<Vec<Vec<double> > >& compMatrices,
			  const Vec<double>& globalFeatures,
			  const KnnNet2& knnNet,
			  int neuralPatchSize,
			  int neuralFileMode,
			  unsigned int knnLevel,
			  int verboseLevel)
{
  Vec<double> featureVector;
  switch (neuralFileMode) {
  case 9: featureVector = generateNeuralFileElement9b(compMatrices,
						      globalFeatures, neuralPatchSize, nx, ny);
    break;
  case 10: featureVector = generateNeuralFileElement10b(compMatrices,
							globalFeatures, neuralPatchSize, nx, ny);
    break;
  default: ERROR("Undefined neural file mode!");
  }
  Vec<double> result = knnNet.predictClassProbLevel(featureVector, knnLevel);
  if (verboseLevel > 1) {
    cout << result << " is result at level " << knnLevel << " of " << featureVector << endl;
  }
  return result;
}


/** use known answer, possible write to training file in knnNet */
double
computeKnnTrainPrediction(ostream& os,
			  int nx, int ny,
			  const Vec<Vec<Vec<double> > >& compMatrices,
			  const Vec<double>& globalFeatures,
			  const ClassifierBase& knnNet,
			  int neuralPatchSize,
			  int neuralFileMode,
			  unsigned int trueClass,
			  int verboseLevel)
{
  Vec<double> featureVector;
  switch (neuralFileMode) {
  case 9: featureVector = generateNeuralFileElement9b(compMatrices,
						      globalFeatures, neuralPatchSize, nx, ny);
    break;
  case 10: featureVector = generateNeuralFileElement10b(compMatrices,
							globalFeatures, neuralPatchSize, nx, ny);
    break;
  default: ERROR("Undefined neural file mode!");
  }
  os << nx +1 << " " << ny + 1 << " ";
  double result = knnNet.predictClassProbTrain(featureVector, trueClass, os)[0];
  if (verboseLevel > 1) {
    cout << result << " is result of " << featureVector << endl;
  }
  POSTCOND((result >= 0.0) && (result <= 1.0));
  return result;
}


double
computeKnnPrediction(int nx, int ny,
		     const Vec<Vec<Vec<double> > >& compMatrices,
		     const Vec<double>& globalFeatures,
		     const ClassifierBase& knnNode,
		     const Vec<unsigned int>& knnMask,
		     int neuralPatchSize,
		     int neuralFileMode,
		     int neuralAllMode,
		     int verboseLevel)
{
  Vec<double> featureVector;
  switch (neuralFileMode) {
  case 9: featureVector = generateNeuralFileElement9b(compMatrices,
						      globalFeatures, neuralPatchSize, nx, ny);
    break;
  case 10: featureVector = generateNeuralFileElement10b(compMatrices,
				globalFeatures, neuralPatchSize, nx, ny);
    break;
  default: ERROR("Undefined neural file mode!");
  }
  // bugfix: changes "1" to "0"
  double result = knnNode.predictClassProb(
			    getSubset(featureVector, knnMask))[0];
//   double result = knnNode.predictClassProb(
// 			    getSubset(featureVector, knnMask))[1];
  if (verboseLevel > 1) {
    cout << result << " is result of " << getSubset(featureVector, knnMask)
	 << endl;
  }
  POSTCOND((result >= 0.0) && (result <= 1.0));
  return result;
}

Vec<Vec<double> >
computeKnnPrediction(const Vec<Vec<Vec<double> > >& compMatrices,
		     const Vec<double>& globalFeatures,
		     const ClassifierBase& knnNode,
		     const Vec<unsigned int>& knnMask,
		     int neuralPatchSize,
		     int neuralFileMode,
		     int neuralAllMode,
		     const Vec<Vec<double> >& referenceMatrix,
		     double scoreLimit,
		     int verboseLevel)
{
  cout << "Start of computeKnnPrediction!" << endl;
  Vec<Vec<double> > result(compMatrices[0].size(), Vec<double>(compMatrices[0].size(), 0.0));
  for (int i = 0; i < static_cast<int>(result.size()); ++i) {
    for (int j = 0; j < i-neuralPatchSize; ++j) {
      result[i][j] = computeKnnPrediction(i, j, compMatrices, globalFeatures,
	    knnNode, knnMask, 
	    neuralPatchSize, neuralFileMode, neuralAllMode, 0);
      result[j][i] = result[i][j];
      if ((verboseLevel > 1) && (referenceMatrix[i][j] >= scoreLimit)) {
	cout << "Result of " << i << " " << j << " " << referenceMatrix[i][j] << " "
	     << result[i][j] << " " << endl;
	// recomputed with higher verbose level for output
	computeKnnPrediction(i, j, compMatrices, globalFeatures,
	       	     knnNode, knnMask, 
	       	     neuralPatchSize, neuralFileMode, neuralAllMode, 2);
      }
    }
  }
  cout << "End of computeKnnPrediction!" << endl;
  return result;
}

/** compute knn prediction for given feature vectors.
    knnNet: network of knn classifiers
    borderKnnNetVec: used for predictions near border.
    precondition: knnSubset must be sorted if non empty
*/
Vec<Vec<double> >
computeKnnPrediction(const Vec<Vec<Vec<double> > >& compMatrices,
		     const Vec<Vec<int> >& filterMatrix,
		     const Vec<double>& globalFeatures,
		     const ClassifierBase& knnNet,
		     const Vec<KnnNet2>& knnBorderVec,
		     int neuralPatchSize,
		     int neuralFileMode,
		     const Vec<Vec<double> >& referenceMatrix,
		     double scoreLimit,
		     Vec<unsigned int> knnStartSubset,
		     Vec<unsigned int> knnStopSubset,
		     int dMax,
		     int dMin,
		     int sumMax,
		     int sumMin,
		     int verboseLevel)
{
  cout << "Start of computeKnnPrediction!" << endl;
  ERROR_IF((dMax < dMin), "Maximum index difference must be larger than minimum index difference!");
  Vec<Vec<double> > result(compMatrices[0].size(), Vec<double>(compMatrices[0].size(), 0.0));
  // if smaller zero, take default values:
  if (knnStartSubset.size() == 0) {
    knnStartSubset = generateStair(result.size());
  }
  if (knnStopSubset.size() == 0) {
    knnStopSubset = generateStair(result.size());
  }
  for (int ii = 0; ii < static_cast<int>(knnStartSubset.size()); ++ii) {
    int i = knnStartSubset[ii];
    for (int jj = 0; jj < static_cast<int>(knnStopSubset.size()); ++jj) {
      int j = knnStopSubset[jj];
      // if (j + neuralPatchSize ) >= i) ...
      // if outside "corridor"
      if ((filterMatrix[i][j] == 0) || (abs(j-i) > dMax) || (abs(j-i) < dMin)
          || ((i+j) > sumMax) || ((i + j) < sumMin)) {
	continue;
      }
      
      result[i][j] = computeKnnPrediction(i, j, compMatrices, globalFeatures,
					  knnNet, neuralPatchSize, neuralFileMode, 0);
      result[j][i] = result[i][j];
      if (verboseLevel > 2) {
	cout << "Running knn prediction for " << i + 1 << " " << j + 1 << " " << result[i][j] << endl;
      }
      if ((verboseLevel > 1) && (referenceMatrix[i][j] >= scoreLimit)) {
	cout << "Result of " << i << " " << j << " " << referenceMatrix[i][j] << " "
	     << result[i][j] << " " << endl;
	// recomputed with higher verbose level for output
	computeKnnPrediction(i, j, compMatrices, globalFeatures,
			     knnNet, neuralPatchSize, neuralFileMode, 2);
      }
    }
  }
  int bb = knnBorderVec.size(); // number of special predictions for border
  for (int k = 0; k < bb; ++k) { // choose classifier
    cout << "Applying knn classifier for border " << k + 1 << endl;
    for (int i = neuralPatchSize; i < static_cast<int>(result.size()); ++i) {
      int j = k;
      result[i][j] = computeKnnPrediction(i, j, compMatrices, globalFeatures,
					  knnBorderVec[k], neuralPatchSize, neuralFileMode, 0);
      result[j][i] = result[i][j];
      if ((verboseLevel > 1) && (referenceMatrix[i][j] >= scoreLimit)) {
	cout << "Result of border(1) " << k << " : " << i << " " << j 
	     << " " << referenceMatrix[i][j] << " "
	     << result[i][j] << " " << endl;
	// recomputed with higher verbose level for output
	computeKnnPrediction(i, j, compMatrices, globalFeatures,
			     knnNet, neuralPatchSize, neuralFileMode, 2);
      }
    }
    int i = result.size()-1;
    for (int j = 0; j < i - neuralPatchSize; ++j) {
      result[i][j] = computeKnnPrediction(i, j, compMatrices, globalFeatures,
					  knnBorderVec[k], neuralPatchSize, neuralFileMode, 0);
      result[j][i] = result[i][j];
      if ((verboseLevel > 1) && (referenceMatrix[i][j] >= scoreLimit)) {
	cout << "Result of border(2) " << k << " : " << i << " " << j 
	     << " " << referenceMatrix[i][j] << " "
	     << result[i][j] << " " << endl;
	// recomputed with higher verbose level for output
	computeKnnPrediction(i, j, compMatrices, globalFeatures,
			     knnBorderVec[k], neuralPatchSize, neuralFileMode, 2);
      }
    }
  }
  cout << "End of computeKnnPrediction!" << endl;
  return result;
}

Vec<Vec<double> >
computeKnnTrainPrediction(ostream& os,
			  const Vec<Vec<Vec<double> > >& compMatrices,
			  const Vec<double>& globalFeatures,
			  const ClassifierBase& knnNet,
			  int neuralPatchSize,
			  int neuralFileMode,
			  const Vec<Vec<double> >& referenceMatrix,
			  double scoreLimit,
			  int verboseLevel)
{
  cout << "Start of computeKnnPrediction!" << endl;
  Vec<Vec<double> > result(compMatrices[0].size(), Vec<double>(compMatrices[0].size(), 0.0));
  for (int i = 0; i < static_cast<int>(result.size()); ++i) {
    for (int j = 0; j < i-neuralPatchSize; ++j) {
      unsigned int trueClass = 0;
      if (referenceMatrix[i][j] >= scoreLimit) {
	trueClass = 1;
      }
      result[i][j] = computeKnnTrainPrediction(os, i, j, compMatrices, globalFeatures,
					       knnNet, neuralPatchSize, neuralFileMode, 
					       trueClass, 0);
      result[j][i] = result[i][j];
      if ((verboseLevel > 1) && (referenceMatrix[i][j] >= scoreLimit)) {
	cout << "Result of " << i << " " << j << " " << referenceMatrix[i][j] << " "
	     << result[i][j] << " " << endl;
	// recomputed with higher verbose level for output
	computeKnnPrediction(i, j, compMatrices, globalFeatures,
			     knnNet, neuralPatchSize, neuralFileMode, 2);
      }
    }
  }
  cout << "End of computeKnnPrediction!" << endl;
  return result;
}

void
writeList(ostream& os, 
	  const Vec<Vec<double> >& compMatrix,
	  double scoreLimit)
{
  for (unsigned int i = 0; i < compMatrix.size(); ++i) {
    for (unsigned int j = i + 2; j < compMatrix.size(); ++j) {
      if (compMatrix[i][j] >= scoreLimit) {
	os << i+1 << " " << j+1 << " " << compMatrix[i][j] << endl;
      }
    }
  }
}

void
writeList(ostream& os, 
	  const Vec<Vec<double> >& compMatrix,
	  const Vec<Vec<Vec<double> > >& compMatrices,
	  double scoreLimit,
	  const KnnNet2& knnNet,
	  const Vec<double>& globalFeatures,
	  int neuralPatchSize,
	  int neuralFileMode,
	  int verboseLevel)
{
  unsigned int knnLevel = knnNet.getNumLevels();
  if (knnLevel > 1) {
    --knnLevel; // use one minus highest level for printout
  }
  for (unsigned int i = 0; i < compMatrix.size(); ++i) {
    for (unsigned int j = i+2; j < compMatrix.size(); ++j) {
      if (compMatrix[i][j] >= scoreLimit) {
	os << i+1 << " " << j+1 << " " << compMatrix[i][j] << " ";
	// compute knn level vector
	Vec<double> knnResult = computeKnnPredictionLevel(i,j, compMatrices,
				  globalFeatures, knnNet,neuralPatchSize, 
				  neuralFileMode, knnLevel, verboseLevel);
	os << knnResult;
      }
    }
  }
}

void
findTwoHighestNonDiagonal(const Vec<Vec<double> >& matrix,
			  int i,
			  double& highest,
			  double& second,
			  int offDiag)
{
  highest = matrix[i][0];
  int highestj = 0;
  if ((i-0) < offDiag) {
    highest = matrix[i][matrix[i].size()-1];
    highestj = matrix[i].size()-1;
  }

  for (int j1 = 0; j1 < static_cast<int>(matrix[i].size()); ++j1) {
    if (abs(i-j1) < offDiag) {
      continue;
    }
    if (matrix[i][j1] > highest) {
	highest = matrix[i][j1];
	highestj = j1;
    }
  }
  second = -1e30;
  for (int j2 = 0; j2 < static_cast<int>(matrix[i].size()); ++j2) {
    if ((abs(i-j2) < offDiag) || (abs(highestj - j2) < 4)) {
      continue;
    }
    if (matrix[i][j2] > second) {
      second = matrix[i][j2];
    }      
  }
}


double
computeSwitchingProbability(const Vec<Vec<double> >& compMatrix,
			    const Vec<Vec<double> >& switchParams,
			    const Vec<Vec<double> >& nonSwitchParams,
			    Vec<double>& contrib)
{
  double sumSwitch = 0.0;
  double sumNonSwitch = 0.0;
  int numDiv = switchParams.size();
  int binx, biny;
  contrib = Vec<double>(compMatrix.size(), 0.0);
  // int count = 0;
  int offDiag = 3;
  double highest, second;
  for (int i = 0; i < static_cast<int>(compMatrix.size()); ++i) {
    findTwoHighestNonDiagonal(compMatrix, i, highest, second, offDiag);
    binx = static_cast<int>(numDiv * highest);
    biny = static_cast<int>(numDiv * second);
    if ((binx < 0.0) || (biny < 0.0)) {
      continue;
    }
    if (binx >= static_cast<int>(switchParams.size())) {
      binx = switchParams.size()-1;
    }
    if (biny >= static_cast<int>(switchParams[binx].size())) {
      biny = switchParams[binx].size()-1;
    }
    //  ++count;
    sumSwitch += switchParams[binx][biny];
    sumNonSwitch += nonSwitchParams[binx][biny];

    contrib[i] = switchParams[binx][biny] - nonSwitchParams[binx][biny];

  }
  cout << "Raw sumSwitch: " << sumSwitch << endl;
  cout << "Raw sumNonSwitch: " << sumNonSwitch << endl;  

  /*
  sumSwitch /= count;
  sumNonSwitch /= count;
  double sumSwitchExp = exp(sumSwitch);
  double sumNonSwitchExp = exp(sumNonSwitch);

  double r = sumSwitchExp / sumNonSwitchExp;
  double x = exp(1.0 / static_cast<double>(count));

  double result = (r * () /  ();
  */

  return sumSwitch - sumNonSwitch;
}

/** returns true if contact between two nucleotides is "possible" 
    Basically allow AU, GC, GU basepairs, nothing else. */
bool
contactPossible(int n, int m,
		const SequenceAlignment& ali,
		const CompensationScorer& scorer)
{
  int MIN_OFF_DIAG = 4;
  if (abs(n-m) < MIN_OFF_DIAG) {
    return false;
  }
  const string& s = ali.getSequence(0); // look at target sequence
  ASSERT((n < static_cast<int>(s.size())) && (m < static_cast<int>(s.size())));
  return scorer.isAllowedPair(s[n], s[m]);
}

/** returns true if contact between two nucleotides is "possible" 
    Basically allow AU, GC, GU basepairs, nothing else. */
void
filterImpossible(Vec<Vec<double> >& compMatrix,
		 const SequenceAlignment& ali,
		 const CompensationScorer& scorer)
{
  for (int i = 0; i < static_cast<int>(compMatrix.size()); ++i) {
    for (int j = 0; j < static_cast<int>(compMatrix[i].size()); ++j) {
      if (!contactPossible(i, j, ali, scorer)) {
	compMatrix[i][j] = 0.0;
      }
    }
  }
}

Vec<unsigned int>
findPossibleContacts(const Vec<Vec<double> >& compMatrix,
		     int n,
		     double scoreLimit)
{
  Vec<unsigned int> result;
  for (int i = 0; i < static_cast<int>(compMatrix.size()); ++i) {
    if (abs(n-i) < 3) {
      continue;
    }
    if (compMatrix[n][i] >= scoreLimit) {
      result.push_back(static_cast<unsigned int>(i));
    }
  }
  return result;
}

/** write constraints that can be read by RNAfold */
void
writeConstraintAlignment(ostream& os,
			 const Vec<Vec<double> >& compMatrix, 
			 const SequenceAlignment& ali, 
			 double constraintCutoff,
			 double conflictCutoff,
			 bool withGaps)
{
  Vec<Stem> stems = generateStemsFromMatrix(compMatrix,
					    1, constraintCutoff, ali.getSequence(0));
  for (unsigned int i = 0; i < ali.size(); ++i) {
    os << ">" << ali.getName(i) << endl;
    string seq = ali.getSequence(i);
    string constraints = stemsToBracketFasta(stems, ali.getSequence(i));
    if (!withGaps) {
      seq = removeFromStringParallel(seq, '-', constraints, constraints);
    }
    os << seq << endl;
    os << constraints << endl;
  }					 
}

void
readOptimizeParameterFile(istream& is,
			  Vec<int>& startA, 
			  Vec<int>& startB, 
			  Vec<int>& optTotLen, 
			  Vec<int>& optBorderLen)
{
  int numEntries;
  is>> numEntries;
  ERROR_IF((numEntries < 0) || (numEntries > 10000),
	   "Invalid number of entries in readOptimizeParameterFile!");
  int a, b, c, d;
  startA.clear();
  startB.clear();
  optTotLen.clear();
  optBorderLen.clear();
  for (int i = 0; i < numEntries; ++i) {
    is >> a >> b >> c >> d;
    --a; // convert to internal counting
    --b;
    ERROR_IF(!is, "Error reading readOptimizeParameterFile!");
    startA.push_back(a);
    startB.push_back(b);
    optTotLen.push_back(c);
    optBorderLen.push_back(d);
  }
}

Vec<double>
generatePostProcessFeatureVec(int x, 
			      int y, 
			      const Vec<Vec<double> >& compMatrix, 
			      const Vec<Vec<Vec<double> > >& compMatrices,
			      double referenceVal)
{
  ERROR_IF((x < 0) || (y < 0) 
	   || (x >= static_cast<int>(compMatrix.size())) 
	   || (y >= static_cast<int>(compMatrix.size())),
	   "Internal error in line 4111!");
  unsigned int numDim = compMatrices.size() + 2 + 6;
  Vec<double> result(numDim, 0.0);
  unsigned int pc = 0;
  result[pc++] = x;
  result[pc++] = y;
  for (unsigned int i = 0; i < compMatrices.size(); ++i) {
    result[pc++] = compMatrices[i][x][y];
  }
  if (((x +2) < static_cast<int>(compMatrix.size())) && (y >= 2)) {
    result[pc++] = compMatrix[x+2][y-2];
  }
  else {
    result[pc++] = 0.0;
  }
  if ((x >= 2) && ((y+2) < static_cast<int>(compMatrix.size()))) {
    result[pc++] = compMatrix[x-2][y+2];
  }
  else {
    result[pc++] = 0.0;
  }
  if (((x + 1) < static_cast<int>(compMatrix.size())) && (y >= 1)) {
    result[pc++] = compMatrix[x+1][y-1];
  }
  else {
    result[pc++] = 0.0;
  }
  if ((x >= 1) && ((y+1) < static_cast<int>(compMatrix.size()))) {
    result[pc++] = compMatrix[x-1][y+1];
  }
  else {
    result[pc++] = 0.0;
  }
  result[pc++] = compMatrix[x][y];
  result[pc++] = referenceVal;
  POSTCOND(result.size() == pc);
  return result;
}

/** for each field, write its properties AND prediction results of neighbors and reference data */
void
writeLoopFile(ostream& os, 
	      const Vec<Vec<double> >& compMatrix, 
	      const Vec<Vec<Vec<double> > >&compMatrices, 
	      const Vec<Vec<double> >& referenceMatrix)
{
  // unsigned int numMatrices = compMatrices.size();
  Vec<double> featureVec;
  for (int i = 2; i + 2 < static_cast<int>(compMatrix.size()); ++i) {
    for (int j = i + 3; j + 2 < static_cast<int>(compMatrix.size()); ++j) {
      featureVec = generatePostProcessFeatureVec(i, j, compMatrix, compMatrices, referenceMatrix[i][j]);
      for (unsigned int k = 0; k < featureVec.size(); ++k) {
	os << featureVec[k] << " ";
      }
      os << endl;
    }
  }
}

/** for each field, write its properties AND prediction results of neighbors and reference data */
void
writeLoopFile2(ostream& os, 
	       const Vec<Vec<double> >& compMatrix, 
	       const Vec<Vec<Vec<double> > >&compMatrices, 
	       const Vec<Vec<double> >& referenceMatrix,
	       int maxDiff)
{
  // unsigned int numMatrices = compMatrices.size();
  Vec<double> featureVec;
  for (int i = 2; i + 2 < static_cast<int>(compMatrix.size()); ++i) {
    for (int j = i + 3; j + 2 < static_cast<int>(compMatrix.size()); ++j) {
      featureVec = generatePostProcessFeatureVec(i, j, compMatrix, compMatrices, referenceMatrix[i][j]);
      for (unsigned int k = 0; k < featureVec.size(); ++k) {
	os << featureVec[k] << " ";
      }
      os << endl;
    }
  }
}



void
growMatrixRound(Vec<Vec<double> >& compMatrix,
		const ClassifierBase& knnNet,
		const Vec<Vec<Vec<double> > >& compMatrices,
		int borderSize,
		Vec<unsigned int> knnStartSubset,
		Vec<unsigned int> knnStopSubset,
		int dMax,
		int dMin,
		int sumMax,
		int sumMin)
{
  Vec<Vec<double> > newMatrix(compMatrix.size(), Vec<double>(compMatrix.size(), 0.0));
  double term = 0.0;
  Vec<double> featureVec;
  if (knnStartSubset.size() == 0) {
    knnStartSubset = generateStair(compMatrix.size());
  }
  if (knnStopSubset.size() == 0) {
    knnStopSubset = generateStair(compMatrix.size());
  }
//   for (int i = borderSize; i + borderSize < static_cast<int>(compMatrix.size()); ++i) {
//     for (int j = i + 3; j + borderSize < static_cast<int>(compMatrix.size()); ++j) {

  for (int ii = 0; ii < static_cast<int>(knnStartSubset.size()); ++ii) {
    int i = knnStartSubset[ii];
    if ((i < borderSize) || (i+borderSize >= static_cast<int>(compMatrix.size()))) {
      continue;
    }
    for (int jj = 0; jj < static_cast<int>(knnStopSubset.size()); ++jj) {
      int j = knnStopSubset[jj];
      if ((j < borderSize) 
	  || (j+borderSize >= static_cast<int>(compMatrix.size()))
      	  || (abs(j-i) > dMax) || (abs(j-i) < dMin)
	  || ((i + j) > sumMax) || ((i + j) < sumMin)) {
	continue;
      }
      featureVec = generatePostProcessFeatureVec(i, j, compMatrix, compMatrices, 0.0);
      term = knnNet.predictClassProb(featureVec)[0];
      newMatrix[i][j] = term;
      newMatrix[j][i] = newMatrix[i][j];
    }
  }
  for (unsigned int i = borderSize; i + borderSize < compMatrix.size(); ++i) {
    for (unsigned int j = i + 3; j + borderSize < compMatrix.size(); ++j) {
      // if (newMatrix[i][j] > compMatrix[i][j]) {
      compMatrix[i][j] = newMatrix[i][j];
      compMatrix[j][i] = compMatrix[i][j];
      // }
    }
  }
}

void
growMatrix(Vec<Vec<double> >& compMatrix,
	   const Vec<KnnNet2> & knnNet,
	   const Vec<Vec<Vec<double> > >& compMatrices,
	   Vec<unsigned int> knnStartSubset,
	   Vec<unsigned int> knnStopSubset,
	   int dMax,
	   int dMin,
	   int sumMax,
	   int sumMin,
	   int borderSize,
	   int numRounds)
{
  if (knnStartSubset.size() == 0) {
    knnStartSubset = generateStair(compMatrix.size());
  }
  if (knnStopSubset.size() == 0) {
    knnStopSubset = generateStair(compMatrix.size());
  }
  for (unsigned int k = 0; k < knnNet.size(); ++k) {
    for (int i = 0; i < numRounds; ++i) {
      cout << "Applying matrix post processing knn method: classifier net "
	   << k + 1 << " round " << i + 1 << endl;
      growMatrixRound(compMatrix, knnNet[k], compMatrices, borderSize,
		      knnStartSubset, knnStopSubset, dMax, dMin, 
		      sumMax, sumMin);
    }
  }
}

double
fracMatching(const SequenceAlignment& ali,
	     unsigned int n, 
	     unsigned int m,
	     const CompensationScorer& scorer,
	     char GAP_CHAR)
{
  string s1 = ali.getColumn(n);
  string s2 = ali.getColumn(m);
  unsigned int allowedCount = 0;
  unsigned int totCount = 0;
  for (unsigned int i = 0; i < s1.size(); ++i) {
    if ((s1[i] == GAP_CHAR) || (s2[i] == GAP_CHAR)) {
      continue;
    }
    if (scorer.isAllowedPair(s1[i], s2[i])) {
      ++allowedCount;
    }
    ++totCount;
  }
  if (totCount == 0) {
    return 0.0;
  }
  return (static_cast<double>(allowedCount) / static_cast<double>(totCount));
}

int
simpleGrow(Vec<Vec<double> >& compMatrix, 
	   const SequenceAlignment& ali,
	   double simpleGrowLimit,
	   double simpleGrowMatchLimit,
	   const CompensationScorer& scorer)
{
  cout << "Called simpleGrow with " << simpleGrowLimit << " " << simpleGrowMatchLimit << endl;
  int x, y;
  int nn = compMatrix.size();
  int numChanged = 0;
  for (int i = 0; i < static_cast<int>(compMatrix.size()); ++i) {
    for (int j = i + 3; j < static_cast<int>(compMatrix[i].size()); ++j) {
      if (compMatrix[i][j] > simpleGrowLimit) {
	for (int k = 1; ; ++k) {
	  x = i + k;
	  y = j - k;
	  if ((x >= nn) || (y < 0)) {
	    break;
	  }
	  if ((compMatrix[x][y] < simpleGrowLimit)
	      && (fracMatching(ali, 
			       static_cast<unsigned int>(x), 
			       static_cast<unsigned int>(y), 
			       scorer, GAP_CHAR) 
		  > simpleGrowMatchLimit)) {
	    compMatrix[x][y] = compMatrix[i][j];
	    compMatrix[y][x] = compMatrix[x][y];
	    ++numChanged;
	  }
	  else {
	    // 	    cout << "Breaking up search at : " << i +1  << " " << j + 1 << " "
	    // 		 << compMatrix[i][j] << " " << k << " " << x << " " << y << " " << compMatrix[x][y]
	    // 		 << fracMatching(ali, static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
	    // 				 scorer, Alignment::GAP_CHAR) << endl;
	    break;
	  }
	}
	for (int k = -1; ; --k) {
	  x = i + k;
	  y = j - k;
	  if ((x < 0) || (y >= nn)) {
	    break;
	  }
	  if ((compMatrix[x][y] < simpleGrowLimit)
	      && (fracMatching(ali, 
			       static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
			       scorer, GAP_CHAR) > simpleGrowMatchLimit)) {
	    compMatrix[x][y] = compMatrix[i][j];
	    compMatrix[y][x] = compMatrix[x][y];
	    ++numChanged;
	  }
	  else {
// 	    cout << "Breaking up search at : " << i +1  << " " << j + 1 << " "
// 		 << compMatrix[i][j] << " " << k << " " << x << " " << y << " " << compMatrix[x][y]
// 		 << fracMatching(ali, static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
// 				 scorer, Alignment::GAP_CHAR) << endl;
	    break;
	  }
	}
      }
    }
  }
  return numChanged;
}

/** similar to simpleGrow, but in this case extending stem as long as nucleotides of 
    reference sequence are matching
*/
int
simpleGrow2(Vec<Vec<double> >& compMatrix, 
	    const SequenceAlignment& ali,
	    double simpleGrowLimit,
	    int refSeqIdOrig,
	    const CompensationScorer& scorer)
{
  int x, y;
  int refSeqId;
  if (refSeqIdOrig < 0) {
    refSeqId = 0; // first sequence is reference sequence per default
  }
  else {
    refSeqId = static_cast<unsigned int>(refSeqIdOrig);
  }
  ERROR_IF(refSeqId >= static_cast<int>(ali.size()),
	   "Too large reference sequence index!");
  const string& sequence = ali.getSequence(refSeqId);
  int nn = compMatrix.size();
  int numChanged = 0;
  for (int i = 0; i < static_cast<int>(compMatrix.size()); ++i) {
    for (int j = i + 3; j < static_cast<int>(compMatrix[i].size()); ++j) {
      if (compMatrix[i][j] > simpleGrowLimit) {
	for (int k = 1; ; ++k) {
	  x = i + k;
	  y = j - k;
	  if ((x >= nn) || (y < 0)) {
	    break;
	  }
	  if ((compMatrix[x][y] < simpleGrowLimit)
	      && (scorer.isAllowedPair(sequence[x], sequence[y]))) {
	    compMatrix[x][y] = compMatrix[i][j];
	    compMatrix[y][x] = compMatrix[x][y];
	    ++numChanged;
	  }
	  else {
// 	    cout << "Breaking up search at : " << i +1  << " " << j + 1 << " "
// 		 << compMatrix[i][j] << " " << k << " " << x << " " << y << " " << compMatrix[x][y]
// 		 << fracMatching(ali, static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
// 				 scorer, Alignment::GAP_CHAR) << endl;
	    break;
	  }
	}
	for (int k = -1; ; --k) {
	  x = i + k;
	  y = j - k;
	  if ((x < 0) || (y >= nn)) {
	    break;
	  }
	  if ((compMatrix[x][y] < simpleGrowLimit)
	      && (scorer.isAllowedPair(sequence[x], sequence[y]))) {
	    compMatrix[x][y] = compMatrix[i][j];
	    compMatrix[y][x] = compMatrix[x][y];
	    ++numChanged;

	  }
	  else {
// 	    cout << "Breaking up search at : " << i +1  << " " << j + 1 << " "
// 		 << compMatrix[i][j] << " " << k << " " << x << " " << y << " " << compMatrix[x][y]
// 		 << fracMatching(ali, static_cast<unsigned int>(x), static_cast<unsigned int>(y), 
// 				 scorer, Alignment::GAP_CHAR) << endl;
	    break;
	  }
	}
      }
    }
  }
  return numChanged;
}

Vec<double>
readSequenceWeights(const string& fileName) {
  ifstream file(fileName.c_str());
  ERROR_IF(!file, "Error openening sequence weight file!");
  int numEntries;
  file >> numEntries;
  ERROR_IF((!file) || (numEntries <= 0),
	   "Weight file: Number of entries expected as first item!");
  int id;
  Vec<double> result(numEntries, 0.0);
  for (int i = 0; i < numEntries; ++i) {
    file >> id >> result[i];
  }
  return result;
}

Vec<Vec<double> >
readMatrixFormat(istream& forceMatrixFile, int matrixFormat)
{
  Vec<Vec<double> > compMatrix;
  switch (matrixFormat) {
  case 1: 
    forceMatrixFile >> compMatrix;
    break;
  case 2:
    compMatrix = readListMatrix(forceMatrixFile, 0.0, true); // true: use mirror image too
    break;
  case 3:
    compMatrix = readPlainMatrix(forceMatrixFile);
    break;
  default:
    ERROR("Unknown force matrix file format!");
  }
  return compMatrix;
}

double
fractionMatching(const SequenceAlignment& ali, 
		 unsigned int i, unsigned int j,
		 const CompensationScorer& scorer) {
  string s1 = ali.getColumn(i);
  string s2 = ali.getColumn(j);
  unsigned int countAllowed = scorer.countAllowedPairs(s1, s2);
  return (double)countAllowed/((double)(s1.size()));
}

/** sets all entries to zero that have a too low fraction of compl. base pairs */
void
filterMatchThreshold(Vec<Vec<int> >& filterMatrix, 
		     const SequenceAlignment& ali, 
		     double matchThreshold,
		     const CompensationScorer& scorer)
{
  for (unsigned int i = 0; i < filterMatrix.size(); ++i) {
    for (unsigned int j = i; j < filterMatrix[i].size(); ++j) {
      if (fractionMatching(ali, i,j, scorer) < matchThreshold) {
	filterMatrix[i][j] = 0;
	filterMatrix[j][i] = 0;
      }
    }
  }
}


int
main(int argc, char ** argv)
{
  bool aliChangedFlag = false; // was alignment modified by program?
  bool helpMode = false;
  int algorithm = 14;
  int argcFile = 0;
  int checkAndRepairMode = 0; // if true, run checkAndRepair algorithm that replaces unknown characters
  int cleanAlignmentNamesMode = 0; // if true, remove sequences with duplicate names
  int gapFracRescaleMode = 0;
  int gOpen = 8;
  int gExt = 1;
  int aliCharMax = 12; // 12; // maximum length of RNA fold output names
  int aliLengthMax = 15000; // do not allow greater sequence lengths
  int badColumnMax = 0;
  int collapseId = -1; // if greater zero: collapse alignment (remove gaps) with respect to sequence with this id
  int collapseMode = 1; // defines algorithm for collapsing
  int constraintGapMode = 0; // gaps in constraint output (option -C, --constraint-gaps)
  int dMax = 10000; // maxiumum difference in sequence position of nucleotides
  int dMin = 3; // minimum difference in sequence position of nucleotides
  int exitCode = 0;
  int filterImpossibleMode = 0; // if true, filter out impossible contacts with respect to first sequence
  int forcedStemFormat = 1; // read region file
  int indexOffset = 0; // nucleotide position coutning offset for output
  int individualMode = 1;
  int inputFormat = 1; // 1: fasta 6;
  int knnOptimizeId = -1;
  int knnOptimizeSteps = 200;
  int knnRounds2 = 1; // number of postprocessing rounds
//   int knnSegMin = -1;
//   int knnSegMax = -1; // if greater of equal zero: define region for knn prediction
  int knnWriteLevel = 1; // if > 0: write after n'th level to file
  int matrixFormat = 3; // 1: standard matrix input, 2: read as list of entries
  int matrixNormalMode = 1; // if nonzero, write position write row 1 first, otherwise write row n first
  int neuralAllMode = 1; // if true, compute even borders if possible
  int neuralFileMode = 10; // 5: difference between diagonals 4: average difference between diagonals 3 : differennce between diagonals, 2: diagonals, 1: full square
  int neuralFileMode2 = 9; // 5: difference between diagonals 4: average difference between diagonals 3 : differennce between diagonals, 2: diagonals, 1: full square
  int neuralAliMatrixMode = 2;
  int neuralBothSideMode = 0;
  int neuralMaxLines = -2; // if greater zero, write only this many randomly chosen lines
  int neuralPatchSize = 1; // gives square patch with 2n+1 side length
  int neuralRenameMode = 0;
  int neuralSelfMode = 0; // if one, ignore closest hit of knn search
  int numCharMinFilter = 0;
  Vec<int> optimizeStartA;
  Vec<int> optimizeStartB;; // used with optimization of subset of alignment
  Vec<int> optimizeLengths;
  Vec<int> optimizeBorderLengths;
  int outputFormat = 1;
  int overwriteMode = 0;
  int probConsensusMode = 3; // 1: average, 2: average, but zero if one is zero, 3: non-linear (standard)
  int probMatrixFormat = 3;
  int refSeqId = -1; // which sequence of alignemnt is reference?
  int rnaProbFormat = 1;
  int sequenceWeightMode = 0; // 0: no weighting, 1: simple weighting, 2: external file
  int stemFormat = 1;// 1: region file, 2: bracket notation
  int stemOutFormat = 1;// 1: region file, 2: bracket notation
  int sumMax = 20000; // maxiumum difference in sequence position of nucleotides
  int sumMin = 4; // minimum difference in sequence position of nucleotides
  int winnerMode = 0; // if true, apply winner-takes-all filter to matrix
  char ** argvFile = 0;
  char gapChar = '-';
//   unsigned int addLeft = 0;
//   unsigned int addRight = 0;
  // unsigned int cleaveLength = 0; // length of alignmnet after pruning
  // unsigned int cleavePos = 0; // position (column) to start pruning 
  // unsigned int clipAfter = 0;
  //   unsigned int knnK = 5;
  //   unsigned int knnK2 = 5;
  unsigned int minStemLength = 1;
  unsigned int knnOptimizeTrials = 5000;
  int knnTrainMax = -1; // 0: all lines, > 0: only this many output line. Use with option --knn-train-max & --knn-write-file
  unsigned int wrongPos = 0;
  int neuralConservationMatrixMode = 0; // 1; // include z-score of single column information
  int neuralEnergyMatrixMode = 0; // 1; // include energy matrix
  // include probability matrix if set to one
  int neuralProbabilityMatrixMode = 1; 
  int optimizeMode = 0;
  unsigned int numCharMin = 15; // minimum number of characters in column
  unsigned int optRounds = 1;
  unsigned int pkCounter = 0;
  // unsigned int pruneLength = 0; // length of alignmnet after pruning
  // unsigned int prunePos = 0; // position (column) to start pruning 
  // unsigned int randomizeLength = 0;
  unsigned int randomSeed = 5; // set this random number generator seed if greater zero
  unsigned int subsetOptimizeTrials = 100;
  //   Vec<unsigned int> knnMask;
  //   Vec<unsigned int> knnMask2;
  Vec<int> renormMatrixMode; // (1, 6);
  Vec<unsigned int> subsetOptimizePositions;
  int verboseLevel = 1;
  //   double fm0 = 1.0;
  //   double  fmv = -7.0/6.0;
  //   double fmh = fmv;
  //   double fmd = fmv; // antidiagonal
  //   double fma = 1.0 / 3.0;
  //   double fm0 = 1.0/3.0;
  //   double  fmv = 0.0;
  //   double fmh = 0.0;
  //   double fma = 0.0; // antidiagonal
  //   double fmd = 1.0 / 3.0;
  double constraintCutoff = 0.99;
  double constraintConflictCutoff = 0.5;
  double distanceScaleWeight = 0.0; // changed by EB on Oct 25, 2005 from 0.01;
  double distanceScaleMean = 50.0;
  double entropyRescaleAlpha = 0.0;
  double forcedWeight = -1.0;
  double gapFracAlpha = 1.0; // linear rescaling of result matrix with fraction of gaps
  double gaussDist = -1.0;
  // used for output of neural network feature vector output file: bias towards points close to stem
  double growSimpleLimit = 0.0; // threshold for simple grow function
  double growSimpleMatchLimit = 0.8; // minimum fraction of matching pairs for simple grow function
  double individualThreshold = 0.1;
  double knnGaussDev = 0.2;
  double knnNodeCutoff = 0.001;
  double knnNodeCutoff2 = 0.05;
  double knnOptimizeStepWidth = 0.3;
  double lengthScaleWeight = 0.0;
  double lengthScaleMean = 120.0;
  double matchThreshold = 0.0; // if greater zero, set all output probabilities with small complementariy fraction to zero (not used in publication)
  double minStemAnyways = 0.9;
  double neuralDistBias = 5.0; 
  double normMDLimit = -100.0;
  double optEntropyWeight = 1.0;
  double optMatchWeight = 0.0;
  double optRatioWeight = 0.0;
  double predictionConfidence = 0.8; // if 0.0, use apriori probability, if 1.0, use completely this prediction. Used for --stem-outfile2
  double pStatAvg = 0.0;
  double pStatDev = 0.0;

  double refThresh = 0.5;
  double scoreLimit = 0.15;
  double scoreLimitMediumHigh = 0.5; // originally 0.7
  double scoreLimitHigh = 0.7;
  double scoreStemFileThreshold = -5.0; // limit for stem score in stem file (use with option --stem-outfile2, --stem-outfile2-limit)
  double scoreStemFileWeight = 1.0; // weight for stem score in stem file (use with option --stem-outfile2, --stem-outfile2-weight)
  double stemWeight = 0.0;
  double winnerCutoff = 0.02; // used for speedup of winner takes all 
  double wrongPosFrac = 0.0;
  string aliOutFileName;
  string aliRefName;
  string alphabet = "ACGU";
  string commandFileName;
  string constraintFileName;
  string corrOutName; //  = "compass.corr"
  string entropyMatrixOutFileName;
  string entropyMatrixReadFileName;
  string forceMatrixFileName;
  string forcedStemFileName;
  string inputFileName;
  string listFileName;
  string knnFileName;
  string knnTrainFileName;
  // string knnFileName2;
  string knnReadDir;
  string knnReadDir2;
  string knnSaveFileName;
  string knnStartSubsetString;
  string knnStopSubsetString;
  string knnWriteFileName;
  string logFileName; //  = "mainprogramtemplate.log";
  string loopFileName;
  string matrixOutIdsString;
  string neuralFileName;
  string neuralHeadFileName;
  string parameterFileName = "compass.prm";
  string probDirName;
  string probFileName;
  string probMatrixFileName;
  string probMatrixOutFileName;
  string pStatFileName;
  // string outFileName = "compass.out";
  string optimizeParameterFileName;
  string outputFileName; //  = "compass.out";
  string rescaleFileName;
  string rootDir = "/home/eloop/bindewae/project/supermol3/resources/consensus";  string scoreStemFileName;
  string secondaryAlignmentFileName;
  string sequenceWeightFileName;
  string stemFileName;
  string stemOutFileName;
  string treeStatFileName;
  SimpleSequenceAlignment ali, secondaryAli, aliRef;
  CompensationScorer scorer, stemScorer;
  KnnNet2 knnNet;
  Random& rnd = Random::getInstance();
  // Timer timer;
  Vec<unsigned int> knnStartSubset; // compute prediction only for this subset
  Vec<unsigned int> knnStopSubset; // compute prediction only for this subset
  Vec<unsigned int> matrixOutIds; // write which matrices to file
  Vec<Vec<int> > sub; // substitution matrix
  Vec<Vec<double> > rescaleData;
  Vec<Stem> referenceStems;
  Vec<Stem> forcedStems;
  Vec<string> knnFileName2; // for "growth" iterations
  Vec<string> knnBorderNames; // for computing border
  Vec<string> probNames;
  // store optional pairing probability matrices of sequences
  Vec<Vec<RankedSolution5<unsigned int, unsigned int> > > probVectors; 
  getArg("-help", helpMode, argc, argv);
  if ((argc < 2) || helpMode)  {
    helpOutput(cout);
    exit(0);
  }
  ERROR_IF(!checkArg(argc, argv),
	   "Unknown option found! Try option --help");
  cout << optEntropyWeight << " " << optMatchWeight << " " << optRatioWeight << endl;
  getArg("-root", rootDir, argc, argv, rootDir);
  addSlash(rootDir);
  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);
  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.");
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }
  getArg("-algorithm", algorithm, argcFile, argvFile, algorithm);
  getArg("-algorithm", algorithm, argc, argv, algorithm);
  getArg("-ali-length", aliLengthMax, argcFile, argvFile, aliLengthMax);
  getArg("-ali-length", aliLengthMax, argc, argv, aliLengthMax);
  getArg("-ali-out", aliOutFileName, argcFile, argvFile, aliOutFileName);
  getArg("-ali-out", aliOutFileName, argc, argv, aliOutFileName);
  getArg("-ali-ref", aliRefName, argcFile, argvFile, aliRefName);
  getArg("-ali-ref", aliRefName, argc, argv, aliRefName);
//   getArg("-add-left", addLeft, argcFile, argvFile, addLeft);
//   getArg("-add-left", addLeft, argc, argv, addLeft);
//   getArg("-add-right", addRight, argcFile, argvFile, addRight);
//   getArg("-add-right", addRight, argc, argv, addRight);
  getArg("C", constraintFileName, argcFile, argvFile, constraintFileName);
  getArg("C", constraintFileName, argc, argv, constraintFileName);
  getArg("-char-min", numCharMin, argcFile, argvFile, numCharMin);
  getArg("-char-min", numCharMin, argc, argv, numCharMin);
  getArg("-char-min-filter", numCharMinFilter, argcFile, argvFile, numCharMinFilter);
  getArg("-char-min-filter", numCharMinFilter, argc, argv, numCharMinFilter);
//   getArg("-cleave-length", cleaveLength, argcFile, argvFile, cleaveLength);
//   getArg("-cleave-length", cleaveLength, argc, argv, cleaveLength);
//   getArg("-cleave-pos", cleavePos, argcFile, argvFile, cleavePos);
//   getArg("-cleave-pos", cleavePos, argc, argv, cleavePos);
  // getArg("-clip-after", clipAfter, argc, argv, clipAfter);
  getArg("-collapse", collapseId, argcFile, argvFile, collapseId);
  getArg("-collapse", collapseId, argc, argv, collapseId);
  --collapseId; // internal counting starts at zero
  getArg("-collapse-mode", collapseMode, argcFile, argvFile, collapseMode);
  getArg("-collapse-mode", collapseMode, argc, argv, collapseMode);
  getArg("-constraint-gap-mode", constraintGapMode, 
	 argcFile, argvFile, constraintGapMode);
  getArg("-constraint-gap-mode", constraintGapMode,
	 argc, argv, constraintGapMode);
  getArg("-corr-file", corrOutName, argcFile, argvFile, corrOutName);
  getArg("-corr-file", corrOutName, argc, argv, corrOutName);
  getArg("-distance-mean", distanceScaleMean, argcFile, argvFile,
	 distanceScaleMean);
  getArg("-distance-mean", distanceScaleMean, argc, argv,
	 distanceScaleMean);
  getArg("-distance-weight", distanceScaleWeight, argcFile, argvFile,
	 distanceScaleWeight);
  getArg("-distance-weight", distanceScaleWeight, argc, argv,
	 distanceScaleWeight);
  getArg("-dmax", dMax, argcFile, argvFile, dMax); 
  getArg("-dmax", dMax, argc, argv, dMax); 
  getArg("-dmin", dMin, argcFile, argvFile, dMin); 
  getArg("-dmin", dMin, argc, argv, dMin); 
  getArg("i", inputFileName, argcFile, argvFile, inputFileName);
  getArg("i", inputFileName, argc, argv, inputFileName);
  getArg("-if", inputFormat, argcFile, argvFile, inputFormat);
  getArg("-if", inputFormat, argc, argv, inputFormat);
  getArg("-energy", scorer.energyWeight, argcFile, argvFile, 
	 scorer.energyWeight);
  getArg("-energy", scorer.energyWeight, argc, argv, scorer.energyWeight);
  getArg("-energy-min", scorer.compMin, argcFile, argvFile, scorer.compMin);
  getArg("-energy-min", scorer.compMin, argc, argv, scorer.compMin);
  getArg("-entropy", scorer.entropyWeight, argcFile, argvFile, 
	 scorer.entropyWeight);
  getArg("-entropy", scorer.entropyWeight, argc, argv, scorer.entropyWeight);
  getArg("-entropy-matrix-out", entropyMatrixOutFileName, argcFile, argvFile, 
	 entropyMatrixOutFileName);
  getArg("-entropy-matrix-out", entropyMatrixOutFileName, argc, argv, 
	 entropyMatrixOutFileName);
  getArg("-entropy-read", entropyMatrixReadFileName, argcFile, argvFile, entropyMatrixReadFileName);
  getArg("-entropy-read", entropyMatrixReadFileName, argc, argv, entropyMatrixReadFileName);
  getArg("-entropy-rescale", entropyRescaleAlpha,
	 argcFile, argvFile, entropyRescaleAlpha);
  getArg("-entropy-rescale", entropyRescaleAlpha,
	 argc, argv, entropyRescaleAlpha);
  getArg("-exit", exitCode, argcFile, argvFile, exitCode);
  getArg("-exit", exitCode, argc, argv, exitCode);
  getArg("-filter", filterImpossibleMode, argcFile, argvFile, filterImpossibleMode);
  getArg("-filter", filterImpossibleMode, argc, argv, filterImpossibleMode);
  getArg("-filter-bad", badColumnMax, argcFile, argvFile, badColumnMax);
  getArg("-filter-bad", badColumnMax, argc, argv, badColumnMax);
  getArg("-forced-stems", forcedStemFileName, argcFile, argvFile, 
	 forcedStemFileName);
  getArg("-forced-stems", forcedStemFileName, argc, argv, 
	 forcedStemFileName);
  getArg("-forced-stems-format", forcedStemFormat, argcFile, argvFile, 
	 forcedStemFormat);
  getArg("-forced-stems-format", forcedStemFormat, argc, argv, 
	 forcedStemFormat);
  getArg("-forced-weight", forcedWeight, argcFile, argvFile, 
	 forcedWeight);
  getArg("-forced-weight", forcedWeight, argc, argv, 
	 forcedWeight);
  getArg("g", gaussDist, argcFile, argvFile, gaussDist);
  getArg("g", gaussDist, argc, argv, gaussDist);
  getArg("-gap-frac-alpha", gapFracAlpha, argcFile, argvFile, gapFracAlpha);
  getArg("-gap-frac-alpha", gapFracAlpha, argc, argv, gapFracAlpha);
  if (isPresent("-gap-frac-alpha", argcFile, argvFile)
      || isPresent("-gap-frac-alpha", argc, argv)) {
    gapFracRescaleMode = 1;
  }
  getArg("-grow-limit", growSimpleLimit, argcFile, argvFile, growSimpleLimit);
  getArg("-grow-limit", growSimpleLimit, argc, argv, growSimpleLimit);
  ERROR_IF((growSimpleLimit < 0.0) || (growSimpleLimit > 1.0),
	   "--grow-limit value must be between 0 and 1.");
  getArg("-grow-match", growSimpleMatchLimit, argcFile, argvFile, 
	 growSimpleMatchLimit);
  getArg("-grow-match", growSimpleMatchLimit, argc, argv, 
	 growSimpleMatchLimit);
  ERROR_IF((growSimpleMatchLimit < 0.0) || (growSimpleMatchLimit > 1.0),
	   "--grow-limit value must be between 0 and 1.");
  getArg("-knn-cutoff2", knnNodeCutoff, argcFile, argvFile, knnNodeCutoff);
  getArg("-knn-cutoff2", knnNodeCutoff, argc, argv, knnNodeCutoff);
  getArg("-knn-cutoff22", knnNodeCutoff2, argcFile, argvFile, knnNodeCutoff2);
  getArg("-knn-cutoff22", knnNodeCutoff2, argc, argv, knnNodeCutoff2);
  getArg("-knn-data", knnFileName, argcFile, argvFile, knnFileName);
  getArg("-knn-data", knnFileName, argc, argv, knnFileName);
  getArg("-knn-data2", knnFileName2, argcFile, argvFile);
  getArg("-knn-data2", knnFileName2, argc, argv);
  getArg("-knn-data-border", knnBorderNames, argcFile, argvFile);
  getArg("-knn-data-border", knnBorderNames, argc, argv);
  getArg("-knn-dir", knnReadDir, argcFile, argvFile, knnReadDir);
  getArg("-knn-dir", knnReadDir, argc, argv, knnReadDir);
  getArg("-knn-dir2", knnReadDir2, argcFile, argvFile, knnReadDir2);
  getArg("-knn-dir2", knnReadDir2, argc, argv, knnReadDir2);
  getArg("-knn-rounds2", knnRounds2, argcFile, argvFile, knnRounds2);
  getArg("-knn-rounds2", knnRounds2, argc, argv, knnRounds2);
  getArg("-knn-gauss", knnGaussDev, argcFile, argvFile, knnGaussDev);
  getArg("-knn-gauss", knnGaussDev, argc, argv, knnGaussDev);
//   getArg("-knn-min", knnSegMin, argc, argv, knnSegMin);
//   --knnSegMin;
//   getArg("-knn-max", knnSegMax, argc, argv, knnSegMax);
//   --knnSegMax; // conver to internal counting
  getArg("-knn-optimize", knnOptimizeId, argcFile, argvFile, knnOptimizeId);
  getArg("-knn-optimize", knnOptimizeId, argc, argv, knnOptimizeId);
  --knnOptimizeId; // convert to internal counting
  getArg("-knn-optimize-steps", knnOptimizeSteps, 
	 argcFile, argvFile, knnOptimizeSteps);
  getArg("-knn-optimize-steps", knnOptimizeSteps, 
	 argc, argv, knnOptimizeSteps);
  getArg("-knn-save-file", knnSaveFileName, argcFile, argvFile, 
	 knnSaveFileName);
  getArg("-knn-save-file", knnSaveFileName, argc, argv, knnSaveFileName);
  getArg("-subset-start", knnStartSubsetString, argc, argv, knnStartSubsetString); // parse "-3,6-8,12" to 1,2,3,6,7,8,12
  knnStartSubset = parseStringToVector(knnStartSubsetString);
  convert2InternalCounting(knnStartSubset);
  sort(knnStartSubset.begin(), knnStartSubset.end());
  getArg("-subset-stop", knnStopSubsetString, argc, argv, knnStopSubsetString); // parse "-3,6-8,12" to 1,2,3,6,7,8,12
  knnStopSubset = parseStringToVector(knnStopSubsetString);
  convert2InternalCounting(knnStopSubset);
  sort(knnStopSubset.begin(), knnStopSubset.end());
  getArg("-knn-train-file", knnTrainFileName, argcFile, argvFile);
  getArg("-knn-train-file", knnTrainFileName, argc, argv, knnTrainFileName);
  getArg("-knn-write-file", knnWriteFileName, argcFile, argvFile, knnWriteFileName);
  getArg("-knn-write-file", knnWriteFileName, argc, argv, knnWriteFileName);
  getArg("-knn-write-level", knnWriteLevel, argcFile, argvFile, knnWriteLevel);
  getArg("-knn-write-level", knnWriteLevel, argc, argv, knnWriteLevel);
  --knnWriteLevel; // internal counting starts at zero
  getArg("-knn-write-max", knnTrainMax, argcFile, argvFile, knnTrainMax);
  getArg("-knn-write-max", knnTrainMax, argc, argv, knnTrainMax);
  getArg("-length-mean", lengthScaleMean, argcFile, argvFile,
	 lengthScaleMean);
  getArg("-length-mean", lengthScaleMean, argc, argv,
	 lengthScaleMean);
  getArg("-length-weight", lengthScaleWeight, argcFile, argvFile,
	 lengthScaleWeight);
  getArg("-length-weight", lengthScaleWeight, argc, argv,
	 lengthScaleWeight);
  getArg("-list", listFileName, argcFile, argvFile, listFileName);
  getArg("-list", listFileName, argc, argv, listFileName);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("-loop", loopFileName, argcFile, argvFile, loopFileName);
  getArg("-loop", loopFileName, argc, argv, loopFileName);
  getArg("-match-thresh", matchThreshold, argcFile, argvFile, matchThreshold);
  getArg("-match-thresh", matchThreshold, argc, argv, matchThreshold);
  getArg("-matrix", forceMatrixFileName, argcFile, argvFile, forceMatrixFileName);
  getArg("-matrix", forceMatrixFileName, argc, argv, forceMatrixFileName);
  getArg("-matrix-format", matrixFormat, argcFile, argvFile, matrixFormat);
  getArg("-matrix-format", matrixFormat, argc, argv, matrixFormat);
  getArg("-matrix-normal", matrixNormalMode, argcFile, argvFile,
	 matrixNormalMode);
  getArg("-matrix-normal", matrixNormalMode, argc, argv,
	 matrixNormalMode);
  getArg("-matrix-ids", matrixOutIdsString, argc, argv);
  matrixOutIds = parseStringToVector(matrixOutIdsString);
  convert2InternalCounting(matrixOutIds);
  getArg("-md-limit", normMDLimit, argcFile, argvFile, normMDLimit);
  getArg("-md-limit", normMDLimit, argc, argv, normMDLimit);
  getArg("-neural", neuralFileName, argcFile, argvFile, neuralFileName);
  getArg("-neural", neuralFileName, argc, argv, neuralFileName);
  getArg("-neural-all", neuralAllMode, argcFile, argvFile, neuralAllMode);
  getArg("-neural-all", neuralAllMode, argc, argv, neuralAllMode);
  getArg("-neural-both", neuralBothSideMode, argcFile, argvFile, neuralBothSideMode);
  getArg("-neural-both", neuralBothSideMode, argc, argv, neuralBothSideMode);
  getArg("-neural-dist", neuralDistBias, argcFile, argvFile, neuralDistBias);
  getArg("-neural-dist", neuralDistBias, argc, argv, neuralDistBias);
  getArg("-neural-conserve", neuralConservationMatrixMode, 
	 argcFile, argvFile, neuralConservationMatrixMode); 
  getArg("-neural-conserve", neuralConservationMatrixMode, 
	 argc, argv, neuralConservationMatrixMode); 
  getArg("-neural-energy-mode", neuralEnergyMatrixMode, argcFile, argvFile, 
	 neuralEnergyMatrixMode); 
  getArg("-neural-energy-mode", neuralEnergyMatrixMode, argc, argv, 
	 neuralEnergyMatrixMode); 
  getArg("-neural-head", neuralHeadFileName, argcFile, argvFile, 
	 neuralHeadFileName);
  getArg("-neural-head", neuralHeadFileName, argc, argv, neuralHeadFileName);
  getArg("-neural-max", neuralMaxLines, argcFile, argvFile, 
	 neuralMaxLines);
  getArg("-neural-max", neuralMaxLines, argc, argv, neuralMaxLines);
  getArg("-neural-mode", neuralFileMode, argcFile, argvFile, neuralFileMode);
  getArg("-neural-mode", neuralFileMode, argc, argv, neuralFileMode);
  getArg("-neural-mode2", neuralFileMode2, argcFile, argvFile, neuralFileMode2);
  getArg("-neural-mode2", neuralFileMode2, argc, argv, neuralFileMode2);
  getArg("-neural-mode3", neuralAliMatrixMode, argcFile, argvFile, neuralAliMatrixMode);
  getArg("-neural-mode3", neuralAliMatrixMode, argc, argv, neuralAliMatrixMode);
  getArg("-neural-patch", neuralPatchSize, argcFile, argvFile, 
	 neuralPatchSize);
  getArg("-neural-patch", neuralPatchSize, argc, argv, neuralPatchSize);
  getArg("-neural-prob-mode", neuralProbabilityMatrixMode, argcFile, argvFile, neuralProbabilityMatrixMode); 
  getArg("-neural-prob-mode", neuralProbabilityMatrixMode, argc, argv, neuralProbabilityMatrixMode); 
  getArg("-neural-rename-mode", neuralRenameMode, argcFile, argvFile, neuralRenameMode);
  getArg("-neural-rename-mode", neuralRenameMode, argc, argv, neuralRenameMode);
  getArg("-neural-self-mode", neuralSelfMode, argcFile, argvFile, 
	 neuralSelfMode); 
  getArg("-neural-self-mode", neuralSelfMode, argc, argv, 
	 neuralSelfMode); 
  getArg("-norm", renormMatrixMode, argc, argv);
  // getArg("-norm", renormMatrixMode, argc, argv, renormMatrixMode);
  getArg("o", outputFileName, argcFile, argvFile, outputFileName);
  getArg("o", outputFileName, argc, argv, outputFileName);
  getArg("-of", outputFormat, argcFile, argvFile, outputFormat);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("-optimize", optimizeMode, argcFile, argvFile, optimizeMode);
  getArg("-optimize", optimizeMode, argc, argv, optimizeMode);
  getArg("-optimize-entropy", optEntropyWeight, argcFile, argvFile, optEntropyWeight);
  getArg("-optimize-entropy", optEntropyWeight, argc, argv, optEntropyWeight);
  getArg("-optimize-match", optMatchWeight, argcFile, argvFile, optMatchWeight);
  getArg("-optimize-match", optMatchWeight, argc, argv, optMatchWeight);
  getArg("-optimize-parameters", optimizeParameterFileName, argcFile, argvFile,
	 optimizeParameterFileName);
  getArg("-optimize-parameters", optimizeParameterFileName, argc, argv,
	 optimizeParameterFileName);
  getArg("-optimize-ratio", optRatioWeight, argcFile, argvFile, optRatioWeight);
  getArg("-optimize-ratio", optRatioWeight, argc, argv, optRatioWeight);
  getArg("-optimize-rounds", optRounds, argcFile, argvFile, optRounds);
  getArg("-optimize-rounds", optRounds, argc, argv, optRounds);
  getArg("-overwrite", overwriteMode, argcFile, argvFile, overwriteMode);
  getArg("-overwrite", overwriteMode, argc, argv, overwriteMode);
  getArg("p", parameterFileName, argcFile, argvFile, parameterFileName);
  getArg("p", parameterFileName, argc, argv, parameterFileName);
  getArg("-pfile", pStatFileName, argcFile, argvFile, pStatFileName);
  getArg("-pfile", pStatFileName, argc, argv, pStatFileName);
  addPathIfRelative(parameterFileName, rootDir);
  getArg("-prediction-confidence", predictionConfidence,
	 argcFile, argvFile, predictionConfidence);
  getArg("-prediction-confidence", predictionConfidence,
	 argc, argv, predictionConfidence);
  getArg("-prob-consensus", probConsensusMode, argcFile, argvFile,
	 probConsensusMode);
  getArg("-prob-consensus", probConsensusMode, argc, argv,
	 probConsensusMode);
  getArg("-prob-dir", probDirName, argcFile, argvFile, probDirName);
  getArg("-prob-dir", probDirName, argc, argv, probDirName);
  addSlash(probDirName);
  getArg("-prob-file", probFileName, argcFile, argvFile, probFileName);
  getArg("-prob-file", probFileName, argc, argv, probFileName);
  getArg("-prob-format", rnaProbFormat, argcFile, argvFile, rnaProbFormat);
  getArg("-prob-format", rnaProbFormat, argc, argv, rnaProbFormat);
  getArg("-prob-matrix", probMatrixFileName, argcFile, argvFile, probMatrixFileName);
  getArg("-prob-matrix", probMatrixFileName, argc, argv, probMatrixFileName);
  getArg("-prob-matrix-format", probMatrixFormat, argcFile, argvFile, probMatrixFormat);
  getArg("-prob-matrix-format", probMatrixFormat, argc, argv, probMatrixFormat);
  getArg("-prob-matrix-out", probMatrixOutFileName, argcFile, argvFile, 
	 probMatrixOutFileName);
  getArg("-prob-matrix-out", probMatrixOutFileName, argc, argv, 
	 probMatrixOutFileName);
//   getArg("-prune-length", pruneLength, argcFile, argvFile, pruneLength);
//   getArg("-prune-length", pruneLength, argc, argv, pruneLength);
//   getArg("-prune-pos", prunePos, argcFile, argvFile, prunePos);
//   getArg("-prune-pos", prunePos, argc, argv, prunePos);  
  getArg("-secondary-alignment", secondaryAlignmentFileName,
	 argcFile, argvFile, secondaryAlignmentFileName);
  getArg("-seed", randomSeed, argcFile, argvFile, randomSeed);
  getArg("-seed", randomSeed, argc, argv, randomSeed);
  if (randomSeed > 0) {
    rnd.resetWithSeed(randomSeed); // set seed
  }
  getArg("-sequence-weight-file", sequenceWeightFileName, argcFile, argvFile, sequenceWeightFileName);
  getArg("-sequence-weight-file", sequenceWeightFileName, argc, argv, sequenceWeightFileName);
  getArg("-sequence-weight-mode", sequenceWeightMode, argcFile, argvFile, sequenceWeightMode);
  getArg("-sequence-weight-mode", sequenceWeightMode, argc, argv, sequenceWeightMode);
  getArg("-stem-min", minStemLength, argcFile, argvFile, minStemLength);
  getArg("-stem-min", minStemLength, argc, argv, minStemLength);
  getArg("-stem-min-anyways", minStemAnyways, argcFile, argvFile, minStemAnyways);
  getArg("-stem-min-anyways", minStemAnyways, argc, argv, minStemAnyways);
  getArg("-stem-file", stemFileName, argcFile, argvFile, stemFileName);
  getArg("-stem-file", stemFileName, argc, argv, stemFileName);
  getArg("-stem-outfile", stemOutFileName, argcFile, argvFile, stemOutFileName);
  getArg("-stem-outfile", stemOutFileName, argc, argv, stemOutFileName);
  getArg("-stem-outfile2", scoreStemFileName, argcFile, argvFile, scoreStemFileName);
  getArg("-stem-outfile2", scoreStemFileName, argc, argv, scoreStemFileName);
  getArg("-stem-outfile2-limit", scoreStemFileThreshold, argcFile, argvFile, scoreStemFileThreshold);
  getArg("-stem-outfile2-limit", scoreStemFileThreshold, argc, argv, scoreStemFileThreshold);
  getArg("-stem-outfile2-weight", scoreStemFileWeight, argcFile, argvFile, 
	 scoreStemFileWeight);
  getArg("-stem-outfile2-weight", scoreStemFileWeight, argc, argv, 
	 scoreStemFileWeight);
  getArg("-stem-outfile-format", stemOutFormat, 
	 argcFile, argvFile, stemOutFormat);
  getArg("-stem-outfile-format", stemOutFormat, argc, argv, stemOutFormat);
  getArg("-stem-format", stemFormat, argcFile, argvFile, stemFormat);
  getArg("-stem-format", stemFormat, argc, argv, stemFormat);
  getArg("-stem-weight", stemWeight, argcFile, argvFile, stemWeight);
  getArg("-stem-weight", stemWeight, argc, argv, stemWeight);
  getArg("-subset-optimize", subsetOptimizePositions, argc, argv);
  convert2InternalCounting(subsetOptimizePositions);
  getArg("-sum-max", sumMax, argcFile, argvFile, sumMax); 
  getArg("-sum-max", sumMax, argc, argv, sumMax); 
  sumMax -= 2; // internal index counting
  getArg("-sum-min", sumMin, argcFile, argvFile, sumMin); 
  getArg("-sum-min", sumMin, argc, argv, sumMin); 
  sumMin -= 2; // internal index counting
//   getArg("-randomize", randomizeLength, argcFile, argvFile, randomizeLength);
//   getArg("-randomize", randomizeLength, argc, argv, randomizeLength);
  getArg("-reference", refSeqId, argcFile, argvFile, refSeqId);
  getArg("-reference", refSeqId, argc, argv, refSeqId);
  --refSeqId; // external counting starts from one, internal from zero
  getArg("-rescale", rescaleFileName, argcFile, argvFile, rescaleFileName);
  getArg("-rescale", rescaleFileName, argc, argv, rescaleFileName);
  getArg("-thresh", scoreLimit, argcFile, argvFile, scoreLimit);
  getArg("-thresh", scoreLimit, argc, argv, scoreLimit);
  getArg("-thresh-mediumhigh", scoreLimitMediumHigh, argcFile, argvFile, 
	 scoreLimitMediumHigh);
  getArg("-thresh-mediumhigh", scoreLimitMediumHigh, argc, argv, 
	 scoreLimitMediumHigh);
  getArg("-thresh-high", scoreLimitHigh, argcFile, argvFile, scoreLimitHigh);
  getArg("-thresh-high", scoreLimitHigh, argc, argv, scoreLimitHigh);
  getArg("-thresh-individual", individualThreshold, argcFile, argvFile, individualThreshold);
  getArg("-thresh-individual", individualThreshold, argc, argv, individualThreshold);
  getArg("-tree-stat", treeStatFileName, argcFile, argvFile, treeStatFileName);
  getArg("-tree-stat", treeStatFileName, argc, argv, treeStatFileName);
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);
  getArg("-winner", winnerMode, argcFile, argvFile, winnerMode);
  getArg("-winner", winnerMode, argc, argv, winnerMode);

  scorer.verboseLevel = verboseLevel;

  // timer.start(); // record overall time needed

  if (verboseLevel > 0) {
    cout << "Programm compass called with following parameters: " << endl;
    parameterOutput(cout, argc, argv);
    cout << "Used random number seed: ";
    if (randomSeed > 0) {
      cout << randomSeed << endl;
    }
    else {
      cout << "none." << endl;
    }
  }

  /***************** MAIN PROGRAM *****************************/

  if (parameterFileName.size() > 0) {
    ifstream parameterFile(parameterFileName.c_str());
    ERROR_IF(!parameterFile, "Error opening parameter file!");
    parameterFile >> alphabet >> sub; //  >> switchParams >> nonSwitchParams;
    parameterFile.close();
//     if (verboseLevel > 0) {
//       cout << "Read switching parameters: " << switchParams << endl;
//       cout << "Read non-switching parameters: " << nonSwitchParams << endl;
//     }
  }
  // reads average and standard deviation of scores
  if (pStatFileName.size() > 0) {
    ifstream pFile(pStatFileName.c_str());
    ERROR_IF(!pFile, "Error opening pFile!");
    string dummy; 
    // ignore actual pStatDev (often zero because numerical errors)
    // use instead pStatDev assuming pStat Avg is zero
    pFile >> dummy >> pStatAvg >> dummy >> dummy >> pStatDev;
    pFile.close();
    if (verboseLevel > 0) {
      cout << "Read average and standard deviation of randomized alignments: "
	   << pStatAvg << " +- " << pStatDev << endl;
    }
  }
  // only used for speed optimized data structure
  knnNet.setClusterCutoff(knnNodeCutoff);
  // knnNet.setClusterCutoff2(knnNodeCutoff2); // not defined for KnnNet2
  knnNet.setVerboseLevel(verboseLevel);
  
  if (knnFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading knn net: " << knnFileName << " " 
	   << knnReadDir << endl;
    }
    ifstream knnFile(knnFileName.c_str());
    ERROR_IF(!knnFile, "Error opening knn net data file!");
    knnNet.read(knnFile, knnReadDir);
    ERROR_IF(!knnFile, "Error reading knn net data file!");
    knnFile.close();
    if (verboseLevel > 0) {
      cout << knnNet.size() << " knn nodes read." << endl;
    }
    ERROR_IF(!knnNet.isValid(), "Error in reading K-nearest neigbor data!");
  }
  Vec<KnnNet2> knnNet2;
  if (knnFileName2.size() > 0) {
    knnNet2 = Vec<KnnNet2>(knnFileName2.size());
    for (unsigned int i = 0; i < knnFileName2.size(); ++i) {
      if (verboseLevel > 0) {
	cout << "Reading knn net 2: " << knnFileName2[i] << " " 
	     << knnReadDir2 << endl;
      }
      ifstream knnFile2(knnFileName2[i].c_str());
      ERROR_IF(!knnFile2, "Error opening knn net 2 data file!");
      knnNet2[i].setVerboseLevel(verboseLevel);
      knnNet2[i].read(knnFile2, knnReadDir2);
      ERROR_IF(!knnFile2, "Error reading knn net 2 data file!");
      knnFile2.close();
      if (verboseLevel > 0) {
	cout << knnNet2[i].size() << " knn nodes 2 read." << endl;
      }
      ERROR_IF(!knnNet2[i].isValid(), "Error in reading K-nearest neigbor data!");
      knnNet2[i].setGaussDev(knnGaussDev);
      if (neuralSelfMode != 0) {
	knnNet2[i].setNoSelfMode(true); // avoid problems with training vectors used several times: ignore closest hit always, assume that is occured because of overalp between training and testing
      }
    }
  }
  Vec<KnnNet2> knnBorderVec;
  if (knnBorderNames.size() > 0) {
    knnBorderVec = Vec<KnnNet2>(knnBorderNames.size());
    for (unsigned int i = 0; i < knnBorderNames.size(); ++i) {
      if (verboseLevel > 0) {
	cout << "Reading knn net 2: " << knnBorderNames[i] << " " 
	     << knnReadDir2 << endl;
      }
      ifstream knnFile2(knnBorderNames[i].c_str());
      ERROR_IF(!knnFile2, "Error opening border knn net data file!");
      knnBorderVec[i].setVerboseLevel(verboseLevel);
      knnBorderVec[i].read(knnFile2, knnReadDir2);
      ERROR_IF(!knnFile2, "Error reading border knn net data file!");
      knnFile2.close();
      if (verboseLevel > 0) {
	cout << knnBorderVec[i].size() << " border knn nodes read." << endl;
      }
      ERROR_IF(!knnBorderVec[i].isValid(), "Error in reading K-nearest neigbor data!");
      knnBorderVec[i].setGaussDev(knnGaussDev);
    }
  }
  
  knnNet.setGaussDev(knnGaussDev);

  scorer.setAlphabet(alphabet);
  scorer.setSubMatrix(sub);
    
  ifstream inputFile(inputFileName.c_str());
  ERROR_IF(!inputFile, "Error opening input file!");
  switch (inputFormat) {
  case 1: ali.readFasta(inputFile);
    break;
//   case 4: ali.loadDanforth(inputFile);
//     break;
//   case 6: ali.loadBlastMode6(inputFile);
//     break;
//   case 7: ali.loadCE(inputFile);
//     break;
  default: ERROR("Unknown input format!");
  }
  inputFile.close();

  cout << "Sequence alignment with " << ali.size() << " sequences and length "
       << ali.getLength() << " read." << endl;

  ERROR_IF(static_cast<int>(ali.getLength()) > aliLengthMax, "Alignment sequence length greater than maximum allowed value!");

  if (aliRefName.size() > 0) {
    cout << "Reading reference alignment used for collapsing: " << aliRefName << endl;
    ifstream aliRefFile(aliRefName.c_str());
    ERROR_IF(!aliRefFile, "Error opening input reference file!");
    switch (inputFormat) {
    case 1: aliRef.readFasta(aliRefFile);
      break;
//     case 4: aliRef.loadDanforth(aliRefFile);
//       break;
//     case 6: aliRef.loadBlastMode6(aliRefFile);
//       break;
//     case 7: aliRef.loadCE(aliRefFile);
//       break;
    default: ERROR("Unknown input format!");
    }
    aliRefFile.close();
  }

  // ali.outputLineLength = 0; // each sequence in one line
  // adjust sequences:
  ali.upperCaseSequences();
  ali.replace('T', 'U'); // replace DNA alphabet to RNA
  ali.replace('.', gapChar); // replace "." with "-"
  
  if (verboseLevel > 1) {
      cout << "Read alignment: " << endl;
      ali.writeFasta(cout);
  }

  // read contact predictions from external source:
  if (probFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading pairing probability file list: "
	   << probFileName << endl;
      cout << "Reading pairing probability file: ";
    }
    ifstream probFile(probFileName.c_str());
    ERROR_IF(!probFile, "Error opening prob file!");
    while (probFile) {
      string line = getLine(probFile);
      if (line.size() == 0) {
	continue;
      }
      Vec<string> words = getTokens(line);
      if (words.size() < 2) {
	cout << "Warning: ignoring strange line in alignmnet prob file: " << line << endl;
	continue;
      }
      string fileName = probDirName + words[1];
      ifstream aliProbFile(fileName.c_str());
      if (!aliProbFile) {
	cout << "Warning: Could not read alignment prob file: " << fileName << endl;
	// ERROR("Error opening alignment prob file!");
      }
      if (verboseLevel > 0) {
	cout << fileName << " ";
      }
      Vec<RankedSolution5<unsigned int, unsigned int> > probValues;
      string probName;
      // read probability matrix (in RNAfold postscript format!!!)
      switch (rnaProbFormat) {
      case 1: // read postscript file produces by RNAfold
	aliProbFile >> probName;
	probValues = readRnaFoldProbFile(aliProbFile, gapChar); 
	break;
      case 2: // our own format 
	aliProbFile >> probName;
	probValues = readProbFile(aliProbFile); 
	break;
      case 3: // region file format
	probValues = readProbFromRegionFile(aliProbFile, ali.getLength()); 
	break;
      default:
	ERROR("Unknown RNA probability format!");
      }

      if (probValues.size() == 0) {
	cout << "Warning:  no prob values defined!" << endl;
      }
      probVectors.push_back(probValues);
      probNames.push_back(words[0]);
      // cout << "Reading: " << words[2] << " done!" << endl;
	aliProbFile.close();
    }
    if (verboseLevel > 0) {
	cout << endl;
    }
    probFile.close();
    if (verboseLevel > 1) {
      cout << "Read prob names: " << probNames << endl;
      cout << "Read prob vectors: " << probVectors << endl;
    }
  }
  
  if (secondaryAlignmentFileName.size() > 0) {
    ifstream secondaryAlignmentFile(secondaryAlignmentFileName.c_str());
    ERROR_IF(!secondaryAlignmentFile, "Error opening secondary alignment file!");
    secondaryAli.readFasta(secondaryAlignmentFile);
    secondaryAlignmentFile.close();
  }
  
  // remove duplicates, tr "." character to "-": 
  if (cleanAlignmentNamesMode) {
    cleanAlignment(ali);
    cout << "Sequence alignment after cleanup: " << ali.size() 
	 << " sequences and length "
	 << ali.getLength() << " read." << endl;
  }

  // last letter of alphabet must be default character like 'X'
  ERROR_IF(alphabet.size() == 0, 
	   "Undefined alphabet! Specify in parameter file.");
  if (checkAndRepairMode) {
    cout << "Starting check and repair alignment!" << endl;
    checkAndRepairAlignment(ali, alphabet, gapChar, gapChar);
    cout << "Ending check and repair alignment!" << endl;
    string errorMsg = checkAlignment(ali, alphabet, gapChar);
    cout << "Ending check alignment!" << endl;
    if (errorMsg.size() > 0) {
      cout << errorMsg << endl;
    }
    ERROR_IF(errorMsg.size() > 0, "Illegal sequence character found!");
  }

//   if ((clipAfter > 0) && (clipAfter < ali.size())) {
//     cout << "Clipping alignment after " << clipAfter << " sequences." << endl;
//     ali.cutTemplate(clipAfter);
//     cout << "Result size: " << ali.size() << endl;
//   }
  
//   if (pruneLength > 0) {
//     if (verboseLevel > 0) {
//       cout << "Pruning at position " << prunePos << " length: "
// 	   << pruneLength << endl;
//     }
//     ali.prune(prunePos, pruneLength);
//     aliChangedFlag = true;
//     if (verboseLevel > 0) {
//       cout << "Length after pruning: " << ali.getLength() << endl;
//     }
//   }

  // erase "middle" part of alignmnet
  // avoid using both prune and cleave at the same time, 
  // because the indices are not indipendent
//   if (cleaveLength > 0) {
//     if (verboseLevel > 0) {
//       cout << "Cleaving at position " << cleavePos << " length: "
// 	   << cleaveLength << endl;
//     }
//     ali.cleave(cleavePos, cleaveLength);
//     aliChangedFlag = true;
//     if (verboseLevel > 0) {
//       cout << "Length after cleaving: " << ali.getLength() << endl;
//     }
//   }
  // adjust sequences: (done earlier already)
  //   ali.upperCaseSequences();
  //   ali.replaceChar('T', 'U'); // replace DNA alphabet to RNA

//   if ((addLeft > 0) || (addRight > 0)) {
//     if (verboseLevel > 0) {
//       cout << "Adding gaps: " << addLeft << " " << addRight << endl;
//     }
//     ali.addAllGapEnds(addLeft, addRight);
//     aliChangedFlag = true;
//   }

//   if (randomizeLength > 0) {
//     ali.addAllGapEnds(randomizeLength, randomizeLength);
//     randomizeAlignment(ali, randomizeLength);
//     aliChangedFlag = true;
//     if (verboseLevel > 1) {
//       cout << "Alignment quality score after randomization: " 
// 	   << alignmentQualityScore(ali, alphabet,
// 				    optEntropyWeight, optMatchWeight,
// 				    optRatioWeight, scoreLimit) << endl;
//     }
//     cout << "Alignment MD score after randomization: " 
// 	 << computeMD(ali.getSequences(), sub, alphabet) << endl;
//   }  

  /******* editing of alignmnet finished, no more modifications of alignmnets after here
	   except optimization of alignment *****/

  // simple weighting scheme
//   switch (sequenceWeightMode) {
//   case 0:
//     ali.setWeights(Vec<double>(ali.size(), 1.0)); // uniform weights
//     break;
//   case 1:
//     ali.setWeights(computeSequenceWeights(ali));
//     break;
//   case 2:
//     if (verboseLevel > 0) {
//       cout << "Reading sequence weights from file: "
// 	   << sequenceWeightFileName << endl;
//     }
//     ali.setWeights(readSequenceWeights(sequenceWeightFileName));
//     break;
//   case 3: // inverse weights:
//     if (verboseLevel > 0) {
//       cout << "Reading sequence and inverting weights from file: "
// 	   << sequenceWeightFileName << endl;
//     }
//     ali.setWeights(vecInverse(readSequenceWeights(sequenceWeightFileName)));
//     break;
//   default: ERROR("Unknown sequence weighting mode!");
//   }
  Vec<double> aliWeights(ali.size(), 1.0); // all weights are set to one
  // generate consensus probability matrix:
  Vec<Vec<double> > consProbMatrix;
  if (probMatrixFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading probability matrix file: " << probMatrixFileName << endl;
    }
    ifstream probMatrixFile(probMatrixFileName.c_str());
    ERROR_IF(!probMatrixFile, "Error opening probability matrix file!");
    switch (probMatrixFormat) {
    case 1: 
      probMatrixFile >> consProbMatrix;
      break;
    case 2:
      consProbMatrix = readListMatrix(probMatrixFile, 0.0, true); // true: use mirror image too
      break;
    case 3:
      consProbMatrix = readPlainMatrix(probMatrixFile);
      break;
    default:
      ERROR("Unknown probability matrix file format!");
    }
    probMatrixFile.close();
  }
  else {
    if (verboseLevel > 0) {
      cout << "Calling generateProbabilityMatrix with " 
	   << probVectors.size() << " vectors and mode " 
	   << probConsensusMode << endl;
    }
    switch (probConsensusMode) {
    case 1: 
      consProbMatrix = generateProbabilityMatrix(probVectors, 
						 probNames, ali, gapChar, 
						 aliWeights);
      break;
    case 2: 
      consProbMatrix = generateProbabilityMatrix2(probVectors, 
						  probNames, ali, gapChar, 
						  aliWeights);
      break;
    case 3: 
      consProbMatrix = generateProbabilityMatrix3(probVectors, 
						  probNames, ali, gapChar, 
						  aliWeights);
      break;
    default:
      ERROR("Unknown probability consensus mode!");
    }
  }
  ERROR_IF(consProbMatrix.size() == 0,
	   "No probability matrix defined!");
  // renormalize probability matrix (currently using same renormalizations as for other matrices)
  // currently: no renormalization of probability matrix! Optional in future?
  // consProbMatrix = renormMatrix(consProbMatrix, renormMatrixMode, verboseLevel);
  // erase start and end part of alignmnet
  // finished cleaning of alignment
  // for each sequence of the alignment, get its probability id
  Vec<unsigned int> probIndices = generateAliNameIndices(ali, probNames, aliCharMax);
  if (verboseLevel > 1) {
    cout << "Cleaned alignment: " << endl;
    ali.writeFasta(cout);
    cout << "Starting quality score!" << endl;
    cout << "Alignment quality score: (threshold: " << scoreLimit 
	 << " ) : "  
	 << alignmentQualityScore(ali, alphabet, 
    		optEntropyWeight, optMatchWeight,
				  optRatioWeight, scoreLimit) << endl;
  }
  ERROR_IF(sub.size() == 0, "Substitution matrix undefined!");
  cout << "Alignment MD score: " 
       << computeMD(ali.getSequences(), sub, alphabet) << endl;
  double normMD = computeNormMD(ali.getSequences(), sub, alphabet, gOpen, gExt);  
  cout << "Alignment NormMD score: " << normMD << endl;
  if (stemFileName.size() > 0) {
    string refSeqTmp;
    Vec<Vec<double> > tmpMatrix;
    pkCounter = 0;
    if (verboseLevel > 0) {
      cout << "Reading reference stems from: " << stemFileName << endl;
    }
    ifstream stemFile(stemFileName.c_str());
    ERROR_IF(!stemFile, "Error opening stem file!");
    switch (stemFormat) {
    case 1:
      // region file
      referenceStems = readStems(stemFile);
      break;
    case 2:
      // bracket notation
      referenceStems = stemsFromBracketFasta(stemFile, pkCounter);
      break;
    case 3:
      tmpMatrix = readPlainMatrix(inputFile);
      // careful: does not work together with collapsing of alignment
      referenceStems = generateStemsFromMatrix(tmpMatrix, 1, refThresh, refSeqTmp);
      break;
    case 4: // read ct file
      readCtSingle(stemFile, referenceStems, refSeqTmp, 1);
      break;
    default:
      ERROR("Unknown stem file format!");
    }
    stemFile.close();
    ERROR_IF(referenceStems.size() == 0,
	     "No reference stems read!");
    if (verboseLevel > 0) {
      cout << "Read reference stems: " << endl;
      writeStems(cout, referenceStems);
    }
    if (pkCounter > 1) {
      cout << "Pseudo-knot encountered in reference secondary structure : "
	   << pkCounter << endl;
    }
  }

  if (forcedStemFileName.size() > 0) {
    pkCounter = 0;
    if (verboseLevel > 0) {
      cout << "Reading reference stems from: " << forcedStemFileName << endl;
    }
    ifstream forcedStemFile(forcedStemFileName.c_str());
    ERROR_IF(!forcedStemFile, "Error opening forced stem file!");
    switch (forcedStemFormat) {
    case 1:
      // region file
      forcedStems = readStems(forcedStemFile);
      break;
    case 2:
      // bracket notation
      forcedStems = stemsFromBracketFasta(forcedStemFile, pkCounter);
      break;
    default:
      ERROR("Unknown forced stem file format!");
    }
    if (forcedStems.size() == 0) {
      cout << "Warning: no forced stems defined!" << endl;
    }
    forcedStemFile.close();
    if (pkCounter > 1) {
      cout << "Pseudo-knot encountered in forced secondary structure : "
	   << pkCounter << endl;
    }
    if (verboseLevel > 0) {
      cout << "Read forced stems: " << endl;
      writeStems(cout, forcedStems);
    }
  }

  // remove all gaps of a reference sequence:
  if (collapseId >= 0) {
    if (verboseLevel > 0) {
      cout << "Collapsing gaps with respect to sequence "
	   << collapseId + 1 << " using algorithm " << collapseMode << endl;
    }
    switch (collapseMode) {
    case 0: 
      break; // do nothing
    case 1:collapseAlignment(ali, collapseId, consProbMatrix, referenceStems, 
			     forcedStems, gapChar);
      break;
    case 2:collapseAlignmentAllGaps(ali, collapseId, consProbMatrix, referenceStems, 
				    forcedStems, gapChar);	
      break;
    case 3:collapseAlignmentTooManyGaps(ali, collapseId, consProbMatrix, referenceStems, 
					forcedStems, numCharMin, gapChar);	
      break;
    case 4: collapseAlignmentTooManyGaps(ali, aliRef, collapseId, consProbMatrix, referenceStems, 
					forcedStems, numCharMin, gapChar);	
      break;
    case 5: collapseAlignmentGapOrTooManyGaps(ali, aliRef, collapseId, consProbMatrix, referenceStems, 
					forcedStems, numCharMin, gapChar);	
      break;
    default: ERROR("Undefined collapse algorithm id!");
    }
    aliChangedFlag = true;
    cout << "New length of alignment after collapsing: " << ali.getLength() << endl;
    if (verboseLevel > 0) {
      cout << "Adjusted reference stems: " << endl;
      writeStems(cout, referenceStems);
    }
  }
  if (stemOutFileName.size() > 0) {
    if (referenceStems.size() > 0) {
      if (verboseLevel > 0) {
	cout << "Writing " << referenceStems.size()
	     << " reference stems to: " << stemOutFileName << endl;
	if (referenceStems.size() == 0) {
	  cout << "Warning: no reference stems defined!" << endl;
	}
      }
      ofstream stemFile(stemOutFileName.c_str());
      ERROR_IF(!stemFile, "Error opening stem file for writing!");
      switch (stemOutFormat) {
      case 1:
	// region file
	writeStems(stemFile, referenceStems);
	break;
      case 2:
	// bracket notation
	stemFile << inputFileName << endl;
	stemFile << stemsToBracketFasta(referenceStems, ali.getLength());
	break;
      break;
      case 4: // write as contact matrix
	if (matrixNormalMode) {
	  writeMatrix(stemFile, generateMatrixFromStems(referenceStems,
							ali.getLength()));
	}
	else {
	  writeMatrixReverse(stemFile, generateMatrixFromStems(referenceStems,
							       ali.getLength()));
	}
	break;
      default:
	ERROR("Unknown stem file format!");
      }
      stemFile.close();
    }
    else {
      cout << "No output of reference stems because no reference stems defined!" << endl;
    }
  }

  if (aliOutFileName.size() > 0) {
    string oFileName = aliOutFileName;
//     if (overwriteMode == 0) {
//       oFileName = fileName(oFileName);
//     }
    ofstream outputFile(oFileName.c_str());
    ERROR_IF(!outputFile, "Error opening output file!");
    cout << "Writing alignment to file : " << oFileName << endl;
    switch (outputFormat) {
    case 1: ali.writeFasta(outputFile);
      break;
//     case 2: ali.saveClustal(outputFile);
//       break;
//     case 3: ali.saveMSAF(outputFile);
//       break;
//     case 4: ali.saveDSAF(outputFile);
//       break;
    default:
      ERROR("Unknown output format!");
    }
    outputFile.close();
  }

  if (probMatrixOutFileName.size() > 0) {
    if (verboseLevel > 0) {
	cout << "writing probability matrix to " << probMatrixOutFileName
	<< endl;
    }
    ofstream probMatrixOutFile(probMatrixOutFileName.c_str());
    ERROR_IF(!probMatrixOutFile, "Error opening probability output matrix file!");
    if (matrixNormalMode) {
      writeMatrix(probMatrixOutFile, consProbMatrix);
    }
    else {
      writeMatrixReverse(probMatrixOutFile, consProbMatrix);
    }
    probMatrixOutFile.close();
  }


  if (optimizeParameterFileName.size() > 0) {
    ifstream optimizeParameterFile(optimizeParameterFileName.c_str());
    ERROR_IF(!optimizeParameterFile, "Error opening optimize parameter file!");
    readOptimizeParameterFile(optimizeParameterFile,
			      optimizeStartA, optimizeStartB, optimizeLengths,
			      optimizeBorderLengths);
    optimizeParameterFile.close();
  }

  if (optimizeMode != 0) {
    cout << "Optimizing alignment with mode " << optimizeMode << endl;
    switch (optimizeMode) {
    case 0: // do nothing
      break;
    case 1:
      ali = optimizeAlignment(ali, alphabet, optRounds, optEntropyWeight, 
			      optMatchWeight, optRatioWeight, scoreLimit);
      break;
    case 2:
      ERROR_IF(optimizeParameterFileName.size() < 1,
	       "No optimization parameter filename defined! Use option --optimize-paramters");
      ali = optimizeAlignmentSubset(ali, secondaryAli, alphabet, optRounds, optEntropyWeight, 
				    optMatchWeight, optRatioWeight, scoreLimit,
				    optimizeStartA, optimizeStartB, 
				    optimizeLengths, optimizeBorderLengths);
      break;
    default:
      ERROR("Unknown optimization mode!");
    }
    
    if (verboseLevel) {
      cout << "Alignment quality score after optimization: " 
	   << alignmentQualityScore(ali, alphabet,
				    optEntropyWeight, optMatchWeight,
				    optRatioWeight, scoreLimit) << endl;
    }
    if (outputFileName.size() > 0) {
      string outputFileNameOpt = outputFileName + ".opt.fasta";
//       if (overwriteMode == 0) {
// 	outputFileNameOpt = fileName(outputFileNameOpt);
//       }
      ofstream outputFile(outputFileNameOpt.c_str());
      ERROR_IF(!outputFile, "Error opening output file!");
      cout << "Writing alignment to file : " << outputFileName << endl;
      switch (outputFormat) {
      case 1: ali.writeFasta(outputFile);
	break;
//       case 2: ali.saveClustal(outputFile);
// 	break;
      default:
	ERROR("Unknown output format!");
      }
      outputFile.close();
    }
    else {
      cout << "No filename for alignment output specified." << endl;
    }
  }
  
//   if (normMD < normMDLimit) {
//     cout << "Exiting because normMD was too small: " 
// 	 << normMD << " " << normMDLimit << endl;
//     exit(0);
//   }

  if (exitCode == 10) {
//     timer.stop();
//     cout << "Overall time needed: " << timer << endl;
    cout << "Exiting program before prediction because of exit code "
	 << exitCode << endl;
    exit(0);
  }

  Vec<Vec<double> > compMatrix;
  bool compMatrixOk = false; // this variable tracks, if compMatrix contains mutualInformation data
  if ((forceMatrixFileName.size() == 0)
      && (forcedStemFileName.size() == 0)) {
    cout << "Compute compensatory score with algorithm " << algorithm
	 << endl;
    scorer.algorithm = algorithm;
    compMatrix = generateCompensatoryMatrix(ali, scorer, algorithm,
					    renormMatrixMode, neuralPatchSize, 2 * neuralPatchSize, verboseLevel);
    compMatrixOk = true;
    // not clean, should be part of generateCompensatoryMatrix:
    if (gaussDist > 0.0) {
      compMatrix = smoothAntiDiagGauss(compMatrix, gaussDist);
    }
  }
  else if (forceMatrixFileName.size() > 0) {
    ifstream forceMatrixFile(forceMatrixFileName.c_str());
    ERROR_IF(!forceMatrixFile, "Error opening forced matrix file!");
    compMatrix = readMatrixFormat(forceMatrixFile, matrixFormat);
//     switch (matrixFormat) {
//     case 1: 
//       forceMatrixFile >> compMatrix;
//       break;
//     case 2:
//       compMatrix = readListMatrix(forceMatrixFile, 0.0, true); // true: use mirror image too
//       break;
//     case 3:
//       compMatrix = readPlainMatrix(forceMatrixFile);
//       break;
//     default:
//       ERROR("Unknown force matrix file format!");
//     }
    compMatrixOk = false;
    if (verboseLevel > 0) {
      cout << "Prediction matrix read from: " << forceMatrixFileName << "!" << endl;
      if (verboseLevel > 1) {
	cout << "result: " << endl << compMatrix << endl;
      }
    }
  }
  else if (entropyMatrixReadFileName.size() > 0) {
    ifstream entropyReadFile(entropyMatrixReadFileName.c_str());
    ERROR_IF(!entropyReadFile, "Error opening entropy read file!");
    compMatrix = readMatrixFormat(entropyReadFile, matrixFormat);
    entropyReadFile.close();
    compMatrixOk = true;
  }
  else {
    if (verboseLevel > 0) {
      cout << "Initializing prediction matrix with size : " << ali.getLength() << endl;
    }
    compMatrix = Vec<Vec<double> >(ali.getLength(), Vec<double>(ali.getLength(), 0.0));
    compMatrixOk = false;
  }
  /////////// possible rescale compMatrix (mutual information matrix) with emperical factor:
  if (entropyRescaleAlpha > 0.0) {
    entropyRescale(compMatrix, ali, entropyRescaleAlpha);
  }

  Vec<Vec<double> > referenceMatrix(compMatrix.size(),
				    Vec<double>(compMatrix.size(), 0.0));
  if ((referenceStems.size() > 0)) {
    cout << "Addding information about reference stems to reference matrix with weight " << 1 << endl;
    addStemsToMatrix(referenceMatrix, referenceStems, 1.0);
    // cout << "Reference matrix: " << referenceMatrix << endl;
  }

  if ((stemWeight != 0.0) && (referenceStems.size() > 0)) {
    cout << "Addding information about reference stems to matrix with weight " << stemWeight << endl;
    addStemsToMatrix(compMatrix, referenceStems, stemWeight);
  }
  if (forcedStemFileName.size() > 0) {
    compMatrixOk = false;
    if (forcedWeight < 0.0) {
      cout << "Addding information about forced stems to *cleared* matrix with weight " 
	   << -forcedWeight << endl;
      for (unsigned int ii = 0; ii < compMatrix.size();++ii) {
	for (unsigned int jj = 0; jj < compMatrix[ii].size();++jj) {
	  compMatrix[ii][jj]=0.0;
	}
      }
      addStemsToMatrix(compMatrix, forcedStems, -forcedWeight);
    }
    else {
      cout << "Addding information about forced stems to matrix with weight " 
	   << forcedWeight << endl;
      addStemsToMatrix(compMatrix, forcedStems, forcedWeight);
    }
  }

  if (treeStatFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Writing mutation statistics to file: "
	   << treeStatFileName << endl;
    }
    string contactFileName = treeStatFileName + ".contact";
    string noContactFileName = treeStatFileName + ".nocontact";
    ofstream contactFile(contactFileName.c_str());
    ofstream noContactFile(noContactFileName.c_str());
    unsigned int nn = scorer.getAlphabet().size();
    nn *= nn; // square
    Vec<Vec<double> > contactMatrix(nn, Vec<double>(nn, 0.0));
    Vec<Vec<double> > noContactMatrix(nn, Vec<double>(nn, 0.0));
    generateMutationMatrices(ali, referenceMatrix, scorer.getAlphabet(), 
			     contactMatrix, noContactMatrix);
    writeMatrix(contactFile, contactMatrix);
    writeMatrix(noContactFile, noContactMatrix);
    contactFile.close();
    noContactFile.close();
  }

  // if wanted, compute position specific optimal subsets:
  if (subsetOptimizePositions.size() > 0) {
    cout << "Finding optimized subsets with respect to the following positions: " << endl;
    cout << externalCounting(subsetOptimizePositions);
    findOptimalSubset(ali, subsetOptimizePositions,
		      aliWeights, scorer, subsetOptimizeTrials);
  }


  /****** OUTPUT ********/

  if (probVectors.size() > 0) {
    double mfoldCorr = probabilityCorrelation(ali, compMatrix, probVectors,
				       probIndices, corrOutName);
    cout << "Correlation with secondary structure prediction: "
	 << mfoldCorr << endl;
  }
  if (!compMatrixOk) {
    cout << "Warning: the following values of mutual information have been read in , they might not be meaningful..." << endl;
  }
  cout << "Matrix entropy: " << matrixEntropy(compMatrix, -1.0, 0.05, 50) 
       << endl;
//   cout << "Running bimodal separation: " << 
//     bimodalSeparation(matrix2Vec(compMatrix), optRatioWeight) << endl;
  if (verboseLevel > 1) {
    cout << "average diagonal projection of matrix: " 
	 << endl << getDiagonalAverages(compMatrix) << endl;
    cout << "anti-diagonal projection of matrix: " 
	 << endl << getAntiDiagonalAverages(compMatrix) << endl;
  }
  cout << "Skewness: " << skewness(compMatrix, scoreLimit) << endl;
  cout << "BestSkewness: " << bestSkewness(compMatrix) << endl;
  // find highest value:
  unsigned int maxITmp, maxJTmp;
  findMaxIndices(compMatrix, maxITmp, maxJTmp);
  cout << "Maximum value of compMatrix: " << maxITmp << " "
       << maxJTmp << " " << compMatrix[maxITmp][maxJTmp] << endl;
  if (verboseLevel > 1) {
    cout << "Alignment quality score: " 
	 << alignmentQualityScore(ali, alphabet,
				  optEntropyWeight, optMatchWeight,
				  optRatioWeight, scoreLimit) << endl;
  }
  if (neuralHeadFileName.size() > 0) {
    ofstream neuralHeadFile(neuralHeadFileName.c_str());
    ERROR_IF(!neuralHeadFile, "Error opening neural matrix header file!");
    neuralHeadFile << compMatrix.size() << " " << compMatrix.size() << endl;
    neuralHeadFile.close();
  }
  Vec<double> globalFeatures = computeGlobalFeatures(ali);
  Vec<Vec<Vec<double> > > compMatrices(1);
  CompensationScorer scorer2 = scorer;
  scorer2.energyWeight = 1.0;
  scorer2.entropyWeight = 0.0; // energy only
  scorer2.algorithm = 3;
  if (compMatrixOk) {
    compMatrices[0] = compMatrix; 
  }
  else { // compMatrix does not contain original mutual information data, so recompute:
    scorer.algorithm = algorithm;
    compMatrices[0] = generateCompensatoryMatrix(ali, scorer, scorer.algorithm,
				 renormMatrixMode, neuralPatchSize, 2 * neuralPatchSize, verboseLevel);
  }
  if (entropyMatrixOutFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Writing entropy matrix to file: " << entropyMatrixOutFileName << endl;
    }
    ofstream entropyOutFile(entropyMatrixOutFileName.c_str());
    ERROR_IF(!entropyOutFile, "Error opening entropy matrix output file!");
    if (matrixNormalMode) {
      writeMatrix(entropyOutFile, compMatrix);
    }
    else {
      writeMatrixReverse(entropyOutFile, compMatrix);
    }
    entropyOutFile.close();
  }
  if (exitCode == 12) {
    // timer.stop();
    // cout << "Overall time needed: " << timer << endl;
    cout << "Quitting program because of exitcode 12\n";
    exit(0);
  }
  if (neuralProbabilityMatrixMode) {
    if (verboseLevel > 0) {
      cout << "Adding probability matrix to neural network data!" << endl;
    }
    compMatrices.push_back(consProbMatrix); // consensus probability matrix
  }
  if (neuralEnergyMatrixMode) {
    if (verboseLevel > 0) {
      cout << "Adding matching matrix to neural network data!" << endl;
    }
    compMatrices.push_back(generateCompensatoryMatrix(ali, scorer2,
						      scorer2.algorithm, renormMatrixMode, neuralPatchSize, 
						      2 * neuralPatchSize, verboseLevel));
  }
  if (neuralConservationMatrixMode) {
    if (verboseLevel > 0) {
      cout << "Adding matching matrix to neural network data!" << endl;
    }
    double informationMean = 0.0;
    double informationDev = 1.0;
    computeInformationZScore(ali, scorer, informationMean, informationDev, aliWeights);
    scorer2.userMean = informationMean;
    scorer2.userDev = informationDev;
    scorer2.algorithm = 21;
    compMatrices.push_back(generateCompensatoryMatrix(ali, scorer2,
						      scorer2.algorithm, renormMatrixMode, neuralPatchSize, 
						      2 * neuralPatchSize, verboseLevel));
    scorer2.algorithm = 22;
    compMatrices.push_back(generateCompensatoryMatrix(ali, scorer2,
						      scorer2.algorithm, renormMatrixMode, neuralPatchSize, 
						      2 * neuralPatchSize, verboseLevel));
  }
  if (neuralAliMatrixMode == 0) {
    if (verboseLevel > 0) {
      cout << "No alignment fingerprint computed!" << endl;
    }
  }
  if (neuralAliMatrixMode == 1) {
    Vec<Vec<Vec<double> > > aliMatrices = computeAliMatrices(ali, scorer, gapChar);
    if (verboseLevel > 0) {
      cout << "Generated " << aliMatrices.size() << " dot-matrices with nucleotide frequencies!" << endl;
    }
    for (unsigned int i = 0; i < aliMatrices.size(); ++i) {
      compMatrices.push_back(aliMatrices[i]);
    }
  }
  else if (neuralAliMatrixMode == 2) {
    // use also pseudo counts 
    Vec<Vec<Vec<double> > > aliMatrices = computeAliMatrices2(ali, scorer, gapChar);
    if (verboseLevel > 0) {
      cout << "Generated " << aliMatrices.size() << " dot-matrices with nucleotide frequencies!" << endl;
    }
    for (unsigned int i = 0; i < aliMatrices.size(); ++i) {
      compMatrices.push_back(aliMatrices[i]);
    }
  }
  else {
    ERROR("Unknown alignment fingerprint mode!");
  }
  if (neuralRenameMode != 0) {
    scorer2.algorithm = 23;
    compMatrices.push_back(generateCompensatoryMatrix(ali, scorer2,
						      scorer2.algorithm, renormMatrixMode, neuralPatchSize, 
						      2 * neuralPatchSize, verboseLevel));
  }
  
  if (verboseLevel > 0) {
    cout << "Generated overall " << compMatrices.size()
	 << " matrices!" << endl;
  }
  if (knnNet.isValid()) {
    if (neuralSelfMode != 0) {
      knnNet.setNoSelfMode(true); // avoid problems with training vectors used several times: ignore closest hit always, assume that is occured because of overalp between training and testing
    }
    if (knnOptimizeId >= 0) {
      ERROR_IF(knnOptimizeId >= static_cast<int>(knnNet.size()),
	       "Knn optimize id too large!");
      Vec<double> optimizedScaling = knnNet.optimizeScaling(
			     static_cast<unsigned int>(knnOptimizeId), 
			     knnOptimizeSteps,
			     verboseLevel, knnOptimizeStepWidth, 
			     knnOptimizeTrials);
      cout << "Result of optimized scaling of knn node " << knnOptimizeId 
	   << " is : " << optimizedScaling << endl;
    }
    else if (knnOptimizeId == -1) { // user supplied "0":
      if (verboseLevel > 0) {
	cout << "Optimize scaling of all knn nodes!" << endl;
      }
      knnNet.optimizeScaling(knnOptimizeSteps,
			     verboseLevel, knnOptimizeStepWidth, 
			     knnOptimizeTrials);
    }
    if (knnSaveFileName.size() > 0) {
      if (verboseLevel > 0) {
	cout << "Saving KnnNet definition to file: "
	     << knnSaveFileName << endl;
      }
      ofstream knnSaveFile(knnSaveFileName.c_str());
      ERROR_IF(!knnSaveFile, "Error opening knn save file for output!");
      knnSaveFile << knnNet;
      knnSaveFile.close();
    }
    if (verboseLevel > 0) {
      cout << "Running knn net as defined in : " << knnFileName << endl;
    }
    Vec<Vec<int> > filteredMatrix(ali.getLength(), Vec<int>(ali.getLength(), 1)); // all entries set to "1", no filtering out yet
    if (matchThreshold > 0.0) {
      filterMatchThreshold(filteredMatrix, ali, matchThreshold, scorer);
    }
    if (knnWriteFileName.size() > 0) {
      knnNet.startStatistics();
      ERROR_IF(knnTrainFileName.size() == 0,
	       "Training data file must be specified with option --knn-train-file");
      if (verboseLevel > 0) {
	cout << "Reading training data from file: " << knnTrainFileName << endl;
      }
      ifstream knnTrainFile(knnTrainFileName.c_str());
      ERROR_IF(!knnTrainFile, "Error opening knn training data file!");
      KnnNetTrainer knnNetTrainer;
      knnNetTrainer.readData(knnTrainFile);
      ERROR_IF(knnNetTrainer.size()==0, 
	       "Error reading knn training data file!");
      knnTrainFile.close();
      if (verboseLevel > 0) {
	cout << "Running knn net in training mode as defined in : " 
	     << knnFileName << " " << knnTrainFileName
	     << " writing training file to: " << knnWriteFileName << endl;
      }
      ERROR_IF(knnWriteLevel < 0, "use option --knn-write-level to specify after which level to write");
      ofstream knnWriteFile(knnWriteFileName.c_str());
      ERROR_IF(!knnWriteFile, "Error opening knn training file!");
      knnNet.setWriteLevel(knnWriteLevel);
      knnNetTrainer.writeLevelTrainVectors(knnWriteFile, knnTrainMax, knnNet);
      //       compMatrix = computeKnnTrainPrediction(knnWriteFile,compMatrices, globalFeatures, knnNet,
      //    neuralPatchSize, neuralFileMode,
      //    referenceMatrix, scoreLimit, verboseLevel);
      knnWriteFile.close();
      cout << "Usage histogram of Knn net: " << endl;
      cout << knnNet.getUsageHistogram(1);
      if (exitCode == 15) {
	// timer.stop();
	// cout << "Overall time needed: " << timer << endl;
	cout << "Quitting program because of exitcode 15\n";
	exit(0);
      }
    }
    if (knnStartSubset.size() > 0) {
      cout << "Starting to compute knn prediction for start position subset: " 
	   << externalCounting(knnStartSubset) << endl;
    }
    if (knnStopSubset.size() > 0) {
      cout << "Starting to compute knn prediction for stop position subset: " 
	   << externalCounting(knnStopSubset) << endl;
    }
    if ((knnStartSubset.size() == 0) && (knnStopSubset.size() == 0)) {
      cout << "Starting to compute knn prediction for complete sequence!" 
	   << endl;
    }
    compMatrix = computeKnnPrediction (compMatrices, 
				       filteredMatrix, globalFeatures, knnNet,
				       knnBorderVec,
				       neuralPatchSize, neuralFileMode,
				       referenceMatrix, scoreLimit, 
				       knnStartSubset, knnStopSubset, dMax, dMin, sumMax, sumMin,
				       verboseLevel);
    // compMatrices[0] = compMatrix; // not clean programming
  }
  if (growSimpleLimit > 0) {
    if (verboseLevel > 0) {
      cout << "Simple growth with parameters: " 
	   << growSimpleLimit << " " << growSimpleMatchLimit << endl;
    }
    // int numChanged = simpleGrow(compMatrix, ali, growSimpleLimit, growSimpleMatchLimit, scorer);
    int numChanged = simpleGrow2(compMatrix, ali, growSimpleLimit, refSeqId, scorer);
    cout << "Number of changed matrix elements due to simple growth procedure: " << numChanged << endl;
  }
  if ((knnNet2.size() > 0) && (knnRounds2 > 0)) {
    if (verboseLevel > 0) {
      cout << "Apply postprocessing knn method: " 
	   << " using knn classifier " << knnRounds2 << " rounds." << endl;
    }
    if (outputFileName.size() > 0) {
      string tmpMatrixFileName = outputFileName + ".initial.matrix";
      ofstream tmpMatrixFile(tmpMatrixFileName.c_str());
      ERROR_IF(!tmpMatrixFile, "Error opening initial matrix file!");
      writeMatrix(tmpMatrixFile, compMatrix);
      tmpMatrixFile.close();
    }
    growMatrix(compMatrix, knnNet2, compMatrices, 
	       knnStartSubset, knnStopSubset, 
	       dMax, dMin, sumMax, sumMin, 0, knnRounds2); // "0": no border
  }
  //////////////// possible filter /////////////////////
  if (filterImpossibleMode) {
    if (verboseLevel > 0) {
      cout << "Filtering out \"impossible\" contacts" << endl;
    }
    filterImpossible(compMatrix, ali, scorer);
  }
  
  // rescale using length:
  if (lengthScaleWeight != 0.0) {
    if (verboseLevel > 0) {
      cout << "Rescaling with length dependent exponent!" << endl;
    }
    rescaleWithLength(compMatrix, lengthScaleWeight, lengthScaleMean,
		      ali.getLength());
  }
  // rescale using length:
  if (distanceScaleWeight != 0.0) {
    if (verboseLevel > 0) {
      cout << "Rescaling with length dependent exponent!" << endl;
    }
    rescaleWithDistance(compMatrix, distanceScaleWeight, distanceScaleMean);
  }
  // optional rescaling using external data:
  if (rescaleFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading rescale file!" << endl;
    }
    ifstream rescaleFile(rescaleFileName.c_str());
    if (!rescaleFile) {
      cout << "Error opening rescale file!" << endl;
    }
    else {
      rescaleData = readRescaleData(rescaleFile);
      rescaleFile.close();
      if (rescaleData.size() > 0) {
	if (verboseLevel > 0) {
	  cout << "Rescaling result matrix!" << endl;
	}      
	rescaleMatrix(compMatrix, rescaleData);
      }
      compMatrixOk = false;
    }
  }

  if (numCharMinFilter > 0) {
    if (verboseLevel > 0) {
      cout << "Applyting charMinFilter: setting all predictions with to zero corresponding to alignment columns with less than " 
	   << numCharMinFilter << " non-gap characters." << endl;
    }
    charMinFilter(compMatrix, ali, numCharMinFilter);
  }
  // multiply with pow(f, alpha), with f the fraction of gaps at a certain column pair
  if (gapFracRescaleMode) {
    if (verboseLevel > 0) {
      cout << "Calling gapFracRescale!" << endl;
    }
    gapFracRescale(compMatrix, ali, gapFracAlpha);
  }
  
  if (badColumnMax > 1) {
    Vec<unsigned int> filteredColumns = filterBadColumns(compMatrix, scoreLimit, badColumnMax);
    cout << "Filtered out bad columns: " << externalCounting(filteredColumns) << endl;
    double badFrac = filteredColumns.size() / static_cast<double>(ali.getLength());
    double badFracMax = 0.2; // 20 %
    if (badFrac > badFracMax) {
      cout << "Warning: more than " << 100.0*badFrac << " % bad (" << badColumnMax << ") matrix columns encountered. Consider higher threshold." << endl;
    }
  }

  // optional winner-takes-all operation:
  if (winnerMode) {
    if (verboseLevel > 0) {
      cout << "Applying winner-takes-all filter to matrix!" << endl;
    }
    int diagBorder = neuralPatchSize;
    winnerTakesAll(compMatrix, diagBorder, winnerCutoff);
  }

  if (pStatDev > 0.0) {
    if (verboseLevel > 0) {
      cout << "Converting scores to p-values!" << endl;
    }
    convertToPValues(compMatrix, pStatAvg, pStatDev);
  }

  if (neuralFileName.size() > 0) {
    ofstream neuralFile(neuralFileName.c_str());
    ERROR_IF(!neuralFile, "Error opening neural file!");
    if (verboseLevel > 0) {
      cout << "Writing feature vectors to file: "
	   << neuralFileName << endl;
    }
    writeNeuralFile(neuralFile, compMatrices, referenceMatrix, 
		    globalFeatures, neuralPatchSize, neuralMaxLines,
		    neuralFileMode, neuralAllMode, neuralBothSideMode,
		    neuralDistBias);

    neuralFile.close();
  }
  if (referenceStems.size() > 0) {
    double step = 0.025; 
    unsigned int numLim = 41;
    // double step = 0.1 * scoreLimit;
    unsigned int borderLimit = neuralPatchSize;
    if (neuralAllMode) {
      borderLimit = 0;
    }
    unsigned int diagLimit = 2 * neuralPatchSize;
    double bestAccuracy2 = 0.0;
    double bestCoverage = 0.0;
    double bestMathews = -1e10;
    double bestWrongPosFrac = 1.0;
    double bestLimit = 0.5;
    unsigned int bestWrongPos = 0;
    Vec<Vec<double> > compMatrixWork = compMatrix;
    for (unsigned int i = 1; i <= numLim; ++i) {
      double locLimit = i * step;
      double accuracy = 0.0;
      double accuracy2 = 0.0;
      double coverage = 0.0;
      double mathews = 0.0;
      if (minStemLength > 1) {
	string emptyString = "";
	for (unsigned int ii = 0; ii < compMatrixWork.size(); ++ii) {
	  for (unsigned int jj = 0; jj < compMatrixWork.size(); ++jj) {
	    compMatrixWork[ii][jj] = 0.0;
	  }
	}
	Vec<Stem> tmpStems = generateStemsFromMatrix(compMatrix, minStemLength, 
						     locLimit, emptyString);
	addStemsToMatrix(compMatrixWork, tmpStems, 1.0);
	if (locLimit < minStemAnyways) {
	  for (unsigned int i2 = 0; i2 < compMatrix.size(); ++i2) {
	    for (unsigned int j2 = 0; j2 < compMatrix.size(); ++j2) {
	      if (compMatrix[i2][j2] >= minStemAnyways) {
		compMatrixWork[i2][j2] = 1.0;
	      }
	    }
	  }
	}
      }
      if (refSeqId >= 0) {
	mathews = computeMatrixMathews2(compMatrixWork, referenceStems, 
		       ali.getSequence(static_cast<unsigned int>(refSeqId)),
		       locLimit, accuracy, accuracy2,
				 coverage, wrongPosFrac, wrongPos, 
				0, diagLimit, gapChar); // instead of "borderLimit" use 0 border size
	// 	mathews = computeMatrixMathews(compMatrix, referenceStems, 
	// 		       ali.getSequence(static_cast<unsigned int>(refSeqId)),
	// 		       locLimit, accuracy, coverage, wrongPosFrac, wrongPos, 
	// 		       0, diagLimit, gapChar); // instead of "borderLimit" use 0 border size
      }
      else {
	mathews = computeMatrixMathews2(compMatrixWork, referenceStems,
				locLimit, accuracy, accuracy2, coverage, wrongPosFrac, wrongPos, 
				0, diagLimit); // instead of "borderLimit" use 0 border size
	// 	mathews = computeMatrixMathews(compMatrix, referenceStems,
	// 		       locLimit, accuracy, coverage, wrongPosFrac, wrongPos, 
	// 		       0, diagLimit); // instead of "borderLimit" use 0 border size
      }
      if (mathews > bestMathews) {
	bestMathews = mathews;
	bestWrongPosFrac = wrongPosFrac;
	bestWrongPos = wrongPos;
	bestAccuracy2 = accuracy2;
	bestCoverage = coverage;
	bestLimit = locLimit;
      }
      cout << "Mathews coeefficient, accuracy and coverage of matrix with " << locLimit << " " 
	   << 100.0 * mathews << " " << 100.0 * accuracy << " " << 100.0 * coverage << " " 
	   << wrongPos << " " << 100.0 * wrongPosFrac << " " << 100.0 * accuracy2 << endl;
    }
    cout << "Best Mathews coefficient at cutoff: " << bestLimit << " " 
	 << 100.0 * bestMathews << " " << bestWrongPos << " " 
	 << 100.0 * bestWrongPosFrac << " " << 100.0 * bestAccuracy2 << " " 
	 << 100.0 * bestCoverage << endl;
  }
  cout << "# Predicted contacts: stop start confidence size entropy sec match knn\n" << endl;

  //   if (knnNet.isValid()) {
  //     writeList(cout, compMatrix, compMatrices, scoreLimit, 
  // 	      knnNet, globalFeatures, neuralPatchSize, neuralFileMode, 0);
  //   }
  //   else {
  if (verboseLevel > 1) {
    cout << "Starting writeList!" << endl;
    writeList(cout, compMatrix, scoreLimit);
    cout << "Ending writeList!" << endl;
  }

  if (listFileName.size() > 0) {
    ofstream listFile(listFileName.c_str());
    ERROR_IF(!listFile, "Error opening list file!");
    listFile << "# Predicted contacts: stop start confidence size entropy sec match knn\n" << endl;
    if (knnNet.isValid()) {
      writeList(listFile, compMatrix, compMatrices, 0.01,
		knnNet, globalFeatures, neuralPatchSize, neuralFileMode, 0);
    }
    else {
      writeList(listFile, compMatrix, 0.1);
    }
    listFile.close();
  }
  string emptyString; // no single sequence specified
  Vec<Stem> resultStemsHigh = generateStemsFromMatrix(compMatrix, minStemLength, 
						      scoreLimitHigh, emptyString);
  Vec<Stem> resultStemsMediumHigh = generateStemsFromMatrix(compMatrix, 
	    minStemLength, scoreLimitMediumHigh, emptyString);
  Vec<Stem> resultStemsMedium = generateStemsFromMatrix(compMatrix, minStemLength, 
							scoreLimit, emptyString);
  Vec<Stem> resultStemsLow = generateStemsFromMatrix(compMatrix, minStemLength, 
						     individualThreshold, emptyString);
  cout << "Predicted stems (positions with respect to multiple sequence alignment):" << endl;
  Vec<unsigned int> reportFeatureIndices(1, 0U); // mutual information
  reportFeatureIndices.push_back(1); // RNAfold
  reportFeatureIndices.push_back(compMatrices.size()-1); // matching score
  if (referenceStems.size() > 0) {
    cout << "Reporting reference stems and matrices with indices : " 
	 << externalCounting(reportFeatureIndices);
    writeStems(cout, referenceStems, compMatrices, reportFeatureIndices );
  }
  else {
    cout << "No reference stems defined!" << endl;
  }
  cout << "Reporting predicted stems (high confidence "
       << scoreLimitHigh << " ) and matrices with indices : " 
       << externalCounting(reportFeatureIndices);
  writeStems(cout, resultStemsHigh, compMatrices, reportFeatureIndices,
	     indexOffset);
  cout << "Reporting predicted stems (medium high confidence " 
       << scoreLimitMediumHigh << " ) and matrices with indices : " 
       << externalCounting(reportFeatureIndices);
  writeStems(cout, resultStemsHigh, compMatrices, reportFeatureIndices,
	     indexOffset);
  cout << "Reporting predicted stems (medium confidence "
       << scoreLimit << " ) and matrices with indices : " 
       << externalCounting(reportFeatureIndices);
  writeStems(cout, resultStemsMedium, compMatrices, reportFeatureIndices,
	     indexOffset);
  cout << "Reporting predicted stems (low confidence "
       << individualThreshold << " ) and matrices with indices : " 
       << externalCounting(reportFeatureIndices);
  writeStems(cout, resultStemsLow, compMatrices, reportFeatureIndices,
	     indexOffset);
  if (outputFileName.size() > 0) {
    string predictionOutFileName = outputFileName + ".predict.reg";
//     if (overwriteMode == 0) {
//       predictionOutFileName = fileName(predictionOutFileName);
//     }
    if (verboseLevel > 0) {
      cout << "Writing predicted stems to: " << predictionOutFileName << endl;
    }
    ofstream predictionOutFile(predictionOutFileName.c_str());
    ERROR_IF(!predictionOutFile, "Error opening prediction output file!");
    writeStems(predictionOutFile, resultStemsMedium, indexOffset);
    predictionOutFile.close();

    predictionOutFileName = outputFileName + ".predicthigh.reg";
//     if (overwriteMode == 0) {
//       predictionOutFileName = fileName(predictionOutFileName);
//     }
    if (verboseLevel > 0) {
      cout << "Writing predicted stems to: " << predictionOutFileName << endl;
    }
    ofstream predictionOutFileHigh(predictionOutFileName.c_str());
    ERROR_IF(!predictionOutFileHigh, "Error opening prediction output file!");
    writeStems(predictionOutFileHigh, resultStemsHigh, indexOffset);
    predictionOutFileHigh.close();

    predictionOutFileName = outputFileName + ".predictmediumhigh.reg";
//     if (overwriteMode == 0) {
//       predictionOutFileName = fileName(predictionOutFileName);
//     }
    if (verboseLevel > 0) {
      cout << "Writing predicted stems to: " << predictionOutFileName << endl;
    }
    ofstream predictionOutFileMediumHigh(predictionOutFileName.c_str());
    ERROR_IF(!predictionOutFileMediumHigh, 
	     "Error opening prediction output file!");
    writeStems(predictionOutFileMediumHigh, resultStemsMediumHigh, indexOffset);
    predictionOutFileHigh.close();

    predictionOutFileName = outputFileName + ".predictlow.reg";
//     if (overwriteMode == 0) {
//       predictionOutFileName = fileName(predictionOutFileName);
//     }
    if (verboseLevel > 0) {
      cout << "Writing predicted stems to: " << predictionOutFileName << endl;
    }
    ofstream predictionOutFileLow(predictionOutFileName.c_str());
    ERROR_IF(!predictionOutFileLow, "Error opening prediction output file!");
    writeStems(predictionOutFileLow, resultStemsLow, indexOffset);
    predictionOutFileLow.close();
  }
  if (refSeqId >= 0) {
    cout << "Predicted stems (positions with respect to sequence of alignment (ignoring --index-offset): "
	 << refSeqId + 1 << " " << ali.getName(static_cast<unsigned int>(refSeqId)) << endl;
    writeStems(cout, resultStemsMedium, ali.getSequence(refSeqId));
    cout << endl;
  }
  // this sum of all prediction confidences can be a measure of 
  // the quality of the alignment:
  double matrixSum = elementSum(compMatrix);
  cout << "Element sum of matrix of predicted contacts: " 
       << matrixSum << endl;
  cout << "Element sum of matrix of predicted contacts divided by length: " 
       << matrixSum / compMatrix.size() << endl;
  cout << "Element sum of matrix of predicted contacts divided by length squared: " 
       << matrixSum / (compMatrix.size()*compMatrix.size()) << endl;
  cout << "Contact prediction matrix quality score: " 
       << contactMatrixQuality(compMatrix, 3) << endl; // leave this loop size
  if (constraintFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Writing constraint alignment to file: " << constraintFileName << endl;
    }
    ofstream constraintFile(constraintFileName.c_str());
    ERROR_IF(!constraintFile, "Error opening constraint file!");
    writeConstraintAlignment(constraintFile, compMatrix, ali, constraintCutoff,
  	     constraintConflictCutoff, constraintGapMode); // false: no gaps
    constraintFile.close();
  }

  // rerank read stems and energies:
  
  if (scoreStemFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Writing stem superset to file " << scoreStemFileName << endl;
    }
    ofstream scoreStemFile(scoreStemFileName.c_str());
    ERROR_IF(!scoreStemFile, "Error opening score stem file!");
    entropyStemList(scoreStemFile, compMatrix, ali.getSequence(0), 
		    scorer.getAllowedPairs(),
		    scoreStemFileThreshold, minStemLength,
		    scoreStemFileWeight, predictionConfidence);
    scoreStemFile.close();
  }

  if (loopFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Writing loop file to file " << loopFileName << endl;
    }
    ofstream loopFile(loopFileName.c_str());
    ERROR_IF(!loopFile, "Error opening loop file!");
    writeLoopFile(loopFile, compMatrix, compMatrices, referenceMatrix);
    loopFile.close();
  }

  // reportCompensatoryScores(cout, compMatrix, ali, scoreLimit, verboseLevel);
  if (outputFileName.size() > 0) {
    string oFileName = outputFileName + ".matrix";
//     if (overwriteMode == 0) {
//       oFileName = fileName(oFileName);
//     }
    ofstream matrixOutFile(oFileName.c_str());
    ERROR_IF(!matrixOutFile, "Error opening matrix output file!");
    if (verboseLevel > 0) {
      cout << "Writing matrix to file " << oFileName << endl;
    }
    if (matrixNormalMode) {
      writeMatrix(matrixOutFile, compMatrix);
    }
    else {
      writeMatrixReverse(matrixOutFile, compMatrix);
    }
    matrixOutFile.close();
  }

  if ((matrixOutIds.size() > 0) && (outputFileName.size() > 0)) {
    for (unsigned int i = 0; i < matrixOutIds.size(); ++i) {
      if (matrixOutIds[i] >= compMatrices.size()) {
	cout << "Warning: internal matrix with id " << matrixOutIds[i] << " is not defined, only "
	     << compMatrices.size() << " matrices are defined." << endl;
	continue;
      }
      string oFileName = outputFileName + "." + uitos(matrixOutIds[i] + 1) + ".matrix";
      ofstream matrixOutFile(oFileName.c_str());
      ERROR_IF(!matrixOutFile, "Error opening matrix output file!");
      if (verboseLevel > 0) {
	cout << "Writing matrix with id " << matrixOutIds[i] << " to file " << oFileName << endl;
      }
      if (matrixNormalMode) {
	writeMatrix(matrixOutFile, compMatrices[matrixOutIds[i]]);
      }
      else {
	writeMatrixReverse(matrixOutFile, compMatrices[matrixOutIds[i]]);
      }
      matrixOutFile.close();
    }
  }

  if (individualMode != 0) {
    string individualFileName = outputFileName + ".individual.reg";
    if (verboseLevel > 0) {
      cout << "Writing individualized structures to file: " 
	   << individualFileName << endl;
    }
    ofstream oFile(individualFileName.c_str());
    ERROR_IF(!oFile, 
	     "Error opening output file for individualized predictions!");
    Vec<Vec<double> > individualConsMatrix = writeIndividualStructures(oFile, compMatrix, scorer, ali, individualThreshold, stemOutFormat, minStemLength);
    oFile.close();
    Vec<Stem> individualConsStems = generateStemsFromMatrix(individualConsMatrix, minStemLength, 0.5, string(""));							    
    cout << "Consensus individualized stems: " << endl;
    writeStems(cout, individualConsStems, indexOffset);
    cout << endl;
  }
  
//   timer.stop();
//   cout << "Overall time needed: " << timer << endl;
  cout << "Good bye!" << endl;

  return 0;
}
  
