#ifndef _BED_
#define _BED_

#include <string>
#include <Vec.h>
#include <map>
#include <StringTools.h>
#include <BedChrom.h>

class Bed {

 public:
  typedef int index_type;
  typedef string::size_type size_type;

  
 private:

  map<string,size_type> chromIds;
  Vec<BedChrom> chroms;

 public:
  
  Bed() { }

  Bed(const Bed& other) { copy(other); }

  virtual ~Bed() { }

  Bed& operator = (const Bed& other) {
    if (&other != this) {
      copy(other);
    }
    return (*this);
  }

  void buildIndices(index_type indexInterval) {
    PRECOND(indexInterval > 1);
    for (size_type i = 0; i < chroms.size(); ++i) {
      // cout << "Building index for chromosome " << (i+1) << " : " << chroms[i].getChrom() << endl;
      chroms[i].buildIndices(indexInterval);
    }
  }

  void copy(const Bed& other) {
    chromIds = other.chromIds;
    chroms = other.chroms;
  }

  const BedChrom& getChrom(size_type n) const { PRECOND(n < size()); return chroms[n]; }

  /** Returns "size()" if not found chromosome name */
  size_type findChromId(const string& chromName) const { 
    size_type result = size();
    map<string,size_type>::const_iterator iter = chromIds.find(chromName);
    if (iter != chromIds.end()) {
      result = iter->second;
    }
    return result;
  }

  /** Returns total number of intervals in all chromosomes */
  size_type intervalCount() const {
    size_type count = 0;
    for (Vec<BedChrom>::const_iterator it = chroms.begin(); it != chroms.end(); it++) {
      count += it->size();
    }
    return count;
  }

  /*  bool hasNames() const { 
    if (chroms.size() == 0) {
      return false;
    }
    return chroms[0].hasNames();
  }

  bool hasStrands() const { 
    if (chroms.size() == 0) {
      return false;
    }
    return chroms[0].hasStrands();
  }
  */

  void read(istream& is, int chromCol, int startCol, int endCol, int strandCol, int nameCol, 
                  int scoreCol, const Vec<int> annotationCols) {
    // cout << "# reading BED format file with column ids " << chromCol << " " << startCol << " " 
    // << endCol << " " << strandCol << " " << nameCol << " " << scoreCol << endl;
    while (is) {
      string line = getLine(is);
      if (line.size() == 0 || line[0] == '#') {
	continue;
      }
      vector<string> words = getTokens(line);
      if ((words.size() < 3) || (startCol >= static_cast<int>(words.size())) || (endCol>= static_cast<int>(words.size())) 
	  || (chromCol >= static_cast<int>(words.size())) || (strandCol >= static_cast<int>(words.size()))) {
	continue;
      }
      index_type start = stoi(words[startCol]);
      index_type end = stoi(words[endCol]);
      int strand = -2;
      if (strandCol >= 0) {
	if (words[strandCol] == "+") {
	  strand = 1;
	} else if (words[strandCol] == "-") {
	  strand = -1;
	}
      }
      string name;
      if (nameCol >= 0) {
	name = words[nameCol];
      }
      double score = 0.0;
      if (scoreCol >= 0) {
	score = stod(words[scoreCol]);
      }
      Vec<string> annotation;
      assert(annotation.size() == 0);
      for (size_type k = 0; k < annotationCols.size(); ++k) {
	ERROR_IF(annotationCols[k] >= static_cast<int>(words.size()), 
		 "Undefined annotation column.");
	annotation.push_back(words[annotationCols[k]]);
      }
      string chrom = words[chromCol];
      size_type chromId = 0;
      map<string,size_type>::const_iterator chromIter = chromIds.find(chrom);
      BedChrom * bedChromp = 0;
      if (chromIter != chromIds.end()) {
	chromId = chromIter -> second; // chrom id found
	bedChromp = &(chroms[chromId]);
      } else {
	chromId = chromIds.size();
	chromIds[chrom] = chromId;
	BedChrom bedChrom;
	bedChrom.setChrom(chrom);
	chroms.push_back(bedChrom);
	bedChromp = &(chroms[chromId]);
      }
      ASSERT(bedChromp != 0);
      bedChromp->add(start, end, strand, name, score,annotation);
    }
    POSTCOND(validate());
  }

  size_type size() const { return chroms.size(); }

  bool validate() const {
    for (Vec<BedChrom>::size_type i = 0; i < chroms.size(); ++i) {
      if (!chroms[i].validate()) {
	return false;
      }
    }
    return (chroms.size() == chromIds.size());
  }

  void write(ostream& os, bool writeStrand, bool writeName, bool writeScore) const {
    for (size_type i = 0; i < chroms.size(); ++i) {
      chroms[i].write(os, writeStrand, writeName, writeScore);
    }
  }

  void writeIndexSets(ostream& os) const {
    for (size_type i = 0; i < chroms.size(); ++i) {
      chroms[i].writeIndexSets(os);
    }
  }

};

#endif
