// --*- C++ -*------x---------------------------------------------------------

#ifndef _MAF_ALIGNMENT_TOOLS_
#define _MAF_ALIGNMENT_TOOLS_

#include <Stem.h>
#include <MAFAlignment.h>
#include <set>
#include <RnaSecondaryStructureTools.h>
#include <NucleotideTools.h>

using namespace std;

class MAFAlignmentTools {

 public:

  typedef Stem::index_type index_type;
  typedef MAFAlignment::row_type row_type;
  typedef MAFAlignment::length_type length_type;
  typedef MAFAlignment::size_type size_type;

 public:

  /** Returns true iff two column pairs are Watson-Crick complementary.
   * Only takes assemblies into account that are present at both columns.
   */
  static bool isColumnPairComplementary(const MAFAlignment& maf, length_type colid1, length_type colid2,
					bool allowGu, bool allowGap);

  /** Returns true iff two column pairs are mostly Watson-Crick complementary.
   * Only takes assemblies into account that are present at both columns.
   */
  static bool isColumnPairAlmostComplementary(const MAFAlignment& maf,
					      length_type colid1,
					      length_type colid2,
					      double allowedGuFrac,
					      double allowedGapFrac,
                                              bool convertToComplement,
					      size_type basepairTypeMin);


  /** Returns common assemblies from two different maf blocks
   */
  static set<string> findCommonAssemblies(const MAFAlignment& maf,
					   const MAFAlignment& maf2,
					   length_type aliId1,
					   length_type aliId2);

  /** Returns true iff two column pairs from two MAFs 
   * are Watson-Crick complementary.
   * Only takes assemblies into account that are present at both columns.
   */
  static bool isColumnPairComplementary(const MAFAlignment& maf,
					const MAFAlignment& maf2,
					length_type colid1, length_type colid2,
					bool allowGu,
					bool allowGap);

  /** Returns true iff two column pairs contain the same nucleotides at non-gap positions.
   * Only takes assemblies into account that are present at both columns.
   */
  static bool isColumnPairEquivalent(const MAFAlignment& maf, length_type colid1, length_type colid2);

  /** Returns true iff two column pairs from two MAFs contain the same nucleotides at non-gap positions.
   * Only takes assemblies into account that are present at both columns.
   */
  static bool isColumnPairEquivalent(const MAFAlignment& maf, const MAFAlignment& maf2,
				     length_type colid1, length_type colid2);

  /** For two alignment columns, find common assemblies and writes corresponding columns */
  static void writeCommonColumnPair(ostream& os, const MAFAlignment& maf, length_type colid1, length_type colid2) {
    PRECOND(colid1 >= 0 && colid1 < maf.getTotalLength());
    PRECOND(colid2 >= 0 && colid2 < maf.getTotalLength());
    set<string> commonAssemblies = maf.getCommonAssemblies(maf.getAlignmentId(colid1), maf.getAlignmentId(colid2));
    MAFAlignment::sequence_type columnStart = maf.getSlice(colid1, commonAssemblies);
    MAFAlignment::sequence_type columnStop = maf.getSlice(colid2, commonAssemblies);
    os << columnStart << " " << columnStop;
    for (set<string>::const_iterator it2 = commonAssemblies.begin(); it2 != commonAssemblies.end(); it2++) {
      os << " " << *it2;
    }
  }

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
   * If bowth stem strands are from the same MAF, provide it for both maf and maf2
   */
  static void augmentStemSequence(const MAFAlignment& maf, const MAFAlignment& maf2, Stem& stem, bool reverseMode);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
   */
  static int stemExpandability1Slow(const MAFAlignment& maf, const MAFAlignment& maf2, 
				    const Stem& stem, bool guAllowed, bool complement);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its aligned sequences are still complementary (no GU).
   */
  static int stemConservedExpandabilitySlow(
					    const MAFAlignment& maf, const MAFAlignment& maf2, 
					    const Stem& stem, bool leftMode, bool complementMode, bool reverseMode,
					    bool allowGu);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary. Not regular stem (that would be 
   * method stemExpandabilitySlow1/2), but "forward-matchting stem".
   */
  static int forwardMatchingStemExpandability1Slow(
		   const MAFAlignment& maf, const MAFAlignment& maf2, 
		   const Stem& stem, bool guAllowed, bool complement);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary. Not regular stem (that would be 
   * method stemExpandabilitySlow1/2), but "forward-matchting stem".
   */
  static int forwardMatchingStemExpandability2Slow(
		   const MAFAlignment& maf, const MAFAlignment& maf2, 
		   const Stem& stem, bool guAllowed, bool complement);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
   */
  static int stemExpandability1Fast(const MAFAlignment& maf, const MAFAlignment& maf2, 
				const Stem& stem, bool guAllowed, bool complement);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
   */
  static int stemExpandability2Fast(const MAFAlignment& maf, const MAFAlignment& maf2, 
				const Stem& stem, bool guAllowed, bool complement);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
   */
  static int stemExpandability2Slow(const MAFAlignment& maf, const MAFAlignment& maf2,
				    const Stem& stem, bool guAllowed, bool complement);

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
   */
  static int stemExpandability(const MAFAlignment& maf, const MAFAlignment& maf2,
			       const Stem& stem, bool guAllowed, bool complement,
			       bool complementMode, bool reverseMode) {
    PRECOND((complementMode && reverseMode) || ((!complementMode) && (!reverseMode)));
    // ERROR_IF(!complementMode, "Sorry, currently only reverse-complement mode possible for stemExpandability computation.");
    if (complementMode && reverseMode) {
      return stemExpandability1Slow(maf, maf2, stem, guAllowed, complementMode) 
	+ stemExpandability2Slow(maf, maf2, stem, guAllowed, complementMode);
    } 
    return forwardMatchingStemExpandability1Slow(maf, maf2, stem, guAllowed, complementMode) 
      + forwardMatchingStemExpandability2Slow(maf, maf2, stem, guAllowed, complementMode);
  }

  /** For a stem (coordinates given in assembly coordinates, not internal coordinates)
   * determine how far the stem sequences can be expanded in both directions such that its aligned sequences are still complementary.
   */
  static int stemConservedExpandability(const MAFAlignment& maf, const MAFAlignment& maf2,
					const Stem& stem, 
					bool complementMode, bool reverseMode, bool allowGu) {
    PRECOND((complementMode && reverseMode) || ((!complementMode) && (!reverseMode)));
    return stemConservedExpandabilitySlow(maf, maf2, stem, true, complementMode, reverseMode, allowGu) 
      +    stemConservedExpandabilitySlow(maf, maf2, stem, false, complementMode, reverseMode, allowGu);
  }


};
  
  
#endif
