#ifndef __BOND_H_
#define __BOND_H_

#include <iostream>
#include <utility>
#include <debug.h>
#include <Vec.h>
#include <string>

typedef pair<unsigned int, unsigned int> Bond;

const unsigned int BOND_ORDER_MAXNORMAL = 3; // everything above triple bond must be special internal coding
const unsigned int BOND_ORDER_AR = 4; // internal bond order for aromatic bond
const unsigned int BOND_ORDER_AM = 5; // internal bond order for peptide bond
const unsigned int BOND_ORDER_UNK = 6; // unknown bond

const string BOND_STRING_AR = "ar";
const string BOND_STRING_AM = "am";
const string BOND_STRING_UNK = "unk";


/** get first id of bond */
inline
unsigned int
getFirst(const Bond& b) { return b.first; }

/** get second id of bond */
inline
unsigned int
getSecond(const Bond& b) { return b.second; }

/** get first id of bond */
inline
void
setFirst(Bond& b, unsigned int n) { b.first = n; }

/** get second id of bond */
inline
void
setSecond(Bond& b, unsigned int n) { b.second = n; }

/** generates bond using indices a and b */
inline
Bond
generateBond(unsigned int a, unsigned int b) 
{
  Bond bond;
  setFirst(bond, a);
  setSecond(bond, b);
  return bond;
}



inline
void
outBond(ostream& os, const Bond& b)
{
  os << getFirst(b) + 1 << " " << getSecond(b) + 1;
}

inline
void
outBondList(ostream& os, const Vec<Bond>& bonds)
{
  os << bonds.size() << " used bonds: " << endl;
  for (unsigned int i = 0; i < bonds.size(); ++i) {
    os << i+1 << " ";
    outBond(os, bonds[i]);
  }    
}

// return indices of neighouring atom for given index
Vec<unsigned int>
getNeighbours(unsigned int n, const Vec<Bond>& bonds);

/* return list with neighbour info */
Vec<Vec<unsigned int> > getNeighbourList(const Vec<Bond>& bonds, unsigned int defVal);

/**
 * returns unique list of all atom indices which are occuring in bonds
 */
Vec<unsigned int>
getUniqBondedList(const Vec<Bond>& bonds);

inline
ostream& operator << (ostream& os, const Bond& b) 
{
  os << getFirst(b) << " " << getSecond(b); return os;
}

inline
istream& operator >> (istream& is, Bond& b) 
{
  unsigned int id1, id2;
  is >> id1 >> id2; 
  setFirst(b, id1);
  setSecond(b, id2);
  return is;
}


inline 
bool 
isNeighbour(unsigned int n, unsigned int m, 
	    const Vec<Bond>& bonds)
{
  for (unsigned int i = 0; i < bonds.size(); ++i) {
    if (((getFirst(bonds[i]) == n) && (getSecond(bonds[i]) == m)) 
	|| ((getFirst(bonds[i]) == m) && (getSecond(bonds[i]) == n) ) ) {
      return true;
    }
  }
  return false;
}

inline
bool 
isIndirectNeighbour(unsigned int n, unsigned int m, 
		    unsigned int molSize,
		    const Vec<Bond>& bonds)
{
  for (unsigned int i = 0; i < bonds.size(); ++i) {
    if (getFirst(bonds[i]) == n) {
      if (getSecond(bonds[i]) == m) {
	return false; // direct neighbours
      }
      if (isNeighbour(getSecond(bonds[i]), m, bonds)) {
	return true; // indirect neighbours
      }
    }
    else if (getSecond(bonds[i]) == n) {
      if (getFirst(bonds[i]) == m) {
	return false; // direct neighbours
      }
      if (isNeighbour(getFirst(bonds[i]), m, bonds)) {
	return true; // indirect neighbours
      }
    }
  }
  return false;
}

/* return true, if bond descriptors are consistent with (smaller indices) with maxIndex */
inline
bool
bondsOk(const Vec<Bond>& bonds, unsigned int maxIndex)
{
  for (unsigned int i = 0; i < bonds.size(); ++i) {
    if ( (getFirst(bonds[i]) == getSecond(bonds[i]))
	 || (getFirst(bonds[i]) >= maxIndex)
	 || (getSecond(bonds[i]) >= maxIndex) ) {
      return false;
    }
  }
  return true;
}

inline 
bool 
isInBondList(unsigned int n, unsigned int m, const Vec<Bond>& bonds)
{
  for (unsigned int i = 0; i < bonds.size(); ++i) {
    if ( ((getFirst(bonds[i]) == n) && (getSecond(bonds[i]) == m))
	 || ((getFirst(bonds[i]) == m) && (getSecond(bonds[i]) == n)) ) {
      return true;
    }
  }
  return false;
}

inline
bool
isInBond(unsigned int n, const Bond& bond) {
  return (getFirst(bond) == n) || (getSecond(bond) == n);
}

inline
bool
isInBond(unsigned int n, unsigned int m, const Bond& bond) {
  return isInBond(n, bond) || isInBond(m, bond);
}

unsigned int
findFirstIndex(unsigned int n, const Vec<Bond>& bonds);

unsigned int
findFirstIndex(unsigned int n, unsigned int m, const Vec<Bond>& bonds);

void
shiftBonds(Vec<Bond>& bonds, int shift);

string
bondOrderToString(unsigned int);

unsigned int
stringToBondOrder(string name);


#endif
