// --*- C++ -*------x---------------------------------------------------------
// $Id: CoordinateSystem.h,v 1.1.1.1 2006/07/03 14:43:20 bindewae Exp $
//
// Class:           CoordinateSystem
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     concept of coordinate system. Consists of origin and 
//                  two ORTHOGONAL direction vectors (the third is 
//                  the cross product of them)
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __COORDINATE_SYSTEM_H__
#define __COORDINATE_SYSTEM_H__

// Includes

#include <iostream>
#include <debug.h>
#include <Vector3D.h>
#include <Matrix3D.h>

class CoordinateSystem {

public:

  /* CONSTRUCTORS */

  CoordinateSystem();
  CoordinateSystem(const Vector3D& origin,
		   const Vector3D& x, const Vector3D& y);
  CoordinateSystem(const CoordinateSystem& orig);
  virtual ~CoordinateSystem();

  /* OPERATORS */

  /** Assigment operator. */
  CoordinateSystem& operator = (const CoordinateSystem& orig);
  /** comparison operator */
  friend bool operator==(const CoordinateSystem& left, const CoordinateSystem& right);
  /** output operator */
  friend ostream& operator << (ostream& os, const CoordinateSystem& object);
  /** input operator  */
  friend istream& operator >> (istream& is, CoordinateSystem& object);

  /* PREDICATES */

  const Vector3D& getOrigin() const { return origin; }
  const Vector3D& getXAxis() const { return ex; }
  const Vector3D& getYAxis() const { return ey; }
  const Vector3D& getZAxis() const { return ez; }
  /** return position with respect to THIS coordinate system */
  Vector3D newCoordinates(const Vector3D& v) const {
      return m * (v - origin);  } 
  /** return position with respect to global coordinate system */
  Vector3D oldCoordinates(const Vector3D& v) const {
    return (mt * v) + origin;   }
  /** return true, if determinante of direction vectors is 1.0 */
  bool isProperlyDefined() const;

  /* MODIFIERS */
  void setOrigin(const Vector3D& v) { origin = v; }
  void setAxis(const Vector3D& v1, const Vector3D& v2);
  void resetOrientation() { setAxis(Vector3D(1.0,0.0,0.0),
				    Vector3D(0.0,1.0,0.0)); }
  void reset() { resetOrientation(); setOrigin(Vector3D(0.0, 0.0, 0.0)); }
  /** apply Euler angle rotation to axis of coordinate system
      does not change origin! */
  void rotateEulerAngles(double u, double v, double w);
protected:
  /* OPERATORS  */
  /* PREDICATES */
  virtual bool compare(const CoordinateSystem& other) const;
  /* MODIFIERS  */
  virtual void copy(const CoordinateSystem& orig);
private:
  /** direction vectors are ROWS of matrix! */
  void updateMatrix();
  /* PRIVATE ATTRIBUTES */
  Vector3D origin;
  Vector3D ex,ey,ez; // coordinate axis vectors
  Matrix3D m, mt; // only needed as "cache"
}; // class CoordinateSystem

/** matrix is contains direction vectors of 
    new coordinate system as ROWS.
    mt is transposed of m. */
inline
void 
CoordinateSystem::updateMatrix() 
{ 
  m = Matrix3D(ex.x(),ex.y(),ex.z(),
	       ey.x(),ey.y(),ey.z(), 
	       ez.x(),ez.y(),ez.z());
  mt = Matrix3D(ex.x(),ey.x(),ez.x(),
		ex.y(),ey.y(),ez.y(), 
		ex.z(),ey.z(),ez.z());
}


#endif /* __A_CLASS_H__ */

