// --*- C++ -*------x---------------------------------------------------------
// $Id: 
//
// Program:         averager
//
// Author:          Eckart Bindewald
//
// Project name:    averager
//
// Description:     read multi column list with numbers,
//                  return for each line average, standard deviation,
//                  average + and - standard deviation
//                  Demonstrates a little word parser
// -----------------x-------------------x-------------------x-----------------

#include <math.h>
#include <iostream>
#include <Vec.h>
#include <string>
#include <sstream>
#include <vectornumerics.h>
#include <GetArg.h>

#include <debug.h> // own routines for error checking and debugging

// return string with actual line. Reads last newline, but does not include it.
/*
string
getLine(istream& is)
{
  PRECOND(is);
  char c;
  string resultString; 
  is.get(c);
  while ((is) && (!is.eof()) && (c != '\n') )
    {
      resultString = resultString + c;
      is.get(c);
    }
  return resultString;
}
*/

// return average of Vec of double 
double
average(const Vec<double>& a)
{
  PRECOND(a.size()>0);
  double sum = 0.0;
  for (unsigned int i = 0; i < a.size(); ++i)
    {
      sum += a[i];
    }
  return sum/a.size();
}

// return standard deviation of single values 
double
stddev(const Vec<double>& a)
{
  PRECOND(a.size()>0);
  double avg = average(a);
  double sum = 0.0;
  if (a.size() < 2) {
    return 0.0;
  }
  for (unsigned int i = 0; i < a.size(); ++i)
    {
      sum += (a[i] - avg) * (a[i] - avg);
    }
  return sqrt(sum/(a.size()-1)); 
}

// error: if the input is no number (e.g. char or string)  then error
// changes string into double:
/* 
double stod( const string &s )
{
  PRECOND(s.size() > 0);
  double d;
  double temp; // temporary integer

  const char *pointer;
  pointer = s.c_str();
  istringstream ist( pointer );
  ist >> temp;
  // if ist==NULL (s is no number) then error:
  if( ist == NULL )
    {
      cerr << "\"" << s << "\" is no double!!" << endl;
      ERROR( "Wrong format" ); 
    }
  d = temp;
  return d;
}
*/

// tokenize text
/*
Vec<string> getTokens(const string& text) 
{
  istringstream ist(text.c_str());
  char* charLine = new char[text.size()+1]; // size of string
  Vec<string> v;
  string s;
  while (!ist.eof()) {
    ist >> charLine;
    s = charLine; // assignment of c-strings to string!
    //    DUMP(s);
    if (s != "") { 
      v.push_back(s);
    }
  }
  delete[] charLine;
  return v;
}
*/

int
main(int argc, char ** argv)
{
  bool enumMode   = false;
  bool helpMode   = false;

  int countStart = 1; 
  int outputFormat = 2;
  
  const double CUT_BELOW_DEFAULT = -1e30;
  const double CUT_ABOVE_DEFAULT = 1e30;

  double cutAbove = CUT_ABOVE_DEFAULT;
  double cutBelow = CUT_BELOW_DEFAULT;

  Vec<unsigned int> mask;

  getArg("-cut-above", cutAbove, argc, argv, cutAbove);
  getArg("-cut-below", cutBelow, argc, argv, cutBelow);
  getArg("h", helpMode, argc, argv);
  getArg("n", enumMode, argc, argv);
  getArg("m", countStart, argc, argv, countStart);
  getArg("-mask", mask, argc, argv);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  convert2InternalCounting(mask);
  if (helpMode) {
    cout << "The program reads from standard input."
	 << "It expects numerical data in columns."  << endl
	 << "For each row it writes mean, standard deviation,"
	 << " mean plus std-dev, and mean minus std-dev of the" 
	 << " numbers of the columns" << endl
	 << "usage: averager [-h][-n][-m number][--mask mask] < inputfile > outputfile"
	 << endl;
    exit(0);
  }
  Vec<double> avgVec, stdVec, stdMVec;
  Vec<Vec<double> > data;
  while ( cin && (!cin.eof()) ) {
      string line = getLine(cin);
      if ((line.size() == 0) || (line[0] == '#')) {
	continue; // skip empty or comment lines
      }
      Vec<string> words = getTokens(line); // seperate into words
      if (mask.size() > 0) {
	words = getSubset(words, mask);
      }
      Vec<double> aVec(words.size(),0.0);
      for (unsigned int i = 0; i < aVec.size(); ++i) {
	aVec[i] = stod(words[i]); // translate string into double
	if ((cutBelow > CUT_BELOW_DEFAULT)
	&& (aVec[i] < cutBelow)) {
	  aVec[i] = cutBelow;
	}
	if ((cutAbove < CUT_ABOVE_DEFAULT)
	    && (aVec[i] > cutAbove)) {
	  aVec[i] = cutBelow;
	}
      }
      if (aVec.size() == 0) {
	continue; // skip empty line
      }
      data.push_back(aVec);
  }
  ERROR_IF(data.size() == 0, "No data defined!");
  unsigned int numCol = data[0].size();
  switch (outputFormat) {
  case 1:
    for (unsigned int i = 0; i < numCol; ++i) {
      Vec<double> slice = getColumnSave(data, i);
      cout << "Column " << i + 1 << " : ";
      distributionInfo(cout, slice);
    }
    break;
  case 2:
    cout << "# ";
    for (unsigned int i = 0; i < numCol; ++i) {
      cout << "avg-" << i+1 << " std-" << i+1 << " ; ";
    }
    cout << endl;
    for (unsigned int i = 0; i < numCol; ++i) {
      Vec<double> slice = getColumnSave(data, i);
      double avg = average(slice);
      double stds = stddev(slice); // standard deviation of single values
      // output standard deviation of single value and of mean
      cout << avg << " +- " << stds << " ( " << stds/sqrt(static_cast<double>(slice.size())) << " ) ; ";
    }
    cout << endl;
    break;
  case 3:
    for (unsigned int i = 0; i < numCol; ++i) {
      Vec<double> slice = getColumnSave(data, i);
      double avg = average(slice);
      cout << (i+1) << " " << avg << endl;
    }
    break;
  case 4:
    for (unsigned int i = 0; i < numCol; ++i) {
      Vec<double> slice = getColumnSave(data, i);
      double sum = elementSum(slice);
      cout << (i+1) << " " << sum << endl;
    }
    break;    
  default: ERROR("Unknown switch format!");
  }

  return 0;
}







