// --*- C++ -*------x---------------------------------------------------------
// $Id: Vec.h,v 1.2 2011/07/27 19:49:28 bindewae Exp $
//
// Class:           Vec
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Matthias Heiler, Eckart Bindewald
//
// Description:     "vector" with range checking. 
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __VEC_H__
#define __VEC_H__

// Includes

#include <libintl.h>
#include <vector>

#include <debug.h>

using namespace std;

/** "vector" with range checking. 

    Behaves like STL's vector. The [] Selector is range checked,
    unless "NDEBUG" is defined at compile time. This implementation
    differs from Stroustrup's book, since our STL doesn't provide an
    "at" Selector.

    This class offers such an "at" method which *always* performs range
    checking (even if NDEBUG is defined at compile time).

    @see    Stroustrup, "The C++ Programming Language", 3rd Edition, p.53 */
template<class T>
class Vec: public vector<T> {
public:
  typedef typename vector<T>::size_type size_type;
  typedef typename vector<T>::iterator iterator;
  typedef typename vector<T>::const_iterator const_iterator;

public:
  Vec(): vector<T>() { }
  Vec(const vector<T>& orig) : vector<T>(orig) { }
  Vec(size_type s): vector<T>(s) { }
  Vec(size_type s, const T& def): vector<T>(s, def) { }

  // Predicates

  inline T& operator [] (size_type i);
  inline const T& operator [] (size_type i) const;

  inline T& at(size_type i);
  inline const T& at(size_type i) const;

  iterator begin() { return vector<T>::begin(); }

  const_iterator begin() const { return vector<T>::begin(); }

  size_type size() const { return vector<T>::size(); }

  // Friends

//   friend istream& operator >> <>(istream& is, Vec<T>& v);
//   friend ostream& operator << <>(ostream& os, const Vec<T>& v);

};

// *******************************************

template <class T>
inline 
T& 
Vec<T>::operator [] (size_type i) 
{
#ifndef NDEBUG
  if (i >= size()) 
    {
      ERROR("Range error.");
    }
#endif
  return *(begin() + i); 
}

template <class T>
inline 
const T& 
Vec<T>::operator [] (size_type i) const
{
#ifndef NDEBUG
  if (i >= size()) 
    {
      ERROR("Range error.");
    }
#endif
  return *(begin() + i); 
}

/** "at" always performs range checking.

  Unlike operator[] "at" always checks for range errors. */
template <class T>
inline 
T& 
Vec<T>::at(size_type i) 
{
  if (i >= size()) 
    {
      ERROR("Range error.");
    }
  else 
    {
      return *(begin() + i); 
    }
}

/** "at" always performs range checking.

  Unlike operator[] "at" always checks for range errors. */
template <class T>
inline 
const T& 
Vec<T>::at(size_type i) const
{
  if (i >= size()) 
    {
      ERROR("Range error.");
    }
  else 
    {
      return *(begin() + i); 
    }
}

/** Sends a Vec to stream.

  The data format is very simple: the first element is the length of
  the vector followed by a space-seperated list of elements. */
template <class T>
inline
istream& 
operator >> (istream& is, Vec<T>& v)
{
  PRECOND(is);
  typename Vec<T>::size_type size;
  v.clear(); // first remove all existing elements EB 02/2001
  is >> size;
  ERROR_IF(!is, "Vec: Number of elements expected as first item!");
  // maximum number of input elements:
  ERROR_IF(size > 200000, "Vec: Too large number of input elements!");
  for (typename Vec<T>::size_type i = 0; i < size; i++)
    {
      T element;
      is >> element;
      ERROR_IF(!is, "Vec: Error reading element!");
      v.push_back(element);
    }

//   v.reserve(size);
//   while (v.size() > size)
//     {
//       v.erase(v.begin());
//     }
//   for (Vec<T>::size_type i = 0; i < size; i++)
//     {
//       is >> v[i];
//     }


  POSTCOND(is);
  return is;

}

/** Writes a Vec to a stream.

  The data format is very simple: the first element is the length of
  the vector followed by a space-seperated list of elements. */
template <class T>
inline
ostream& 
operator << (ostream& os, const Vec<T>& v)
{
  PRECOND(os);
  os << v.size() << "   ";
  // copy(v.begin(), v.end(), ostream_iterator<T>(os, " "));
  for (unsigned int i = 0; i < v.size(); ++i) {
    os << v[i] << " ";
  }
  os << endl;
  POSTCOND(os);
  return os;
}


/** Writes a Vec to a stream.

  The data format is very simple: the first element is the length of
  the vector followed by a space-seperated list of elements. */
template <class T>
inline
ostream& 
outList(ostream& os, const Vec<T>& v)
{
  PRECOND(os);
  os << v.size() << endl;
  // copy(v.begin(), v.end(), ostream_iterator<T>(os, "\n"));
  for (unsigned int i = 0; i < v.size(); ++i) {
    os << v[i] << endl;
  }
  os << endl;
  POSTCOND(os);
  return os;
}

/** Writes a Vec to a stream.

  The data format is very simple: the first element is the length of
  the vector followed by a lines with id number and content of vector element */
template <class T>
inline
ostream& 
outListNice(ostream& os, const Vec<T>& v)
{
  PRECOND(os);
  os << v.size() << endl;
  for (typename Vec<T>::size_type i = 0; i < v.size(); ++i) {
    os << i << " : " << v[i] << endl;
  }
  POSTCOND(os);
  return os;
}

/**
 comparison. return true, if == is true for all elements 
*/
template <class T>
inline
bool
operator == (const Vec<T>& lval, const Vec<T>& rval)
{
  if (lval.size() != rval.size()) {
    return false;
  }
  for (unsigned int i = 0; i < lval.size(); ++i) {
    if (!(lval[i] == rval[i])) {
      return false;
    }
  }
  return true;
}

#endif /* __A_CLASS_H__ */

