library(bio3d)

# R script computes theoretical hydrodynamic radius for a PDB file of an RNA structure.
# see: http://en.wikipedia.org/wiki/Hydrodynamic_radius
# Usage: Rscript trhydro.R PDBFILE
# or R --vanilla --args PDBFILE < trhydro.R
# with PDBFILE being the filename of the PDB structure. The script has in this example to be in the 
# same directory as the PDB file.
# Eckart Bindewald, 2010

# return Van de Waals radius of atom. Input: "C","N","P","O", "H"
radius.vdw <- function(nameOrig, verbose=0) {
  result = 1.0
  name <- substr(nameOrig,1,1)
  # if first character is a number, take second character
  # example : 1HD3 becomes H
  if ((name == "*") || (!is.na(as.numeric(name)))) {
    name <- substr(nameOrig,2,2)
  }
  stopifnot(nchar(name) == 1)
  if (name == "C") {
    result = 1.70
  } else if (name == "N") {
    result=1.55
  } else if (name == "P") {
    result=1.80
  } else if (name == "O") {
    result=1.52
  } else if (name == "H") {
    result=1.2
  } else {
    cat("Unknown element name in radius.vdw:", name, "\n")
    stop()
  }
  if (verbose > 0) {
    cat("Radius of atom type ", name, ":", result, "\n")
  }
  result
}

test.radius.vdw <- function() {
  stopifnot(radius.vdw("C") == 1.7)
  stopifnot(radius.vdw("N") == 1.55)
  stopifnot(radius.vdw("O") == 1.52)
  stopifnot(radius.vdw("O1P") == 1.52)
  stopifnot(radius.vdw("H") == 1.2)
  stopifnot(radius.vdw("2HD3") == 1.2)
  stopifnot(radius.vdw("*HD3") == 1.2)
}

# atomic mass in dalton based on carbon 12 standard
# see: http://www.sisweb.com/referenc/source/exactmaa.htm
mass.atom <- function(nameOrig) {
  result = 1.0
  name <- substr(nameOrig,1,1)
  # if first character is a number, take second character
  # example : 1HD3 becomes H
  if ((name == "*") || (!is.na(as.numeric(name)))) {
    name <- substr(nameOrig,2,2)
  }
  stopifnot(nchar(name) == 1)
  if (name == "C") {
    result = 12.0
  } else if (name == "N") {
    result=14.003074
  } else if (name == "P") {
    result=30.973763
  } else if (name == "O") {
    result=15.994915
  } else if (name == "H") {
    result= 1.007825
  } else {
    cat("Unknown element name in mass.atom:", name, "\n")
    stop()
  }
  result
}

test.mass.atom <- function() {
  stopifnot(mass.atom("C") == 12)
  stopifnot(mass.atom("N") == 14.003074)
  stopifnot(mass.atom("O") == 15.994915)
  stopifnot(mass.atom("O1P") == 15.994915)
  stopifnot(mass.atom("H") == 1.007825)
  stopifnot(mass.atom("1H3") == 1.007825)
  stopifnot(mass.atom("*H3") == 1.007825)
}

helpOutput <- function() {
  cat("Program computes the maximum distance between any two residues of an RNA structure. Each residue position is represented through its phosphate atom position. Input is a PDB file. This corresponds to the minimum radius of a sphere containing the structure.\n")
}

dist.euclid <- function(x1,y1,z1,x2,y2,z2) {
  sqrt((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2) )
}

dist.euclidSq <- function(x1,y1,z1,x2,y2,z2) {
  (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2)
}

dist.euclidSqD <- function(x1,x2,x3) {
 x1*x1 + x2*x2 + x3*x3
}

test.dist.euclid <- function() {
   print(dist.euclid(3,0,0,0,4,0))
}

radius.max <- function(pdb) {
  p.inds <- atom.select(pdb, "///////")
#  n <- length(p.inds)
  xyz <- pdb$xyz[ p.inds$xyz ]
  n3 <- length(xyz)
  xv <- xyz[seq(1,n3,3)]
  yv <- xyz[seq(2,n3,3)]
  zv <- xyz[seq(3,n3,3)]
  n <- length(xv)
  stopifnot(length(xv)==length(zv))
  stopifnot((n * 3) == n3)
  stopifnot(n == nrow(pdb$atom))
  rMax <- 0
  xsum <- 0
  ysum <- 0
  zsum <- 0
  massSum <- 0
  for (i in 1:n) {
    mass <- mass.atom(pdb$atom[i,"elety"])
    stopifnot(mass > 0)
    xsum <- xsum + mass * xv[i]
    ysum <- ysum + mass * yv[i]
    zsum <- zsum + mass * zv[i]
    massSum <- massSum + mass
  }
  xc <- xsum / massSum
  yc <- ysum / massSum
  zc <- zsum / massSum  # coordinates of center of mass
  cat("Center of mass", xc,yc,zc, "\n")
  bestIndex <- 0
  for (i in 1:n) {
       vdw <- radius.vdw(pdb$atom[i,"elety"])
        dx <- xc - xv[i]
        dy <- yc - yv[i]
        dz <- zc - zv[i]
        r <- sqrt(dx*dx + dy*dy + dz*dz) + vdw
        if (r > rMax) {
            rMax <- r
            bestIndex <- i

        } 
  }
  result <- rMax 
  cat("best radius", result, "found for:\n")
  print(pdb$atom[bestIndex,])
  radius.vdw(pdb$atom[bestIndex,"elety"], verbose=2)
  result
}

## main program

args <- commandArgs(trailingOnly=TRUE)
if (length(args) < 1) {
	helpOutput()
} else {
 filename <- args[1]
 pdb <- read.pdb(filename)
 rMax <- radius.max(pdb)
 cat("Sphere radius:", rMax, "Angstroem\n")
}
