package rnadesign.designapp;

import java.util.logging.Logger;
import java.io.File;
import java.lang.Double;
import java.lang.Boolean;
import commandtools.AbstractCommand;
import commandtools.Command;
import commandtools.CommandExecutionException;
import commandtools.IntParameter;
import commandtools.StringParameter;
import rnadesign.rnacontrol.Object3DGraphController;
import rnadesign.rnacontrol.Object3DGraphControllerConstants;
import rnadesign.rnacontrol.PdbJunctionControllerParameters;
import rnadesign.rnamodel.CorridorDescriptor;
import rnadesign.rnamodel.JunctionScanner;

import static rnadesign.designapp.PackageConstants.NEWLINE;

public class LoadJunctionsCommand extends AbstractCommand {

    private static Logger log = Logger.getLogger("NanoTiler_debug");
    
    public static final String COMMAND_NAME = "loadjunctions";

    private int orderMax = 5; // currently fixed
    private int stemLengthMin = 3;
    private String fileName;
    private String usedDirectory;
    private int loopLengthSumMax = JunctionScanner.LOOP_LENGTH_SUM_MAX;
    private String writeName;
    private String writeDir;
    private boolean noWriteFlag = false;
    private boolean intHelixCheck = true;
   
    boolean rnaMode = true; // currently fixed: expect RNA and RNAView output in pdb file

    int branchDescriptorOffset = 2; // if nothing specified, extend helices by this many base pairs

    private CorridorDescriptor corridorDescriptor = new CorridorDescriptor(Object3DGraphControllerConstants.CORRIDOR_DEFAULT_RADIUS,
									   Object3DGraphControllerConstants.CORRIDOR_DEFAULT_START);

    private Object3DGraphController controller;
    
    public LoadJunctionsCommand(Object3DGraphController controller) {
	super(COMMAND_NAME);
	assert controller != null;
	this.controller = controller;
    }

    public Object cloneDeep() {
	LoadJunctionsCommand command = new LoadJunctionsCommand(controller);
	command.orderMax = this.orderMax;
	command.fileName = this.fileName;
	command.usedDirectory = this.usedDirectory;
	command.writeName = this.writeName;
	command.writeDir = this.writeDir;
	command.noWriteFlag = this.noWriteFlag;
	command.rnaMode = this.rnaMode;
	command.branchDescriptorOffset = this.branchDescriptorOffset;
	command.stemLengthMin = this.stemLengthMin;
	command.corridorDescriptor = new CorridorDescriptor(this.corridorDescriptor);
	command.intHelixCheck = this.intHelixCheck;
	for (int i = 0; i < getParameterCount(); ++i) {
	    command.addParameter((Command)(getParameter(i).cloneDeep()));
	}
	return command;
    }

    public void executeWithoutUndo() throws CommandExecutionException {
	assert (this.controller != null);
	prepareReadout();
	log.info("Starting to read junctions using a minimum stem-length cutoff of " + stemLengthMin);
	PdbJunctionControllerParameters prm = 
	    new PdbJunctionControllerParameters(orderMax,
						fileName,
						usedDirectory,
						//writeName, 
						writeDir,
						noWriteFlag,
						rnaMode,
						branchDescriptorOffset,
						stemLengthMin,
						corridorDescriptor,
						loopLengthSumMax, intHelixCheck);
	// PdbJunctionController junctionController = (PdbJunctionController)(graphController.getJunctionController());
	assert prm != null;
	this.controller.resetJunctionController(prm); 
	this.controller.setLastReadDirectory(usedDirectory);
	// log.warning("Nanotiler should do something here! LoadJunctionsCommand line 57");
	//this.controller.set... TODO: christine
    }

    public Command execute() throws CommandExecutionException {
	executeWithoutUndo();
	return null;
    }

    private void prepareReadout() throws CommandExecutionException {
	if (getParameterCount() == 0) {
	    throw new CommandExecutionException(helpOutput());
	}
	assert (getParameterCount() > 0);
	assert getParameter(0) instanceof StringParameter;
	StringParameter stringParameter = (StringParameter)(getParameter(0));
	fileName = stringParameter.getValue();
	// TODO : don't know what this code is for ... EB
	log.fine("Nanotiler should write something here. LoadJunctionsCommand line 72");
	//writeName = "something"; //TODO christine
	if (getParameterCount() > 1) {
	    assert (getParameter(1) instanceof StringParameter);
	    StringParameter stringParameter2 = (StringParameter)(getParameter(1));
	    usedDirectory = stringParameter2.getValue();
	}
	else {
	    usedDirectory = (new File(fileName)).getParent(); // use directory
	}
	try {
	    if (getParameterCount() > 2) {
		Command parameter3 = getParameter(2);
		if (parameter3 instanceof StringParameter) {
		    branchDescriptorOffset = Integer.parseInt(((StringParameter)(parameter3)).getValue());
		}
		else if (parameter3 instanceof IntParameter) {
		    branchDescriptorOffset = ((IntParameter)(parameter3)).getValue();
		}
	    }
	    StringParameter stemMinParameter = (StringParameter)(getParameter("stem-min"));
	    if (stemMinParameter != null) {
		this.stemLengthMin = Integer.parseInt(stemMinParameter.getValue());
	    }
	    StringParameter corridorWidthParameter = (StringParameter) (getParameter("corridor-width"));
	    if(corridorWidthParameter != null) {
		double corridorWidth = Double.parseDouble(corridorWidthParameter.getValue());
		this.corridorDescriptor = new CorridorDescriptor (corridorWidth, Object3DGraphControllerConstants.CORRIDOR_DEFAULT_START);
	    }
	    StringParameter intHelixCheckParameter = (StringParameter) (getParameter("inthelixcheck"));
	    if(intHelixCheckParameter != null)
		this.intHelixCheck = Boolean.parseBoolean(intHelixCheckParameter.getValue());
	}
	
    
	catch (NumberFormatException nfe) {
	    throw new CommandExecutionException("Error parsing value: " + nfe.getMessage());
	}
    }

    
    public String helpOutput() {
	return "Correct usage: " + COMMAND_NAME + " fileName [directory [offset]][stem-min=1|2|3|4|...][corridor-width=DOUBLE][inthelixcheck=BOOLEAN]";
    }
    
    public String getShortHelpText() { return helpOutput(); }

    public String getLongHelpText() {
	String helpText = "\"" + COMMAND_NAME + "\" Command Manual" +
	    NEWLINE + NEWLINE;
	helpText += "NAME" + NEWLINE + "     " + COMMAND_NAME +
	    NEWLINE + NEWLINE;
	helpText += "SYNOPSIS" + NEWLINE + "     " + COMMAND_NAME +
	    " [options]" + NEWLINE + NEWLINE;
	helpText += "DESCRIPTION" + NEWLINE +
	    "     LoadJunctions command TODO.";
	helpText += "OPTIONS" + NEWLINE;
	helpText += "     stringparameter  : name of file with relative filenames of junctions" + NEWLINE;
	helpText += "     stringparameter2 : name of directory of junctions" + NEWLINE ;
	helpText += "     branchDescriptorOffset : 0|1|2|... : Length of connector helices. Default: 2. Often used: 3" + NEWLINE + NEWLINE;
	helpText += "     stem-min=1|2|3|...    : Minimum length of considered stems." + NEWLINE;
	helpText += "     corridor-width=DOUBLE : Sets the radius of the corridor in front of helices in which collisions will be checked for. Default: 0. Often used: 3." + NEWLINE;
	helpText += "     inthelixcheck=BOOLEAN : Indicates whether or not the program should check for internal helices in the parsed pdb files. Default: True."+ NEWLINE;
	return helpText;
    }

    
}
