package rnadesign.rnamodel;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Random;
import java.util.logging.Logger;
import org.testng.annotations.Test; /** Testing package. */
import generaltools.Randomizer;
import graphtools.PermutationGenerator;
import numerictools.BaseConversion;
import numerictools.DoubleArrayTools;
import rnadesign.designapp.rnagui.NanoTiler;
import sequence.SequenceStatus;
import static rnadesign.rnamodel.PackageConstants.*; /** Imports constants. */
import tools3d.Junction;
import tools3d.MCSuperposeCollinear;
import tools3d.PointShape;
import tools3d.Superpose;
import tools3d.SuperpositionResult;
import tools3d.Symmetry;
import tools3d.Vector3D;
import tools3d.Vector3DTools;
import tools3d.numerics3d.Geom3DTools;
import tools3d.objects3d.Link;
import tools3d.objects3d.LinkSet;
import tools3d.objects3d.LinkTools;
import tools3d.objects3d.Object3D;
import tools3d.objects3d.Object3DLinkSetBundle;
import tools3d.objects3d.Object3DSet;
import tools3d.objects3d.Object3DTools;
import tools3d.objects3d.SimpleLinkSet;
import tools3d.objects3d.SimpleObject3D;
import tools3d.objects3d.SimpleObject3DLinkSetBundle;
import tools3d.objects3d.SimpleObject3DSet;
import sequence.SequenceStatus;
/** 
 * Uses building block approach (for junctions) to trace a graph with RNA strands 
 */
public class FragmentGridTiler implements StatisticsGridTiler {

    /* CONSTANTS */

    private static final double ANGLE_LIMIT_DEFAULT = 20 * DEG2RAD; /** */
    private static final double NO_JUNCTION_PLACED_PENALTY = 100.0; /** Penalty if no junction is placed. */
    private static final double NO_STEM_PLACED_PENALTY = 10.0; /** Penalty if no stem is placed. */
    private static final int GENERATE_ALL_STEMS_INDEX = -1; /** */
    private static final double RMS_LIMIT_DEFAULT = 3.0; /** */
    private static final NumberFormat THREE_DECIMALS = new DecimalFormat(".000"); /** Formats a number to go to only three decimal places. */
    private static final String JUNCTION_ENDING = "_jnc";

    /* STATIC VARIABLES */

    private static List[] oldSymmetryJunctions = new ArrayList[6]; // TODO : dangerous: contains temp information about last use of class. Move to method
    private static Logger log = Logger.getLogger(LOGFILE_DEFAULT); /** Debug log. */
    private static Random random = Randomizer.getInstance(); /** */

    /* ATTRIBUTES */

    private static StrandJunction3D[] symmetryJunctions = new StrandJunction3D[6]; /** Stores chosen junction fragment. TODO : dangerous: contains temp information about last use of class. Move to method */

    /* ATTRIBUTES */

    private int appendStrandsMode = ConnectingStemGenerator.APPEND_STICKY_ENDS; /** If true, append stem strands to junction strands. */
    private int axialSteps = 36; // number of steps for retrying kissing loop placement in middle of edge
    private boolean junctionPlaceMode = true; /** If true, place junctions at vertices. */
    private int connectionAlgorithm = BranchDescriptorOptimizerFactory.MONTE_CARLO_OPTIMIZER; /** Algorithm used for stem interpolation. */
    private int rerunMax = 5; // number of reruns for tiling
    private boolean connectJunctionsMode = true; /** If true, add stems between junctions. */
    private boolean forbiddenMode = false; /** If true, place each junction only once. */
    private boolean fuseJunctionStrandsMode = false; /** Fuse strands of junctions? Only makes sense when capped with kissing loops. */
    private double fuseStrandCutoff = 12.0; /** Maximum allowed distance between residues. */
    private FitParameters junctionFitParameters = new FitParameters(RMS_LIMIT_DEFAULT, ANGLE_LIMIT_DEFAULT); /** dist, angle, iter */
    private int junctionMinOrder = 1; /** Minimum order of junctions to be placed. User option */
    private TilingStatistics junctionStatistics = new TilingStatistics(); /** Statistics relating to junction placement. */
    private StrandJunctionDB kissingLoopDB; /** first dimension: order of junction (how many connections). Second dim: counter for database entries */
    private boolean kissingLoopJunctionMode = false; /** If true, place kissing loops at vertices. */
    private boolean kissingLoopMode = false; /** If true, use kissing loops in between stems. */
    private TilingStatistics kissingLoopStatistics = new TilingStatistics(); /** Statistics relating to kissing loop placement. */
    private Object3D nucleotideDB; /** Database of nucleotides. */
    private int optimizeJunctionChoiceIterMax = 0; // 10; /** Number of iterations of junction choice optimization. */
    private FitParameters stemFitParameters = new FitParameters(RMS_LIMIT_DEFAULT, 2.0 * ANGLE_LIMIT_DEFAULT); /** dist, angle, iter */
    private TilingStatistics stemStatistics = new TilingStatistics(); /** Statistics relating to stem placement. */
    private StrandJunctionDB strandJunctionDB; /** first dimension: order of junction (how many connections). Second dim: counter for database entries */
    private double scaleFactor = 1.0; // 0.92;
    private boolean symmetryMode = false; /** If true, use always the same junction. */

    /* CONSTRUCTORS */

    /** Default constructor */
    public FragmentGridTiler() {}

    /**
     * constructor needs a defined database of junctions
     *
     * @param strandJunctionDB Database of junctions.
     * @param kissingLoopDB Database of kissing loops.
     */
    public FragmentGridTiler(StrandJunctionDB strandJunctionDB, StrandJunctionDB kissingLoopDB) {
	this.strandJunctionDB = strandJunctionDB;
	this.kissingLoopDB = kissingLoopDB;
    }

    /** Copy constructor */
    public FragmentGridTiler(FragmentGridTiler other) {
	copy(other);
    }

    /**
     * For a junction compute a score depending on its fitting
     * error. Zero score corresponds to perfect fit. Also takes helix
     * fitting problems into account.
     *
     * @param placedJunctions The Junctions that were placed on the graph.
     * @param n The index of the junction we're computing the score of.
     * @param vertexObjectSet The graph.
     * @param links The Links in the graph.
     */
    double computeJunctionFitScore(Object3DSet placedJunctions,
				   int index,
				   Object3DSet vertexObjectSet,
				   LinkSet links) {
	double result = 0.0; //Create variable "result".

	// estimate angle error
	// TODO

	// estimate distance error
	// TODO

	// sum over helix errors
	double stemFitTerm = 0.0; // Penalty score.
	if (placedJunctions.get(index) == null) { // If the Junction was not placed,
	    stemFitTerm += NO_JUNCTION_PLACED_PENALTY;
	}
	else {
	    String baseName = new String("DummyStems");
	    Object3DLinkSetBundle stemBundle = new ConnectingStemGenerator(this, 
									   placedJunctions,
									   baseName,
									   vertexObjectSet, 
									   links,
									   // false, // TODO : take symmetries into account
									   index,
									   ConnectingStemGenerator.APPEND_NONE).generate(); // do not append strands
	    Object3DSet stemSet = Object3DTools.collectByClassName(stemBundle.getObject3D(), "RnaStem3D");
	    int stemFitCount = 0;
	    assert stemSet.size() > 0; // TODO : take out later, just for debugging
	    for (int i = 0; i < stemSet.size(); ++i) {
		if (stemSet.get(i) instanceof RnaStem3D) {
		    Properties properties = stemSet.get(i).getProperties();
		    assert properties != null;
		    String fitScoreString = properties.getProperty("fit_score");
		    assert fitScoreString != null;
		    assert fitScoreString.length() > 0;
		    double fitScore = Double.parseDouble(fitScoreString); // Will be zero if the fit is perfect.
		    stemFitTerm += fitScore; // Add fitScore to penalty measure.
		    stemFitCount++;
		}
		else {
		    log.severe("Strange class found: " + stemSet.get(i).getClassName() + " " + stemSet.get(i).getName() );
		    assert false;
		}
	    }
	    if (stemFitCount > 0) { // If there are stems,
		stemFitTerm /= stemFitCount; // Divide penalty measure by the number of stems.
	    }
	    else {
		stemFitTerm += NO_STEM_PLACED_PENALTY;
	    }
	}
	result += stemFitTerm; //Add the penalty count to the result.
	return result;
    }

    /**  */
    @Test public void testComputeJunctionFitScore() {
    }
    
    /**
     * Each junction gets a score depending on its fitting
     * error. Zero score corresponds to perfect fit. Also takes helix
     * fitting problems into account
     *
     * @param placedJunctions The Junctions that were placed on the graph.
     * @param vertexObjectSet The graph.
     * @param links The Links in the graph.
     */
    double[] computeJunctionFitScores(Object3DSet placedJunctions,
				      Object3DSet vertexObjectSet,
				      LinkSet links) {
	assert placedJunctions.size() > 0;
	double[] result = new double[placedJunctions.size() ];
	for (int i = 0; i < result.length; ++i) {
	    result[i] = computeJunctionFitScore(placedJunctions, i, vertexObjectSet, links);
	    log.fine("Fit score of junction " + (i+1) + " " + result[i]);
	}
	return result;
    }

    /** */
    @Test public void testComputeJunctionFitScores() {
    }

    /** 
     * Generates set of normalized vectors : directions from vertex
     * to its neighbors (using permutation perm).
     *
     * @param vertexPos
     * @param neighbors
     * @param perm
     */
    private Vector3D[] generateDirectionVectors(Vector3D vertexPos,
						Object3DSet neighbors,
						int[] perm) {
	Vector3D[] result = new Vector3D[neighbors.size() ];
	for (int i = 0; i < neighbors.size(); ++i) {
	    result[i] = neighbors.get(perm[i] ).getPosition().minus(vertexPos);
	    assert result[i].length() > 0.0;
	    result[i].normalize();
	}
	return result;
    }

    /**
     * generates set of normalized vectors: 
     * directions from vertex to its neighbors
     * (not using permutation perm)
     */
    private Vector3D[] generateDirectionVectors(Vector3D vertexPos,
						Object3DSet neighbors) {
	Vector3D[] result = new Vector3D[neighbors.size() ];
	for (int i = 0; i < neighbors.size(); ++i) {
	    result[i] = neighbors.get(i).getPosition().minus(vertexPos);
	    assert result[i].length() > 0.0;
	    result[i].normalize();
	}
	return result;
    }

    /** */
    @Test public void testGenerateDirectionVectors() {
    }

    /** generates set of normalized vectors pointing from vertex to neighbor objects */
    static Vector3D[] generateJunctionBranchDirections(StrandJunction3D junction) {
	Vector3D[] result = new Vector3D[junction.getBranchCount() ];
	for (int i = 0; i < junction.getBranchCount(); ++i) {
	    result[i] = junction.getBranch(i).getDirection();
	}
	return result;
    }

    /** */
    @Test public void testGenerateJunctionBranchDirections() {
    }

    /** generates set of normalized vectors pointing from vertex to neighbor objects */
    private Vector3D[] generateJunctionBranchVectors(StrandJunction3D junction) {
	Vector3D[] result = new Vector3D[junction.getBranchCount() ];
	for (int i = 0; i < junction.getBranchCount(); ++i) {
	    result[i] = junction.getBranch(i).getPosition();
	}
	return result;
    }

    /** */
    @Test public void testGenerateJunctionBranchVectors() {
    }

    /**
     * Uses the user provided data and rotates the junctions that have already been placed.
     *
     * @param vertexSet The vertexes of the graph.
     * @param objectTree The graph data.
     * @param placedJunctions The set of junctions that were already placed on the graph.
     */
    private Object3DSet generateSymmetries(Object3DSet vertexSet,
					   Object3D objectTree,
					   Object3DSet placedJunctions ) { 
	log.info("Started generateSymmetries!");
	assert vertexSet != null;
	assert objectTree != null;
	assert placedJunctions != null;
	assert placedJunctions.size() > 0;
	Object3DSet plJunc = new SimpleObject3DSet();
	for (int i = 0; i < placedJunctions.size(); i++) {
	    if (placedJunctions.get(i) != null) {
		StrandJunction3D tmpStr = (StrandJunction3D)placedJunctions.get(i);
		plJunc.add( (StrandJunction3D)tmpStr.cloneDeep() );
	    }
	}
	assert plJunc.size() > 0;
	StrandJunction3D tmpObject = null;
	Object3DSet result = new SimpleObject3DSet();

	log.severe("getNumSymmetries method was removed!! Outdated method. CEV");
	/*
	  assert objectTree.getNumSymmetries() > 0;
	  for (int i = 0; i < objectTree.getNumSymmetries(); i++) {
	  Symmetry tmpSym = objectTree.getSymmetry(i);
	  Symmetry symmetry = (Symmetry)tmpSym.cloneDeep();
	  assert symmetry != null;
	  Vector3D start = symmetry.getStartPoint().getVector3D();
	  assert start != null;
	  Vector3D end = symmetry.getEndPoint().getVector3D();
	  assert end != null;
	  Vector3D axis = end.minus(start);
	  assert axis != null;
	  log.info("Symmetry of type : " + symmetry.getType() + 
	  " found with " + symmetry.getNumJunctions() + 
	  " junctions.");
	  assert symmetry.getNumJunctions() > 0;
	  for (int j = 0; j < symmetry.getNumJunctions(); j++) {
	  assert symmetry.getJunction(j).getNumPoints() > 0;
	  Junction tmpJunc = (Junction)symmetry.getJunction(j);
	  Junction junction = (Junction)tmpJunc.cloneDeep();
	  assert junction.getNumPoints() > 0;
	  assert junction != null;
	  for (int k = 0; k < plJunc.size(); k++) {
	  if (plJunc.get(k) != null) {
	  StrandJunction3D tmpStr = (StrandJunction3D)plJunc.get(k);
	  StrandJunction3D placed = (StrandJunction3D)tmpStr.cloneDeep();
	  assert placed != null;
	  if (junction.containsPoint(placed.getRelativePosition() ) ) {
	  for (int m = 1; m < symmetry.getType(); ++m) { // Repeat for the type number because you have to rotate it (type-1) times.
	  if (m == 1) { // If it is the first rotation,
	  tmpObject = (StrandJunction3D)(placed.cloneDeep() );
	  }
	  else {
	  tmpObject = (StrandJunction3D)(tmpObject.cloneDeep() );
	  }
	  assert tmpObject != null;
	  tmpObject.rotate(start, axis, (Math.PI * 2) / (symmetry.getType() ) );
	  assert tmpObject != null;
	  Object3D o = new SimpleObject3D(symmetry.getCenterPoint(junction).getVector3D() );
	  assert o != null;
	  vertexSet.add( (SimpleObject3D)o.cloneDeep() );
	  // TODO : links between symmetry-generated verteces missing
	  assert tmpObject != null;
	  result.add( (StrandJunction3D)(tmpObject.cloneDeep() ) );
	  }
	  }
	  }
	  }
	  }
	  }
	*/
	log.info( "Ended generateSymmetries!" );
	return result;
    }
    
    /** */
    @Test public void testGenerateSymmetries() {
    }

    /** generates sets of links corresponding to base pairs of connector stems */
    static LinkSet generateJunctionLinks(Object3DSet junctions) {
	LinkSet result = new SimpleLinkSet();
	for (int i = 0; i < junctions.size(); ++i) {
	    if (junctions.get(i) != null) {
		result.merge(StrandJunctionTools.generateJunctionLinks((StrandJunction3D)junctions.get(i)));
	    }
	}
	return result;
    }

    public int countInteractionLinks(LinkSet links) {
	int count = 0;
	for (int i = 0; i < links.size(); ++i) {
	    if (links.get(i) instanceof InteractionLink) {
		++count;
	    }
	}
	return count;
    }

    /** returns set of strand that trace the given set of objects */
    @SuppressWarnings(value="unchecked")
    public Object3DLinkSetBundle generateTilingRun(Object3D objectTree,
						   LinkSet links, 
						   String baseName,
						   char defaultSequenceChar,
						   boolean onlyFirstPathMode) {
	// PRECONDITIONS:
	// copy(this);
	assert objectTree != null;
	assert objectTree.size() > 0;
	assert links != null;
	assert links.size() > 0;
	assert baseName != null;
	assert strandJunctionDB != null;
	reset(); // reset stored best solutions
	log.severe("CEV: hasSymmetries method was removed!! Outdated");
	log.info("Starting FragmentGridTiler.generateTilingRun");	
	// add to final tree:
	Object3D resultTree = new SimpleObject3D();
	resultTree.setName(baseName);
	resultTree.setPosition(objectTree.getPosition() );
	LinkSet newLinks = new SimpleLinkSet();
	Object3DSet vertexSet = new SimpleObject3DSet(objectTree); // Generate object set
	// remove objects that are not linked
 	for (int i = vertexSet.size()-1; i >= 0; --i) {
 	    if (LinkTools.countLinks(vertexSet.get(i), links) == 0) {
		vertexSet.remove(vertexSet.get(i) ); // TODO : fix later
 	    }
 	}
	// choose and place framents for all vertices ("junctions")
	Object3DSet junctions = placeJunctions(vertexSet, links, baseName);
	int numPlJunc = 0;
	for (int i = 0; i < junctions.size(); i++) {
	    if (junctions.get(i) != null)
		numPlJunc++;
	}
	if (numPlJunc < 1) {
	    log.warning( "No junctions were placed so generateTiling was stopped" );
	    return null; //stop generateTiling if no junctions were placed
	}
	
	if (optimizeJunctionChoiceIterMax > 0) {
	    double initialScore = optimizeJunctionChoice(junctions, vertexSet,
							 links, baseName, 1);
	    log.info("Starting to optimize chosen junctions! Initial score: "
		     + initialScore);
	    double optScore = optimizeJunctionChoice(junctions, vertexSet,
						     links, baseName,
						     optimizeJunctionChoiceIterMax);
	    // TODO : optScore is not used yet!
	    log.info("Finished optimize junctions with score: " + optScore + 
		     " ( initial score was: " + initialScore + " )");
	}

	Object3DSet symmetrySet;
	/* //CEV: symmetries removed: PointSetReader3 now!! TODO: update
	  if (symmetries) {
	  symmetrySet = generateSymmetries(vertexSet, objectTree, junctions);
	  log.info("Number of generated symmetric junctions: " + symmetrySet.size() );
	  for (int i = 0; i < symmetrySet.size(); i++) {
	  assert (symmetrySet.get(i) instanceof StrandJunction3D);
	  junctions.add(symmetrySet.get(i) );
	  }
	  // update vertexSet and linkset.
	  newLinks.removeAndAdd(objectTree, vertexSet); // Removes old links and replaces them with new.
	  vertexSet.removeExtras(objectTree); // Removes the points that appear on the lines created during symmetries
	  }
	  else {
	  log.fine("no symmetries detected.");
	  }
	*/
	// generate links corresponding to hydrogen bonds of junction "connectors"
	LinkSet junctionLinks = generateJunctionLinks(junctions);
	newLinks.merge(junctionLinks);
	// generate connecting stems
	if (connectJunctionsMode) {
	    log.info("Connect junctions mode is on");
	    // interpolate strands between fragments
	    ConnectingStemGenerator connectingStemGenerator =
		new ConnectingStemGenerator(this, junctions, baseName, vertexSet, links,
					    GENERATE_ALL_STEMS_INDEX, this.appendStrandsMode);
	    assert connectingStemGenerator != null;
	    Object3DLinkSetBundle stemBundle = connectingStemGenerator.generate();
	    Object3D stemRoot = stemBundle.getObject3D();
	    // add property for sequence optimization:
	    // possible values by convention: sequence_status : adhoc, optimized, fragment
	    Object3DTools.setRecursiveProperty(stemRoot, SequenceStatus.name, SequenceStatus.adhoc, "Nucleotide3D");
	    log.info("Generated " + stemRoot.size() + " connecting stems with " 
		     + stemBundle.getLinks().size() + " links.");
	    resultTree.insertChild(stemRoot);
	    // assert stemBundle.getLinks().size() > 0;
	    int interactionLinkCount = countInteractionLinks(stemBundle.getLinks());
	    log.info("Number of interaction links found: " + interactionLinkCount);
	    newLinks.merge(stemBundle.getLinks() );
	}
	else {
	    log.info("No interpolating stems generated because of user flag");
	}

	// fuse strands
	if (fuseJunctionStrandsMode) {
	    log.info("Fuse junction strands mode is on");
	    for (int i = 0; i < junctions.size(); ++i) {
		StrandJunction3D junction = (StrandJunction3D)(junctions.get(i) );
		if (junction != null) {
			StrandJunctionTools.fuseStrands(junction, fuseStrandCutoff);
		}
	    }
	}
	
	assert junctions != null;

	// TODO : determine cutting points for strands
	// 	log.fine("NumJunctions: " + junctions.size());
	// 	for(int i = 0; i < junctions.size(); i++ ) {
	// 	    if(junctions.get(i) != null) 
	// 	}
	// 	resultTree.insertChild(junctions.get(5)); //just for testing //original junction is placed correctly 10-26-06

	for (int i = 0; i < junctions.size() ; ++i) {
	    if (junctions.get(i) != null) {
		log.finest("Adding junction " + i + ": " + junctions.get(i));
		resultTree.insertChild(junctions.get(i)  );
	    }
	}
	// TODO : careful: using workaround of ignoring StrandJunction3D, BranchDescriptor3D and KissingLoop3D
	//Object3DTools.balanceTree(resultTree); 
	// compute total fraction of placement:
	double totalFraction = 0.5 * (getJunctionStatistics().getCoverage() + getStemStatistics().getCoverage())
	    ; //  + getKissingLoopStatistics().getCoverage();
	resultTree.setProperty("placement_total_fraction", "" + totalFraction);
	Object3DLinkSetBundle result = new SimpleObject3DLinkSetBundle(resultTree, newLinks);
	
	// move best found junctions to old junctions:
	for (int i = 0; i < symmetryJunctions.length; ++i) {
	    if (oldSymmetryJunctions[i] != null) {
		oldSymmetryJunctions[i].add(symmetryJunctions[i]);
	    }
	    else {
		oldSymmetryJunctions[i] = new ArrayList<StrandJunction3D>(); // used for storing best junctions of each order
	    }
	}

	log.info("Finished generateTilingRun!");
	String statText = "Junctions: " + getJunctionStatistics().toString()
	    + ENDL + ENDL + "Stems: " + getStemStatistics().toString() 
	    + ENDL + ENDL + "Kissing Loops: " + getKissingLoopStatistics().toString();
	NanoTiler.tilingStatisticsString = statText;
	//	JOptionPane.showMessageDialog(null, statText , "Junction Details", JOptionPane.PLAIN_MESSAGE);

	log.info(statText);
	log.info("Totall coverage fraction of this run: " + totalFraction);
	// taken out mesage dialog : model should know nothing about gui!
	// JOptionPane.showMessageDialog(null, statText, "Statistics", JOptionPane.PLAIN_MESSAGE);

	//	log.fine("Average angle: " + tilingStatistics.getAverageAngle() + " degrees" + NEWLINE + "Average distance: " + tilingStatistics.getAverageDistance() + " angstroms");

	//	JOptionPane.showMessageDialog(null,"Average angle: " + tilingStatistics.getAverageAngle() + " degrees." + NEWLINE + "Average distance: " + tilingStatistics.getAverageDistance() + " angstroms.", "Final Averages", JOptionPane.PLAIN_MESSAGE);
	assert result != null;    
	return result;
    }

    /** returns set of strand that trace the given set of objects */
    public Object3DLinkSetBundle generateTiling(Object3D objectTree,
						LinkSet links, 
						String baseName,
						char defaultSequenceChar,
						boolean onlyFirstPathMode) {
	Object3DLinkSetBundle result = null;
	double bestScore= 0;
	double placeLim = 0.999999;
	Object3DSet objectSet = new SimpleObject3DSet(objectTree);

	// Vector3D[] positions = new Vector3D[objectSet.size()];
	// for (int i = 0; i < objectSet.size(); ++i) {
	// Object3D object = objectSet.get(i);
	// positions[i] = new Vector3D(object.getPosition()); // save positions
	// }
	log.info("Starting FragmentGridTiler.generateTiling with options: " + toString());
	// TODO: apparently scaling has no implementation anymore...
	for (int k = -1; k <= 1; ++k) {
	    if ((scaleFactor == 1.0 ) && (k != 0)) {
		continue;
	    }
	    // if (scaleFactor != 1.0) {
	    // double scaleFactorUsed = 1.0;
	    // if (k == -1) {
	    // scaleFactorUsed = scaleFactor;
	    // }
	    // else if (k == 0) {
	    // scaleFactorUsed = 1/scaleFactor;
	    // }
	    // else if (k == 1) {
	    // scaleFactorUsed = 1.0;
	    // }
	    // log.info("Scaling object graph tree with factor: " + scaleFactorUsed);
	    // for (int i = 0; i < objectSet.size(); ++i) {
	    // objectSet.get(i).setPosition(positions[i].mul(scaleFactorUsed)); // save positions
	    // }
	    // }
	    for (int i = 0; i < rerunMax; ++i) {
		log.info("Starting tiling run " + (i+1));
		Object3DLinkSetBundle bundle = generateTilingRun(objectTree, links, baseName,
								 defaultSequenceChar, onlyFirstPathMode);
		if (bundle == null) {
		    log.warning("No tiling could be generated at pass " + (i+1));
		    continue;
		}
		assert bundle != null;
		String placeFracString = bundle.getObject3D().getProperty("placement_total_fraction");
		assert placeFracString != null;
		assert placeFracString.length() > 0;
		double score = Double.parseDouble(placeFracString);
		log.info("Placement score of run " + (i+1) + " " + score + " ( " + placeFracString + " )");
		if ((result == null) || (score > bestScore) ) {
		    result = bundle;
		    bestScore = score;
		    log.info("Adopting so far best solution!");
		}
		if (bestScore >= placeLim) { // 100% junctions and stems placed!
		    log.info("Quiting rerun loop, because complete placement found!");
		    break;
		}
	    }
	    if (bestScore >= placeLim) {
		break;
	    }
	}
	return result;

    }

    /** TODO */
    @Test public void testGenerateTiling() {
    }

    public int getAppendStrandsMode() { return this.appendStrandsMode; }
    public int getAxialSteps() { return this.axialSteps; }
    public boolean getJunctionPlaceMode() { return junctionPlaceMode;}
    public int getConnectionAlgorithm() { return this.connectionAlgorithm; }
    public boolean getConnectJunctionsMode() { return connectJunctionsMode; }
    public boolean getForbiddenMode() { return forbiddenMode;}
    public boolean getFuseJunctionStrandsMode() { return fuseJunctionStrandsMode;}
    public double getFuseStrandCutoff() { return this.fuseStrandCutoff; }
    public FitParameters getJunctionFitParameters() { return this.junctionFitParameters; }
    public int getJunctionMinOrder() { return junctionMinOrder; }
    public TilingStatistics getJunctionStatistics() { return junctionStatistics; } /** returns junction tiling statistics */
    public StrandJunctionDB getKissingLoopDB() { return kissingLoopDB; }
    public boolean getKissingLoopJunctionMode() { return kissingLoopJunctionMode; }
    public boolean getKissingLoopMode() { return kissingLoopMode; }
    public TilingStatistics getKissingLoopStatistics() { return kissingLoopStatistics; } /** returns kissing loop tiling statistics */
    public Object3D getNucleotideDB() { return nucleotideDB; }
    public int getOptimizeJunctionChoiceIterMax() { return this.optimizeJunctionChoiceIterMax; }
    public int getRerunMax() { return this.rerunMax; }
    public FitParameters getStemFitParameters() { return this.stemFitParameters; }
    public TilingStatistics getStemStatistics() { return stemStatistics; } /** returns stem tiling statistics */
    public StrandJunctionDB getStrandJunctionDB() { return strandJunctionDB; }
    public boolean getSymmetryMode() { return symmetryMode; }
    
    /** optimizes (exchange) junctions placed so far */
    private double optimizeJunctionChoice(Object3DSet placedJunctions,
					  Object3DSet vertexObjectSet, 
					  LinkSet links, 
					  String baseName,
					  int iterMax) {
	// give each junction a fitting score
	double[] fitScores = computeJunctionFitScores(placedJunctions, vertexObjectSet, links);
	double totScore = DoubleArrayTools.computeSum(fitScores) / fitScores.length;
	int iter = 0;
	double bestScore = totScore;
	double optimizeScoreLimit = 0.1;
	int randomIterMax = 10;
	log.fine("Starting FragmentGridTiler.optimizeJunctionChoice with total fitting score: " + bestScore);
	while ((iter++ < iterMax) && (bestScore > optimizeScoreLimit)) {
	    // choose junction to be optimized
	    int choice = DoubleArrayTools.chooseRouletteWheel(fitScores); // the higher (worse) the score, the more likely a junction will be chosen to be changed next
	    String name = baseName + JUNCTION_ENDING + (choice+1);
	    StrandJunction3D newJunction = placeJunction(vertexObjectSet.get(choice),
							 links, baseName, randomIterMax);
	    StrandJunction3D savedJunction = (StrandJunction3D)(placedJunctions.get(choice) );
	    placedJunctions.set(choice, newJunction);
	    fitScores = computeJunctionFitScores(placedJunctions, vertexObjectSet, links);
	    double newScore = DoubleArrayTools.computeSum(fitScores) / fitScores.length;
	    log.fine("New optimized junction computed. Current fitting score: " + newScore);
	    if (newScore <= bestScore) { // TODO : allow for Metropolis criterion
		bestScore = newScore;
		log.fine("New optimized junction found. Best fitting score: " + bestScore);
	    }
	    else {
		// undo step
		placedJunctions.set(choice, savedJunction);
	    }
	}
	log.fine("Finished FragmentGridTiler.optimizeJunctionChoice with total fitting score: " + bestScore);
	return bestScore;
    }

    /** TODO */
    @Test public void testOptimizeJunctionChoice() {
    }

    /** uses stored set of known junctions to place them for each node
     * @param vertex : graph vertex on which junction is to be placed
     * @param links   : graph edges for vertex
     * @param baseName : name of new junction to be used
     * @param randomTrialsMax if greater zero, run in "random mode": one junction is chosen randomly (trying this many times) */
    StrandJunction3D placeJunction(Object3D vertex,
				   LinkSet links,
				   String baseName, int randomTrialsMax) {
	//	updateModes(this);
	// copy(this);

	assert vertex != null;
	assert links != null;
	assert baseName != null;
	assert (strandJunctionDB != null);
	assert (strandJunctionDB.isValid());
	
	log.fine("started FragmentGridTiler.placeJunction!");
	
	// determine how many links point at vertex (order of vertex)
	//	if( !symmetries ) {
	Object3DSet neighbors = LinkTools.findNeighbors(vertex, links); // TODO: Get neighbors from points: faster
	log.fine("neighbors: " + neighbors);
	int order = neighbors.size(); // number of neighbors
	Vector3D[] neighborVectors = new Vector3D[neighbors.size() ];
	for (int i = 0; i < neighborVectors.length; ++i) {
	    neighborVectors[i] = new Vector3D(vertex.getPosition() );
	}
	// used plausibility check if junction is possible at all:
	Vector3D[] neighborDirectionsOrig = generateDirectionVectors(vertex.getPosition(), neighbors);
// 	}
// 	else { //if there are symmetries
// 	}
	
	// Vector3D[] neighborVectors = generateEdgeVectors(vertex, neighbors);
	// Vector3D[] neighborVectors = generateEdgeVectorsWithDirectionsNoOrigin(vertex, neighbors);
	
	log.fine("trying to place junction of order: " + order);
	MCSuperposeCollinear superposer = new MCSuperposeCollinear(); // new MinSuperpose 
	superposer.setAngleWeight(junctionFitParameters.getAngleWeight() );
	superposer.setAngleLimit(junctionFitParameters.getAngleLimit() );
	
	// 	if (superposeCenterMode) {
	// 	    log.fine("Superpose Center Mode is on!");
	// 	    superposer.setCenterId(0); // first positions must match!!! not center of gravity
	// 	}
	
	// loop over different junctions:
	int bestId = 0;
	double bestRms = 999.9;
	// SuperpositionResult bestSuperposition = null;
	StrandJunction3D[] strandJunctions = new SimpleStrandJunction3D[0];
	if (junctionPlaceMode) {
	    strandJunctions = strandJunctionDB.getJunctions(order);
	    if (kissingLoopJunctionMode) {
		StrandJunction3D[] temp = new StrandJunction3D[strandJunctions.length];
		for (int i = 0; i < strandJunctions.length; i++) {
		    temp[i] = strandJunctions[i];
		}
		strandJunctions = new StrandJunction3D[temp.length + 
						       kissingLoopDB.getJunctions(order).length];
		for (int i = 0; i < temp.length; i++) {
		    strandJunctions[i] = temp[i];
		}
		for (int i = 0; i < kissingLoopDB.getJunctions(order).length; i++) {
		    strandJunctions[i + temp.length] = kissingLoopDB.getJunctions(order)[i];
		}
	    }
	}
	else if (kissingLoopJunctionMode) {
	    strandJunctions = kissingLoopDB.getJunctions(order);
	}
	if (strandJunctions.length == 0) { //no junctions of correct order in db
	    log.warning("Junction was not placed because there were no junctions of the correct order in the DB.");
	    return null;
	}
	StrandJunction3D bestJunction = null;
	assert (strandJunctions != null) && (strandJunctions.length > 0);
	// log.fine("number of database junctions with that order: " + strandJunctions.length);
	int randomTrials = 0;
	Random random = Randomizer.getInstance();
	for (int ii = 0; ii < strandJunctions.length; ++ii) { //for each junction in the DB
	    int i = ii;
	    if (randomTrialsMax > 0) { 
		if ((bestJunction != null) || (randomTrials++ > randomTrialsMax)) {
		    log.fine("Quitting loop over junctions because of random mode");
		    break; // in "random mode" choose on random junction, then quit
		} //End if.
		i = random.nextInt(strandJunctions.length); 
		log.finest("Working on database randomly chosen junction " + (i+1));
	    } //End if.
	    else { //If not in random mode, 
		log.info("Working on database junction " + (i+1));
	    }
	    Vector3D[] junctionDirectionsOrig = generateJunctionBranchDirections(strandJunctions[i]);
	    if (junctionDirectionsOrig.length > 1) {
		assert (junctionDirectionsOrig.length == neighborDirectionsOrig.length);
		double plausScore = Vector3DTools.computeVectorFieldSimilarityScore(junctionDirectionsOrig, neighborDirectionsOrig);
		if (plausScore > (2.0 * junctionFitParameters.getAngleLimit())) {
		    log.warning("Skipping junction " + (i+1) + " because of plausibility score: " + plausScore + " " + junctionFitParameters.getAngleLimit());
		    continue; // skip junction, because angles are too different
		} //End if.
	    } //End if.
	    else
		log.warning("junction of order : " + junctionDirectionsOrig.length + " detected!");
	    if (symmetryMode) { //If user wants the same junction on each vertice,
		log.fine("Symmetry mode is on");
		StrandJunction3D currJunction = strandJunctions[i];
		if ((order < symmetryJunctions.length) && (symmetryJunctions[order] != null) && (currJunction != symmetryJunctions[order])) {
		    continue; // ignore all junctions that are not the first junction
		} 
		// check if currJunction is in oldSymmetryJunctions:
		List oldJunctions = oldSymmetryJunctions[order];
		if (oldJunctions != null) {
		    boolean forbiddenFound = false;
		    for (int j = 0; j < oldJunctions.size(); ++j) {
			StrandJunction3D oldJunction = (StrandJunction3D)(oldJunctions.get(j) );
			if (currJunction == oldJunction) {
			    forbiddenFound = true;
			    break;
			} //End if.
		    } //End for.
		    if (forbiddenFound && forbiddenMode)
			continue; // ignore this junction
		} //End if.
		else
		    oldSymmetryJunctions[order] = new ArrayList<StrandJunction3D>();
	    } //End if(symmetryMode).

 	    // Vector3D[] junctionVectors = generateEdgeVectors(strandJunctions[i]);
 	    // Vector3D[] junctionVectors = generateEdgeVectorsWithDirectionsNoOrigin(strandJunctions[i]);
	    
	    // Vector3D[] neighborVectors = generateRawVectors(vertex, neighbors, perm);
	    // Vector3D[] junctionVectors = generateLongJunctionVectors(tmpJunction, neighborVectors);
	    
	    PermutationGenerator permutator = new PermutationGenerator(neighbors.size());
	    // int[] perm = permutator.getNext();
	    int permCount = 0;
	    // loop over all possible permutation of vertex neighbors:
	    while (permutator.hasMore()) {
		int[] perm = permutator.getNext();
		++permCount;

		try {
		    //   		Vector3D[] neighborVectors = generateEdgeVectors(vertex, neighbors, perm);
		    //   		Vector3D[] junctionVectors = generateEdgeDirectionVectors(tmpJunction);
		    // Vector3D[] junctionVectors = generateEdgeVectors(tmpJunction);
		    
		    StrandJunction3D tmpJunction = (StrandJunction3D)(strandJunctions[i].cloneDeep());
		    Vector3D[] junctionDirections = generateJunctionBranchDirections(tmpJunction);
		    Vector3D[] junctionVectors    = generateJunctionBranchVectors(tmpJunction);
		    assert junctionDirections.length == junctionVectors.length;

		    Vector3D[] neighborDirections = generateDirectionVectors(vertex.getPosition(), neighbors, perm);
		    if (junctionVectors == null)
			continue; // extrapolated vectors could not be generated
		    assert neighborVectors.length == junctionVectors.length;
		    
		    // 		    Object3D debugChild = new SimpleObject3D();
		    // 		    debugChild.setPosition(new Vector3D(junctionVectors[1]));
		    // 		    debugChild.setName("junctionVector_child");
		    // 		    Vector3D debugVector = debugChild.getPosition();
		    // 		    log.fine("original debug child: " + debugChild);
		    if ( order >= junctionMinOrder ) {
			SuperpositionResult superResult = superposer.superpose(neighborVectors, neighborDirections, 
									       junctionVectors, junctionDirections);
			assert superResult != null;
			double rms = superResult.getRms(); // evaluateFit2(tmpJunction, vertex, neighbors, links, perm);
			//			if (rms > junctionSuperposeLimit) {
			if (rms > junctionFitParameters.getRmsLimit() ) {
			    log.info("Junction could not be placed at all, because fitting error is too large: " + rms);
			    continue; // too bad fit! Ignore!
			}
			assert tmpJunction.checkChildrenUnique();
			
			// apply rotation and translation:
			// tmpJunction.insertChild(debugChild);
			superResult.applyTransformation(tmpJunction);
			
			// check if rotation of tree works properly:
			
			// 		    assert tmpJunction.getChild(tmpJunction.size()-1).getPosition().distance(superResult.returnTransformed(debugVector))
			// 			< 0.1;
			// 		    tmpJunction.removeChild(debugChild);
			// 		double centerDist = tmpJunction.getPosition().distance(vertex.getPosition());
			// 		// assert neighborVectors[0].distance(tmpJunction.getPosition()) < 0.1;
			// 		if (centerDist > junctionCenterDistMax) {
			// 		    log.fine("Junction center was too far from vertex: " + centerDist);
			// 		    continue;
			// 		}
			

			if ( (rms < bestRms) || (bestJunction == null) ) {
			    if (permCount > 0) {
				log.fine("Result of permutated superposition: " + rms + " better than " + bestRms
					 + " " + (i+1) + " " + permCount);
			    }
			    else {
				log.fine("Result of first superposition: " + rms + " better than " + bestRms
					 + " " + (i+1) + " " + permCount);
			    }
			    bestRms = rms;
			    bestId = i;
			    // bestSuperposition = superResult;
			    bestJunction = tmpJunction;
			}
		    }
		    else {
			log.warning("Junction was not placed because min order specified was greater");
		    }
		}
		catch (RuntimeException e) {
		    log.severe("RuntimeException in FragmentGridTiler.placeJunction: " + e.getMessage());
		}
	    }
	}
	if (bestJunction == null) {
	    log.info("Junction could not be placed at all!");
	    return null;
	}
	log.fine("Best fit score: " + bestRms + " modified junction: "
		 + bestJunction.getName() + " " + bestJunction.getBranchCount() );
	for (int j = 0; j < bestJunction.getBranchCount(); ++j) {
	    log.fine("Branch " + (j+1) + ": Pos: " + 
		     bestJunction.getBranch(j).getPosition() + 
		     " Dir: " + bestJunction.getBranch(j).getDirection());
	}
	if (symmetryMode && (symmetryJunctions[order] == null) ) {
	    symmetryJunctions[order] = strandJunctions[bestId];
	}
	// set fit score:
	Properties prop = bestJunction.getProperties();
	if (prop == null) {
	    prop = new Properties();
	}
	prop.setProperty("fit_score", ""+bestRms);
	bestJunction.setProperties(prop);
	// assert (bestJunction == null) || (bestRms <= rmsLimit);
	log.fine("finished placeJunction!");
	return bestJunction;
    }

    /** TODO */
    @Test public void testPlaceJunction() {
    }

    /** uses stored set of known junctions to place them for each node */
    private Object3DSet placeJunctions(Object3DSet objectSet, 
				       LinkSet links, 
				       String baseName ) {
	// PRECONDITIONS:
	assert objectSet != null;
	assert objectSet.size() > 0;
	assert links != null;
	assert links.size() > 0;
	assert baseName != null;
	assert (strandJunctionDB != null);
	log.fine("Started placeJunctions!");
	// generate new tiling statistics model
	this.junctionStatistics = new TilingStatistics();

	Object3DSet result = new SimpleObject3DSet();
// 	Object3D newStrands = new SimpleObject3D();
// 	newStrands.setName(baseName + "_strands");
	String[] usedJunctions; //An array of the junctions that were used.
	usedJunctions = new String[1];
	for (int i = 0; i < objectSet.size(); ++i) {
	    Object3D vertex = objectSet.get(i);
	    if (usedJunctions[usedJunctions.length - 1] != null) {
		String[] tempArray = new String[usedJunctions.length];
		tempArray = usedJunctions;
		
		usedJunctions = new String[usedJunctions.length + 1];
		for (int x = 0; x < tempArray.length ; x++) {
		    if (tempArray[x] != null) {
			    usedJunctions[x] = tempArray[x];
		    }
		}
	    }
	    String name = baseName + "_junc" + (i+1);
	    StrandJunction3D junction = new SimpleStrandJunction3D();
	    if (vertex.getProperty("rotateFrom") != null) {
		String rotFromString = vertex.getProperty("rotateFrom");
		assert rotFromString != null;
		log.info("interpreting rotFrom string: " + rotFromString + " of junction " + (i+1));
		int rotateFromIndex = Integer.parseInt(rotFromString);
		assert rotateFromIndex != i;
		StrandJunction3D tmpJunction = (StrandJunction3D)(result.get(rotateFromIndex));
		if (tmpJunction != null) {
		    StrandJunction3D rotateJunction = (StrandJunction3D)(tmpJunction.cloneDeep());
		    assert rotateJunction.size() > 0;
		    double x = Double.parseDouble(vertex.getProperty("axisX"));
		    double y = Double.parseDouble(vertex.getProperty("axisY"));
		    double z = Double.parseDouble(vertex.getProperty("axisZ"));
		    Vector3D center = new Vector3D(x, y, z);
		    double angle = Double.parseDouble(vertex.getProperty("angle"));
		    Vector3D v1 = vertex.getPosition().minus(center);
		    Vector3D v2 = rotateJunction.getPosition().minus(center);
		    Vector3D v3 = v1.cross(v2);
		    rotateJunction.rotate(center, v3, angle);
		    // System.out.println("vertex: " + vertex.getPosition() + " rot: " + rotateJunction.getPosition());
		    junction = rotateJunction;
		}
		else {
		    log.info("RotateFrom junction could not be found, maybe it could not be placed.");
		}
	    }
	    else {
		// if (vertex.getProperty("place") == "true") {
		log.fine("Placing junction " + name);
		junction = placeJunction(vertex, links, baseName, 0);
	    }

	    if (junction != null) {
		junction.setName(name + "_" + junction.getName());
		// StrandJunctionTools.placeStrands(junction, newStrands); // add strands 
		result.add(junction);
		junctionStatistics.updateCoverage(true);
		//find ANGLE
		if (junction.getProperties() != null) { // Properties are null if the junction was rotated.
		    String fitScoreString = junction.getProperties().getProperty("fit_score");
		    assert fitScoreString != null;
		    double fitScore = Double.parseDouble(fitScoreString);
		    junctionStatistics.updateAverageAngle(fitScore);
		    junctionStatistics.updateAverageDistance(fitScore);
		}
		//TODO: How to find angle???
		boolean junctionAlreadyUsed = false;
		for(int x = 0; x < usedJunctions.length; x++) {
		    if ( usedJunctions[x] == name )
			junctionAlreadyUsed = true;
		}
		if(!junctionAlreadyUsed)
		    usedJunctions[usedJunctions.length-1] = baseName;
	    }
	    else {
		log.fine("Junction " + name + " could not be generated!");
		result.add(junction); //  adds null to maintain one-to-one relationship
		// unaddedJunctions++;
		Object3DSet neighbors = LinkTools.findNeighbors(objectSet.get(i), links); // TODO: Get neighbors from points: faster
		if (neighbors.size() > 1) {
		    junctionStatistics.updateCoverage(false);
		}
		else {
		    log.fine("Cannot place hairpins yet!");
		}
	    }
	}
	assert result.size() == objectSet.size();
	// result.add(newStrands);
	log.fine("ended placeJunctions");
	return result;
    }

    /** TODO */
    @Test public void testPlaceJunctions() {
    }

    public void reset() {
	for(int i = 0; i < symmetryJunctions.length; ++i) {
	    symmetryJunctions[i] = null;
	}
    }

    /** TODO */
    @Test public void testReset() {
    }

    public void setAppendStrandsMode( int mode ) { this.appendStrandsMode = mode; }
    public void setAxialSteps(int steps) { this.axialSteps = steps; }
    public void setJunctionPlaceMode( boolean b ) { this.junctionPlaceMode = b;}
    public void setConnectionAlgorithm(int n) { this.connectionAlgorithm = n; }
    public void setConnectJunctionsMode(boolean b) { this.connectJunctionsMode = b; }
    public void setForbiddenMode( boolean b ) { this.forbiddenMode = b;}
    public void setFuseJunctionStrandsMode( boolean b ) {fuseJunctionStrandsMode = b;}
    public void setFuseStrandCutoff(double cutoff) { this.fuseStrandCutoff = cutoff; }
    public void setJunctionFitParameters(FitParameters prm) { this.junctionFitParameters = prm; }
    public void setJunctionMinOrder( int n ) { this.junctionMinOrder = n; }
    public void setJunctionStatistics(TilingStatistics stat) { this.junctionStatistics = stat; }
    public void setKissingLoopDB(StrandJunctionDB db) { this.kissingLoopDB = db; }
    public void setKissingLoopJunctionMode( boolean b ) { this.kissingLoopJunctionMode = b;}
    public void setKissingLoopMode( boolean b ) { this.kissingLoopMode = b; }
    public void setKissingLoopStatistics(TilingStatistics stat) { this.kissingLoopStatistics = stat; }
    public void setNucleotideDB(Object3D obj) { this.nucleotideDB = obj; }  /** sets nucleotide database used for fragment placing */
    public void setOptimizeJunctionChoiceIterMax(int n) { this.optimizeJunctionChoiceIterMax = n; } 
    public void setRerunMax(int n) { this.rerunMax = n; }
    public void setStemFitParameters(FitParameters prm) { this.stemFitParameters = prm; }
    public void setStemStatistics(TilingStatistics stat) { this.stemStatistics = stat; }
    public void setStrandJunctionDB(StrandJunctionDB db) { this.strandJunctionDB = db; }
    public void setSymmetryMode( boolean b ) { this.symmetryMode = b; }
    
    /** returns info string */
    public String toString() {
	StringBuffer buf = new StringBuffer();
	buf.append("FragmentGridTiler: modes: appendStrand: " + appendStrandsMode + " placeJunctions: " + junctionPlaceMode 
		   + " connectJunctions: " + connectJunctionsMode + " forbiddenMode: " + forbiddenMode 
		   + " fuseJunctionsStrands: " + fuseJunctionStrandsMode + " kissingLoopJunction: " + kissingLoopJunctionMode 
		   + " kissingLoop: " + kissingLoopMode + " symmetry: "  + symmetryMode + ENDL);
	buf.append("Junction fit parameters " + junctionFitParameters.toString() 
		   + " Stem fit parameters: " + stemFitParameters.toString());
	return buf.toString();
    }

    private void copy(FragmentGridTiler fgTiler) {
	setAppendStrandsMode(fgTiler.getAppendStrandsMode());
	setAxialSteps(fgTiler.getAxialSteps());
	setJunctionPlaceMode(fgTiler.getJunctionPlaceMode());
	setConnectionAlgorithm(fgTiler.getConnectionAlgorithm());
	setConnectJunctionsMode(fgTiler.getConnectJunctionsMode());
	setForbiddenMode(fgTiler.getForbiddenMode());
	setFuseJunctionStrandsMode(fgTiler.getFuseJunctionStrandsMode());
	setFuseStrandCutoff(fgTiler.getFuseStrandCutoff());
	setJunctionFitParameters(new FitParameters(fgTiler.junctionFitParameters));
	setJunctionMinOrder(fgTiler.getJunctionMinOrder());
	setJunctionStatistics(new TilingStatistics(fgTiler.getJunctionStatistics()));
	setKissingLoopDB(fgTiler.getKissingLoopDB()); // only shallow copy!?
	setKissingLoopJunctionMode(fgTiler.getKissingLoopJunctionMode());
	setKissingLoopMode(fgTiler.getKissingLoopMode());
	setKissingLoopStatistics(new TilingStatistics(fgTiler.getKissingLoopStatistics()));
	setNucleotideDB(fgTiler.getNucleotideDB());
	setOptimizeJunctionChoiceIterMax(fgTiler.getOptimizeJunctionChoiceIterMax());
	setRerunMax(fgTiler.getRerunMax());
	setStemFitParameters(fgTiler.getStemFitParameters());
	setStemStatistics(new TilingStatistics(fgTiler.getStemStatistics()));
	setStrandJunctionDB(fgTiler.getStrandJunctionDB());
	setSymmetryMode(fgTiler.getSymmetryMode());
    }

}
