package rnadesign.rnamodel;

import java.util.logging.Logger;

import generaltools.ApplicationBugException;
import numerictools.DoubleArrayTools;
import rnasecondary.Interaction;
import rnasecondary.InteractionType;
import rnasecondary.RnaInteractionType;
import rnasecondary.SimpleInteraction;
import rnasecondary.SimpleStem;
import rnasecondary.Stem;
import sequence.DnaTools;
import sequence.Residue;
import sequence.Sequence;
import numerictools.DoubleArrayTools;
import tools3d.LineShape;
import tools3d.Vector3D;
import tools3d.Vector3DTools;
import tools3d.objects3d.Link;
import tools3d.objects3d.LinkSet;
import tools3d.objects3d.Object3D;
import tools3d.objects3d.Object3DLinkSetBundle;
import tools3d.objects3d.Object3DSet;
import tools3d.objects3d.Object3DTools;
import tools3d.objects3d.SimpleLinkSet;
import tools3d.objects3d.SimpleObject3D;
import tools3d.objects3d.SimpleObject3DLinkSetBundle;
import tools3d.objects3d.SimpleObject3DSet;

import static rnadesign.rnamodel.PackageConstants.NEWLINE;
import static rnadesign.rnamodel.RnaConstants.C4_NORM_POS;

public class StemTools {

    public static double WATSON_CRICK_RESIDUE_CUTOFF = 18.0;
    public static double WATSON_CRICK_ATOM_CUTOFF = 5.0;
    private static final double threshold = 0.5;
    public static int STEM_LENGTH_MINIMUM = 3; // 4;
    public static boolean atomMode = true;
    public static double backboneAngleMin = 60.0 * (Math.PI/180.0);
    public static double backboneAngleMax = 180.0 * (Math.PI/180.0);
    public static double baseAngleMin = 60.0 * (Math.PI/180.0);
    public static double baseAngleMax = 180.0 * (Math.PI/180.0);
    // public static double drmsCutoff = 5.5;
    // public static double drmsCutoff = 2.0;
    public static double drmsCutoff = 1.5; // 3.0
    public static final int helixLength = 3; // 4; // 3; 
    public static final int range = 1; // careful, superseeded by helixLength?
    public static final HelixParameters idealHelixParameters = new HelixParameters();
    private static final Vector3D[] idealHelix = generateIdealHelixPatch(helixLength, idealHelixParameters);
    private static Vector3D[] scratchHelix = new Vector3D[2*helixLength];
    private static String helixRootAtomName = "C4*"; // TODO careful: does not work with Amber C4' name!
    private static Logger log = Logger.getLogger("NanoTiler_debug");
    public static boolean debugMode = false; // true;

    private static Vector3D[] generateIdealHelixPatch(int length, HelixParameters helixParameters) {
	Vector3D[] result = new Vector3D[2*length];
	int pc = 0;
	for (int i = 0; i < length; ++i) {
	    LineShape line = Rna3DTools.computeHelix(i, C4_NORM_POS, helixParameters);
	    Vector3D pos1 = line.getPosition1();
	    Vector3D pos2 = line.getPosition2();
	    // defines position of new pair:
	    result[pc++] = pos1;
	    result[pc++] = pos2;
	}
	return result;
    }

    /** returns reference position of base. Our ideal helix is computed with respect to atom with name "C4*" */ 
    private static Vector3D obtainHelixBasePos(Residue3D residue, 
					       boolean atomMode) {
	if (atomMode) {
	    int idx = residue.getIndexOfChild(helixRootAtomName);
	    if (idx >= 0) {
		return residue.getChild(idx).getPosition();
	    }
	}
	return residue.getPosition();
    }

    /** generates a small helix starting with pos1 - 1 and length (2*range + 1) */
    private static Vector3D[] generateHelixPatch(NucleotideStrand strand1, int pos1, 
						 NucleotideStrand strand2, int pos2, int offset, boolean atomMode) {
	int pc = 0;
	int startPos = pos1 + offset;
	int endPos = pos2 - offset;
	Vector3D[] result = scratchHelix;
	for (int i = 0; i < helixLength; ++i) {
	    Vector3D p1 = obtainHelixBasePos(strand1.getResidue3D(startPos + i), atomMode);
	    Vector3D p2 = obtainHelixBasePos(strand2.getResidue3D(endPos - i), atomMode);
	    // defines position of new pair:
	    result[pc++] = p1;
	    result[pc++] = p2;
	}
	return result;
    }

    /** returns more detailed info about last and first residues of stem */
    public static String stemResidueInfo(RnaStem3D stem) {
	StringBuffer buf = new StringBuffer();
	int len = stem.getLength();
	buf.append("5' start(stem pos 1, seq 1): " + stem.getStartResidue(0));
	buf.append("5' stop(stem pos 1, seq 2): " + stem.getStopResidue(0));
	buf.append("3' start(stem pos n, seq 1): " + stem.getStartResidue(len-1));
	buf.append("3' stop(stem pos 2, seq 2): " + stem.getStopResidue(len-1));
	return buf.toString();
    }

    /** returns more detailed info about last and first residues of stem */
    public static String stemResidueLongInfo(RnaStem3D stem) {
	StringBuffer buf = new StringBuffer();
	int len = stem.getLength();
	buf.append("Stem info: " + stem.getStemInfo() + NEWLINE);
	// buf.append("Number of base pairs: " + len);
	for (int i = 0; i < len; ++i) {
	    buf.append("base pair: " + (i+1));
	    buf.append("start: " + stem.getStartResidue(i).getName() + " " 
		       + stem.getStartResidue(i).getChild(BranchDescriptorTools.REF_ATOM_NAME));
	    buf.append(" stop: " + stem.getStopResidue(i).getName() + " " 
		       + stem.getStopResidue(i).getChild(BranchDescriptorTools.REF_ATOM_NAME)  + NEWLINE);
	}
	return buf.toString();
    }

    /** convinience method to generate "natural" SimpleRnaStem3D object from given coodinates. 
     * Needs start position of strand1 (5'end) and direction. 
     */
    public static RnaStem3D generateNaturalStem(NucleotideStrand strand1, 
						NucleotideStrand strand2, 
						Stem stem,
						String stemName)
	// 					      int startPos,
	// 					      int stopPos, 
	// 					      int length, 
	// 					      boolean setEqualMode,
	// 					      
	// 					      Vector3D strand1BasePos,
	// 					      Vector3D directionOrig) 
    {
	// log.fine("Computing stem3d: " + strand1BasePos + "  " + directionOrig);
// 	int startPos = stemInfo.getStartPos();
// 	int stopPos = stemInfo.getStopPos();
	// Stem stem = new SimpleStem(startPos, stopPos, length, strand1.getSequence(), strand2.getSequence());
	if (!stem.isValid()) {
	    throw new ApplicationBugException("Invalid stem parameters in RNA3DTools.generateRnaStem3D!");
	}
	// Vector3D base = strand1BasePos;
	// log.fine("Getting residue endPos " + (startPos+length-1) + " of strand1 with size " + strand1.getResidueCount());
	// Vector3D endPos = strand1.getResidue(startPos+ length -1).getPosition();
	// Vector3D direction = new Vector3D(directionOrig); // endPos.minus(base);
	// direction.normalize();
	// log.fine("Getting residue endPos " + (stopPos) + " of strand2 with size " + strand2.getResidueCount());
	// Vector3D base2 = strand2.getResidue(stopPos).getPosition();
	// Vector3D rpTmp = base2.minus(base);
	// Vector3D rp = direction.cross(rpTmp);
	// rp.normalize(); // relative vector pointing to first base pair from base (approximately). Also orth. to direction 

// 	Vector3D p0 = strand1.getResidue(startPos).getPosition();
// 	Vector3D p1Help = strand2.getResidue(stopPos).getPosition();
// 	Vector3D pTmp = p1Help.minus(p0);
// 	pTmp.normalize();
// 	pTmp.scale(WATSON_CRICK_DIST);
// 	Vector3D p1 = p0.plus(pTmp);
// 	Vector3D base = Vector3D.mean(p0, p1);
// 	Vector3D p2 = strand1.getResidue(startPos + (length-1)).getPosition();
// 	Vector3D p3Help = strand2.getResidue(stopPos - (length-1)).getPosition();
// 	pTmp = p3Help.minus(p2);
// 	pTmp.normalize();
// 	pTmp.scale(WATSON_CRICK_DIST);
// 	Vector3D p3 = p2.plus(pTmp);
// 	Vector3D p4 = Vector3D.mean(p2,p3); // top end
// 	Vector3D direction = p4.minus(base); // or should it be p4.minus(base) ?
// 	LineShape l1 = computeHelix(base, p0, direction, 0);
// 	LineShape l2 = computeHelix(base, p0, direction, length-1);

	RnaStem3D stem3D = new SimpleRnaStem3D(stem); // , l1.getPosition1(), l1.getPosition2(), l2.getPosition1(), l2.getPosition2());
	stem3D.setName(stemName);
	// add intermediate points:
	for (int i = 0; i < stem.size(); ++i) {
	    // LineShape line = computeHelix(base, rp, direction, i);
	    // TAKE position from given strand (instead of setting to ideal values)
	    Vector3D pos1 = strand1.getResidue3D(stem.getResidue1(i).getPos()).getPosition();
	    Object3D objectStrand1 = new SimpleObject3D(pos1);
	    String name = "S1N" + (i + 1);
	    objectStrand1.setName(name);
	    // Object3D objectStrand2 = new SimpleObject3D(line.getPosition2());
	    stem3D.insertChild(objectStrand1);
	}
	for (int i = 0; i < stem.size(); ++i) {
	    // LineShape line = computeHelix(base, rp, direction, i);
	    // TAKE position from given strand (instead of setting to ideal values)
	    Vector3D pos2 = strand2.getResidue3D(stem.getResidue2(i).getPos()).getPosition();
	    Object3D objectStrand2 = new SimpleObject3D(pos2);
	    String name = "S2N" + (i + 1);
	    objectStrand2.setName(name);
	    // Object3D objectStrand2 = new SimpleObject3D(line.getPosition2());
	    stem3D.insertChild(objectStrand2);
	}
	int length = stem.size();
	stem3D.setStrand1End5Index(0);
	stem3D.setStrand1End3Index(length-1);
	stem3D.setStrand2End5Index(length);
	stem3D.setStrand2End3Index(2*length-1);
	Object3DTools.balanceTree(stem3D); // new attempt to avoid 0,0,0 position
	// log.fine("Resulting stem3d: " + stem3D);
	assert stem3D != null;
	return stem3D;
    }


    /** Generates set of stems given a matrix. 
     * Converted from stemhelp.cc from supermol3 C++ project
     * careful: matrix does not have to be square!
     * number of rows (first index) is equal to length of strand1, 
     * number of columns (second index) is equal to length of strand2 
     */
    public static Object3DSet generateStemsFromSymmetricMatrix(double[][] mtx, 
							       int minStemLength,
							       double thresh,
							       NucleotideStrand strand1,
							       String stemBaseName)
    {
	assert mtx != null;
	assert minStemLength > 0;
	assert thresh > 0.0;
	log.fine("Starting generateStemsFromSymmetricMatrix... " + mtx.length);
	if (debugMode) {
	    DoubleArrayTools.writeMatrix(System.out, mtx);
	}
	Object3DSet result = new SimpleObject3DSet();
	NucleotideStrand strand2 = strand1;
	// TODO : also allow for Hoogsteen interaction etc
	InteractionType watsonCrick = new RnaInteractionType(RnaInteractionType.WATSON_CRICK);
	boolean openFlag = false;
	int stemCounter = 1;
	// RnaStem3D newStem;
	Sequence sequence = strand1; // .getSequence();
	Stem newStem = new SimpleStem(sequence, sequence);
	for (int nys = 0; nys < mtx.length; ++nys) {
	    for (int  nx = 0; nx < nys; ++nx) {
		int ny = nys - nx;
		if (nx >= ny) { // hitting diagonal
		    if (openFlag) {
			if (newStem.size() >= minStemLength) {
			    // 			    Vector3D pos = generateStemPos(newStem, strand1, strand2);
			    // 			    Object3D obj = new SimpleRnaStem3D(newStem);
			    // obj.setPosition(pos);
			    result.add(generateNaturalStem(strand1, strand2, newStem, 
							   (stemBaseName + (stemCounter++) )));
			    // result.add(obj);
			}
			openFlag = false;
		    }
		    break; // search only one half
		}
		if (mtx[nx][ny] >= thresh) {
		    // if stem does currently not exist
		    if (!openFlag) {
			newStem = new SimpleStem(sequence, sequence);
			// 			newStem.setStop(ny);
			// 			newStem.setLength(1);
			// 			if ((s.size() > 0) && (nx < static_cast<int>(s.size())) 
			// 			    && (ny < static_cast<int>(s.size()))) {
			// 			    // 	    ERROR_IF(nx >= static_cast<int>(s.size()), 
			// 			    // 		     "Internal error in line 29!", exception);
			// 			    newStem.setSequence1(s.substr(nx,1));
			// 			    // 	    ERROR_IF(ny >= static_cast<int>(s.size()), 
			// 			    // 		     "Internal error in line 31!", exception);
			// 			    newStem.setSequence2(s.substr(ny,1));
			// 			}
			openFlag = true;
		    }
		    // else {
		    // extension of stem:
		    // ASSERT(openFlag, exception);
		    // newStem.setLength(newStem.getLength() + 1);
		    // 			if ((s.size() > 0) && (nx < static_cast<int>(s.size())) 
		    // 			    && (ny < static_cast<int>(s.size()))) {
		    // 			    // 	    ERROR_IF(nx >= static_cast<int>(s.size()), 
		    // // 		     "Internal error in line 41!", exception);
		    // // 	    ERROR_IF(ny >= static_cast<int>(s.size()), 
		    // // 		     "Internal error in line 42!", exception);
		    // 			    newStem.setSequence1(newStem.getSequence1() + s.substr(nx,1));
		    // 			    newStem.setSequence2(s.substr(ny,1) + newStem.getSequence2());
		    Residue residue1 = strand1.getResidue(nx); // TODO check if order is ok
		    Residue residue2 = strand2.getResidue(ny);
		    Interaction interaction = new SimpleInteraction(residue1, residue2, watsonCrick);
		    newStem.add(interaction);
		}
		else { // save openend stem
		    if (openFlag) {
			if (newStem.size() >= minStemLength) {
			    // result.add(new SimpleRnaStem3D(newStem)); // TODO add child nodes
			    result.add(generateNaturalStem(strand1, strand2, newStem, 
							   (stemBaseName + (stemCounter++) )));
			}
			openFlag = false;
		    }
		}
	    }
	}
	if (openFlag) {
	    if (newStem.size() >= minStemLength) {
		// result.add(new SimpleRnaStem3D(newStem));
		result.add(generateNaturalStem(strand1, strand2, newStem, 
					       (stemBaseName + stemCounter++ )));

	    }
	    openFlag = false;
	}
	for (int  nxs = 1; nxs < mtx.length; ++nxs) {
	    for (int ny = mtx.length-1; ny >= 0; --ny) {
		int nx = nxs + (mtx.length - ny -1);
		if (nx >= ny) { // hitting diagonal
		    if (openFlag) {
			if (newStem.size() >= minStemLength) {
			    // result.add(new SimpleRnaStem3D(newStem));
			    result.add(generateNaturalStem(strand1, strand2, newStem, 
							   (stemBaseName + stemCounter++ )));
			}
			openFlag = false;
		    }
		    break;
		}
		// 		ERROR_IF(nx >= static_cast<int>(mtx.size()), 
		// 			 "Internal error in line 147!", exception);
		// 		ERROR_IF(ny >= static_cast<int>(mtx[nx].size()), 
		// 			 "Internal error in line 147!", exception);
		// 		ERROR_IF(nx < 0, "Internal error in line 185!", exception);
		// 		ERROR_IF(ny < 0, "Internal error in line 186!", exception);
		if (mtx[nx][ny] >= thresh) {
		    if (!openFlag) {
			newStem = new SimpleStem(strand1, strand2);
			// 			newStem.setStart(nx);
			// 			newStem.setStop(ny);
			// 			newStem.setLength(1);
			// 			if ((s.size() > 0) 
			// 			    && (nx < static_cast<int>(s.size())) 
			// 			    && (ny < static_cast<int>(s.size()))) {
			// 			    ERROR_IF(nx >= static_cast<int>(s.size()), 
			// 				     "Internal error in line 195!", exception);
			// 			    newStem.setSequence1(s.substr(nx,1));
			// 			    ERROR_IF(ny >= static_cast<int>(s.size()), 
			// 		     "Internal error in line 198!", exception);
			// 			    newStem.setSequence2(s.substr(ny,1));
			// 			}
			openFlag = true;
		    }
		    // 		    else {
		    // 			// extension of stem:
		    // 			ASSERT(openFlag, exception);
		    // 			// 			newStem.setLength(newStem.getLength() + 1);
		    // 			if ((s.size() > 0) && (nx < static_cast<int>(s.size())) && (ny < static_cast<int>(s.size()))) {
		    // 			    ERROR_IF(nx >= static_cast<int>(s.size()), 
		    // 				     "Internal error in line 41!", exception);
		    // 			    ERROR_IF(ny >= static_cast<int>(s.size()), 
		    // 				     "Internal error in line 42!", exception);
		    // 			    newStem.setSequence1(newStem.getSequence1() + s.substr(nx,1));
		    // 			    newStem.setSequence2(s.substr(ny,1) + newStem.getSequence2());
		    // 			}
		    //		    }
		    Residue residue1 = strand1.getResidue(nx); // TODO check if order is ok
		    Residue residue2 = strand2.getResidue(ny);
		    Interaction interaction = new SimpleInteraction(residue1, residue2, watsonCrick);
		    newStem.add(interaction);
		}
		else { // save openend stem
		    if (openFlag) {
			if (newStem.size() >= minStemLength) {
			    // result.add(new SimpleRnaStem3D(newStem)); // TODO add child nodes instead
			    result.add(generateNaturalStem(strand1, strand2, newStem, 
							   (stemBaseName + stemCounter++ )));
			}
			openFlag = false;
		    }
		}
	    }
	}
	removeConflictingStems(result);
	// sort(result.begin(), result.end());
	log.fine("Ending generateStemsFromSymmetricMatrix: " + result.size());
	return result;
    }

    

    /** Generates set of stems given a matrix. 
     * Converted from stemhelp.cc from supermol3 C++ project
     * careful: matrix does not have to be square!
     * number of rows (first index) is equal to length of strand1, 
     * number of columns (second index) is equal to length of strand2 
     * TODO: all interactions are set to Watson Crick. Overwrites initial better classification.
     */
    public static Object3DSet generateStemsFromUnsymmetricMatrix(double[][] mtx, 
								 int minStemLength,
								 double thresh,
								 NucleotideStrand strand1,
								 NucleotideStrand strand2,
								 String stemBaseName)
    {
	log.fine("Starting generateStemsFromUnsymmetricMatrix");
	if (debugMode) {
	    DoubleArrayTools.writeMatrix(System.out, mtx);
	}
	Object3DSet result = new SimpleObject3DSet();
	// TODO : also allow for Hoogsteen interaction etc
	InteractionType watsonCrick = new RnaInteractionType(RnaInteractionType.WATSON_CRICK);
	boolean openFlag = false;
	int stemCounter = 1;
	// RnaStem3D newStem;
	int numRows = mtx.length;
	int numCols = mtx[0].length;
	Sequence sequence1 = strand1; // .getSequence();
	Sequence sequence2 = strand2; // .getSequence();
	Stem newStem = new SimpleStem(sequence1, sequence2);
	int[][] visitedMtx = new int[mtx.length][mtx[0].length];
	int bondCount = 0;
	for (int i = 0; i < numRows; ++i) {
	    for (int j = 0; j < numCols; ++j) {
		if (mtx[i][j] >= thresh) {
		    ++bondCount;
		}
		
	    }
	}
	log.fine("Number of interactions between strands: " + bondCount + " " + strand1.getName() + " " + strand2.getName());
	for (int i = 0; i < numRows; ++i) {
	    for (int j = 0; j < numCols; ++j) {
		if (visitedMtx[i][j] == 1) {
		    continue; // point already visited
		}
		if (mtx[i][j] >= thresh) {
		    // if stem does currently not exist
		    newStem = new SimpleStem(sequence1, sequence2);
		    int k = 0; 
		    while ( (i+k<numRows) && (j>=k) && (mtx[i+k][j-k] >= thresh)) { 
			if ((visitedMtx[i+k][j-k] > 0)) {
			    break; // already visited
			}
			visitedMtx[i+k][j-k] = 1;
			Residue residue1 = strand1.getResidue(i+k); // TODO check if order is ok
			Residue residue2 = strand2.getResidue(j-k);
			Interaction interaction = new SimpleInteraction(residue1, residue2, watsonCrick);
			newStem.add(interaction);
			++k;
		    }
		    if (newStem.size() >= minStemLength) {
			// result.add(new SimpleRnaStem3D(newStem)); // TODO add child nodes
			RnaStem3D naturalStem = generateNaturalStem(strand1, strand2, newStem, 
								    (stemBaseName + (stemCounter++) ) );
			assert naturalStem != null;
			log.fine("Generated new stem: " +  naturalStem);
			result.add(naturalStem);
		    }
		}
		else {
		    visitedMtx[i][j] = 1;
		}
	    }
	}
	log.fine("generateStemsFromUnsymmetricMatrix before removeConflictingStems: " + result.size());
	removeConflictingStems(result);
	log.fine("Ending generateStemsFromUnsymmetricMatrix: " + result.size());
	return result;
    }


    /** generates contact matrix for residues. Careful: resulting matrix is not necessarily symmetric! 
     * Assumes really two different strands.
     */
    private static double[][] generateContactMatrix(BioPolymer strand1, 
						    BioPolymer strand2, 
						    boolean atomMode) {
	double[][] mtx = new double[strand1.getResidueCount()][strand2.getResidueCount()];
	boolean symmetric = false;
	int idx = -1;
	int idy = -1;
	for (int i = 0; i < strand1.getResidueCount(); ++i) {
	    for (int j = 0; j < strand2.getResidueCount(); ++j) {
		if (isContacting2(strand1, i,
				 strand2, j, atomMode)) {
		    // log.fine("Setting contacts between " + (i+1) + " and " + (j+1));
		    mtx[i][j] = 1.0;
		    idx = i;
		    idy = j;
		}
		else {
		    mtx[i][j] = 0.0;
		}
	    }
	}
// 	if (idx >= 0) {
// 	    log.fine("Verifying contact: " + mtx[idx][idy]);
// 	}
	return mtx;
    }

    static boolean isResiduesContacting(Residue3D residue1, Residue3D residue2, double cutoff) {
	double dist = residue1.distance(residue2);
	if (dist <= cutoff) {
	    return true;
	}
	return false;
    }

    static boolean isAtomsContacting(Residue3D residue1, Residue3D residue2, double cutoff) {
	for (int i = 0; i < residue1.getAtomCount(); ++i) {
	    for (int j = 0; j < residue2.getAtomCount(); ++j) {
		double dist = residue1.getAtom(i).distance(residue2.getAtom(j));
		if (dist <= cutoff) {
		    return true;
		}
	    }
	}
	return false;
    }

    static double atomMinimumDistance(Residue3D residue1, Residue3D residue2) {
	double minDist = residue1.distance(residue2);
	for (int i = 0; i < residue1.getAtomCount(); ++i) {
	    for (int j = 0; j < residue2.getAtomCount(); ++j) {
		double dist = residue1.getAtom(i).distance(residue2.getAtom(j));
		if (dist <= minDist) {
		    minDist = dist;
		}
	    }
	}
	return minDist;
    }

    // strand must be larger than 2 in each case
    static Vector3D getBackboneVector(BioPolymer strand,
				      int pos) {
	int pnext = pos + 1;
	int pbefore = pos - 1;
	if (pbefore  < 0) {
	    pbefore = 0;
	}
	if (pnext >= strand.getResidueCount()) {
	    pnext = strand.getResidueCount()-1;
	}
	if (pnext <= pbefore) {
	    throw new ApplicationBugException("Internal error in line 338!");
	}
	int pos1Id = strand.getResidue3D(pbefore).getIndexOfChild("P");
	int pos2Id = strand.getResidue3D(pnext).getIndexOfChild("P");
	if ((pos1Id >= 0) && (pos2Id >= 0)) {
	    Vector3D pos1 = strand.getResidue3D(pbefore).getChild(pos1Id).getPosition();
	    Vector3D pos2 = strand.getResidue3D(pnext).getChild(pos2Id).getPosition();

	    return pos2.minus(pos1);
	}
	return new Vector3D();
    }

    // strand must be larger than 2 in each case
    static Vector3D getBaseVector(Residue3D residue) {
	int atom1Id = residue.getIndexOfChild("P");
	int atom2Id = residue.getIndexOfChild("C1*"); // TODO make more general
	if ((atom1Id >= 0) && (atom2Id >= 0)) {
	    return residue.getChild(atom1Id).getPosition().minus(
		residue.getChild(atom2Id).getPosition());
	}
	return new Vector3D();
    }

    static boolean isContactingBackbone(BioPolymer strand1,
					int pos1,
					BioPolymer strand2,
					int pos2) {
	// 	return true; // TODO take out!!!
 	Vector3D v1 = getBackboneVector(strand1, pos1);
 	Vector3D v2 = getBackboneVector(strand2, pos2);
	if ((v1.lengthSquare()>0)&& (v2.lengthSquare()>0)) {
	    double angle = v1.angle(v2);
	    // log.fine("Angle: " + angle);
	    if ((angle >= backboneAngleMin)
		&& (angle <= backboneAngleMax)) {
		return true;
	    }
	}
	return false;
    }

    /** assumes nucleotide strands!!! */
    static boolean isContactingBase(BioPolymer strand1,
				    int pos1,
				    BioPolymer strand2,
				    int pos2) {
	// 	return true; // TODO take out
 	Vector3D v1 = getBaseVector(strand1.getResidue3D(pos1));
 	Vector3D v2 = getBaseVector(strand2.getResidue3D(pos2));
 	if ((v1.lengthSquare() > 0) && (v2.lengthSquare() > 0)) {
 	    double angle = v1.angle(v2);
 	    if ((angle >= baseAngleMin)
	       && (angle < baseAngleMax)) {
 		return true;
 	    }
 	}
 	return false;
    }

    /** returns true if complementary base pair if approx. correct orientation */
    static boolean isContacting(BioPolymer strand1,
				int pos1,
				BioPolymer strand2,
				int pos2,
				boolean atomMode) {
	Residue3D residue1 = strand1.getResidue3D(pos1);
	Residue3D residue2 = strand2.getResidue3D(pos2);
	double residueDist = residue1.distance(residue2);
	if (!DnaTools.isComplementary(residue1.getSymbol().getCharacter(),
				      residue2.getSymbol().getCharacter())) {
	    return false; // if not AU, GC or GU (or AT, GT)
	}
	if (residueDist > WATSON_CRICK_RESIDUE_CUTOFF) {
	    return false;
	}

	if ((strand1.getResidueCount() > 2) && (strand2.getResidueCount() > 2)) {
	    boolean backboneContact = isContactingBackbone(strand1, pos1, strand2, pos2);
	    if (!backboneContact) {
		return false;
	    }
	}
	// generate backbone vectors:
	if (atomMode) {
	    boolean backboneContact = isContactingBackbone(strand1, pos1, strand2, pos2);
	    double atomDist = atomMinimumDistance(residue1, residue2);
	    if (atomDist < WATSON_CRICK_ATOM_CUTOFF) {
		if ((strand1 instanceof NucleotideStrand)
		    && (strand2 instanceof NucleotideStrand)
		    && (!isContactingBase(strand1, pos1, strand2, pos2))) {
		    return false;
		}
		for (int i = pos1 - range; i <= pos1+range; i+=1) {
		    if ((i==pos1) || (i < 0) || (i >= strand1.getResidueCount())) {
			continue;
		    }
		    for (int j = pos2 - range; j <= pos2+range; j+=1) {
			if ((j==pos2)||(j < 0) || (j >= strand2.getResidueCount())) {
			    continue;
			}
			double newDist = atomMinimumDistance(strand1.getResidue3D(i), strand2.getResidue3D(j));
			if (newDist < atomDist) {
			    return false; // other nearby pairing leads to closer distances
			}
		    }
		}
		return true;
	    }		
	}
	return isResiduesContacting(residue1, residue2, WATSON_CRICK_RESIDUE_CUTOFF);
    }

    static double residueDistance(Residue3D res1, Residue3D res2, String atomName) {
	int id1 = res1.getIndexOfChild(atomName);
	int id2 = res2.getIndexOfChild(atomName);
	if ((id1 >= 0) && (id2 >= 0)) {
	    return res1.getChild(id1).distance(res2.getChild(id2));
	}
	return res1.distance(res2);
    }

    /** returns true if complementary base pair if approx. correct orientation */
    static boolean isContacting2(BioPolymer strand1,
				 int pos1,
				 BioPolymer strand2,
				 int pos2,
				 boolean atomMode) {
	if ((pos1 == 0) || (pos2 == 0) || ((pos1 + 1) == strand1.getResidueCount())
	    || ((pos2 + 1) == strand2.getResidueCount())
	    || (strand1.getResidueCount() <= 2) || (strand2.getResidueCount() <= 2)) {
	    return isContacting(strand1, pos1, strand2, pos2, atomMode); // false; // TODO ignoring endpoints so far
	}
	Residue3D residue1 = strand1.getResidue3D(pos1);
	Residue3D residue2 = strand2.getResidue3D(pos2);
	double residueDist = residueDistance(residue1, residue2, "C4*");
 	if (residueDist > WATSON_CRICK_RESIDUE_CUTOFF) {
 	    return false;
 	}
	// TODO : find why checking for complementary bases does not work!?? :-(((
     	if (!DnaTools.isComplementary(residue1.getSymbol().getCharacter(),
     				      residue2.getSymbol().getCharacter())) {
//     	    log.fine("not complementary: " 
//     			       + (pos1 + 1) + " " + residue1.getSymbol() + " " 
//     			       + (pos2 + 1) + " " + residue2.getSymbol());
     	    return false; // if not AU, GC or GU (or AT, GT)
     	}
    	else {
//     	    log.fine("complementary: " 
//     			       + (pos1 + 1) + " " + residue1.getSymbol() + " " 
//     			       + (pos2 + 1) + " " + residue2.getSymbol());
    	}
	// generate small helix, different offsets:
	for (int offset = - helixLength+1; offset <= 0; ++offset) {
	    int startPos = pos1 + offset;
	    int endPos = pos2 - offset;
	    // TODO verify
	    if ((startPos < 0) || ((startPos + helixLength) >= strand1.getResidueCount())
		|| (endPos >= strand2.getResidueCount()) || (endPos - helixLength < 0)) {
		continue;
	    }
	    Vector3D[] helix = generateHelixPatch((NucleotideStrand)strand1, pos1, 
						  (NucleotideStrand)strand2, pos2, offset, atomMode);
	    if (helix.length != idealHelix.length) {
		throw new ApplicationBugException("Internal error in line 530!");
	    }
	    double drms = Vector3DTools.computeDrms(helix, idealHelix);
	    if (drms < drmsCutoff) {
// 		log.fine("True contact found! Drms (offset -1) between " + (pos1 + 1) + " , " + (pos2 + 1) + " : " + drms);
		return true;
	    }
	}
	return false; // true; // TODO should be false, just for debugging
    }


    /** generates contact matrix for residues. Careful: resulting matrix is not necessarily symmetric! 
     */
    private static double[][] generateContactMatrix(BioPolymer strand1, 
						    boolean atomMode) {
	double[][] mtx = new double[strand1.getResidueCount()][strand1.getResidueCount()];
	for (int i = 0; i < strand1.getResidueCount(); ++i) {
	    for (int j = i+4; j < strand1.getResidueCount(); ++j) {
		if (isContacting2(strand1, i, 
				 strand1, j, 
				 atomMode)) {
		    mtx[i][j] = 1.0;
		}
		else {
		    mtx[i][j] = 0.0;
		}
		mtx[j][i] = mtx[i][j];
	    }
	}
	return mtx;
    }

    private static void writeSparseMatrix(double[][] mtx) {
	for (int i = 0; i < mtx.length; ++i) {
	    for (int j = 0; j < mtx[i].length; ++j) {
		if (mtx[i][j] > 0.0) {
		    System.out.println("" + (i+1) + " " + (j+1) + " "
				       + mtx[i][j]);
		}
	    }
	}
    }

    /** Generates inter-strand stems; use for both RNA and DNA strands */
    public static Object3DSet generateStems(NucleotideStrand strand1, NucleotideStrand strand2, String stemBaseName) {
	Object3DSet stemSet = new SimpleObject3DSet();	
	// convert to matrix
	double[][] mtx = generateContactMatrix(strand1, strand2, atomMode);
// 	log.fine("Contact matrix for strands " 
// 		 + strand1.getName() + strand2.getName());
// 	writeSparseMatrix(mtx);
	return generateStemsFromUnsymmetricMatrix(mtx, STEM_LENGTH_MINIMUM, threshold, strand1, strand2, stemBaseName);	
    }

    /** Generate intra-strand stems; use for both RNA and DNA strands */
    public static Object3DSet generateStems(NucleotideStrand strand1,
					    String stemBaseName) {
	Object3DSet stemSet = new SimpleObject3DSet();	
	// convert to matrix
	double[][] mtx = generateContactMatrix(strand1, atomMode);
	// write temporary matrix:
// 	try {
// 	    FileOutputStream ofs = new FileOutputStream("contacts_self.matrix");
// 	    PrintStream ps = new PrintStream(ofs);
// 	    DoubleArrayTools.writeMatrix(ps, mtx);
// 	    ofs.close();
// 	}
// 	catch (IOException exc) {
// 	    // do nothing
// 	}
	writeSparseMatrix(mtx);
	return generateStemsFromSymmetricMatrix(mtx, STEM_LENGTH_MINIMUM, threshold, strand1, stemBaseName);	
    }
    
    public static void removeConflictingStems(Object3DSet stemSet) {
	return; // TODO !!!
// 	boolean allOk = false;
// 	while (!allOk) {
// 	    allOk = true;
// 	    for (int i = 0; allOk && (i < stemSet.size()); ++i) {
// 		Object3D obj1 = stemSet.get(i);
// 		if (obj1 instanceof RnaStem3D) {
// 		    Stem stem1 = ((RnaStem3D)obj1).getStemInfo();
// 		    for (int j = i+1; allOk && (j < stemSet.size()); ++j) {
// 			Object3D obj2 = stemSet.get(j);
// 			if (obj2 instanceof RnaStem3D) {
// 			    Stem stem2 = ((RnaStem3D)obj2).getStemInfo();
// 			    if (false) { // TODO stem1.isConflicting(stem2)) {
// 				// remove shorter stem
// 				if (stem1.size() >= stem2.size()) {
// 				    stemSet.remove(obj2);
// 				    log.info("Removed conflicting stem: " + stem2 + " keep: " + stem1);
// 				}
// 				else {
// 				    stemSet.remove(obj1);
// 				    log.info("Removed conflicting stem: " + stem1 + " keep: " + stem2);
// 				}
// 				allOk = false;
// 				break;
// 			    }
// 			}
// 		    }
// 		}
// 		if (!allOk) {
// 		    break;
// 		}
// 	    }
// 	}
    }

    /** generates for an arbitrary object tree a set of stems. */
    public static Object3DLinkSetBundle generateStems(Object3D root) {
	Object3D resultRoot = new SimpleObject3D();
	Object3DSet resultSet = new SimpleObject3DSet();
	LinkSet resultLinks = new SimpleLinkSet();
	resultRoot.setName("stems");
	Object3DSet strandSet = Object3DTools.collectByClassName(root, "RnaStrand"); // Get all RNA strands.
	strandSet.merge(Object3DTools.collectByClassName(root, "DnaStrand")); // Add DNA strands.
	for (int i = 0; i < strandSet.size(); ++i) {
	    NucleotideStrand strand1 = (NucleotideStrand)(strandSet.get(i));
	    for (int j = i; j < strandSet.size(); ++j) {
		NucleotideStrand strand2 = (NucleotideStrand)(strandSet.get(j));
		String stemBaseName = "H" + (i+1) + "_" + (j+1);
		Object3DSet stemSet;
		if (i == j) { // Same stem.
		    stemSet = generateStems(strand1, stemBaseName);
		}
		else {
		    stemSet = generateStems(strand1, strand2, stemBaseName);
		}
		resultSet.merge(stemSet); // Add found stems to result object root.
	    }
	}
	for (int i = 0; i < resultSet.size(); ++i) { // Translate into tree structure.
	    resultRoot.insertChild(resultSet.get(i));
	}
	Object3DLinkSetBundle result = new SimpleObject3DLinkSetBundle(resultRoot, resultLinks);
	return result;
    }
    
    /** Generates contact matrix of Watson Crick pairs */
    private static double[][] generateMatrixFromLinks(NucleotideStrand strand,
						      LinkSet links) {
	if (debugMode) {
	    assert links.size() > 0; // only for debugging
	}
	int numResidues = strand.getResidueCount();
	assert numResidues > 0;
	double[][] matrix = new double[numResidues][numResidues];
	assert matrix.length == numResidues;
	assert matrix[0].length == numResidues;
	for (int i = 0; i < links.size(); ++i) {
	    Link link = links.get(i);
	    if (! (link instanceof InteractionLink) ) {
		continue;
	    }
	    Interaction interaction = ((InteractionLink)link).getInteraction();
	    InteractionType interactionType = interaction.getInteractionType();	    
	    if (! (interactionType instanceof RnaInteractionType) ) {
		// log.finest("Ignoring link because it is not an RnaInteractionType");
		continue; // do not use backbone interactions
	    }
	    else {
		if (interactionType.getSubTypeId() == RnaInteractionType.BACKBONE) {
		    // log.finest("Ignoring link because it is a backbone interaction: " + interaction);
		    continue; // ignore backbone interactions
		}
	    }
	    Object3D obj1 = link.getObj1();
	    Object3D obj2 = link.getObj2();
	    if ((obj1 instanceof Residue3D) && (obj1.getParent() == strand)
 		&& (obj2 instanceof Residue3D) && (obj2.getParent() == strand) ) {
		int p1 = ((Residue3D)obj1).getPos();
		int p2 = ((Residue3D)obj2).getPos();
		matrix[p1][p2] = 1.0;
		matrix[p2][p1] = 1.0;
	    }
	}
	return matrix;
    }

    /** Generates contact matrix of Watson Crick pairs between strands. */
    private static double[][] generateMatrixFromLinks(NucleotideStrand strand1,
						      NucleotideStrand strand2,
						      LinkSet links) {
	int numResidues1 = strand1.getResidueCount();
	int numResidues2 = strand2.getResidueCount();
	double[][] matrix = new double[numResidues1][numResidues2];
	int count = 0;
	for (int i = 0; i < links.size(); ++i) {
	    Link link = links.get(i);
	    if (! (link instanceof InteractionLink) ) {
		continue;
	    }
	    Interaction interaction = ((InteractionLink)link).getInteraction();
	    InteractionType interactionType = interaction.getInteractionType();	    
	    if (! ((interactionType instanceof RnaInteractionType) 
		   && (interactionType.getSubTypeId() != RnaInteractionType.BACKBONE))) {
		continue; // only use Watson Crick pairs
	    }
	    Object3D obj1 = link.getObj1();
	    Object3D obj2 = link.getObj2();
	    if ((obj1 instanceof Residue3D) && (obj2 instanceof Residue3D) ) {
		int p1 = ((Residue3D)obj1).getPos();
		int p2 = ((Residue3D)obj2).getPos();
		if ( (obj1.getParent() == strand1)
		     && (obj2.getParent() == strand2) ) {
		    matrix[p1][p2] = 1.0;
		    ++count;
		}
		else if ((obj1.getParent() == strand2)
			 && (obj2.getParent() == strand1) ) {
		    matrix[p2][p1] = 1.0;
		    ++count;
		}
	    }
	}
	log.fine("Generating " + count + " contacts in matrix between strands " + strand1.getFullName() 
		 + " and " + strand2.getFullName());
// 	if (count > 0) {
// 	    DoubleArrayTools.writeMatrix(System.out, matrix);
// 	}
	return matrix;
    }

    /** generates for an arbitrary object tree a set of stems. */
    public static Object3DLinkSetBundle generateStemsFromLinks(Object3D root,
							       LinkSet links, int minStemLength) {
	log.fine("Starting generateStemsFromLinks!");
	final double thresh = 0.5;
	Object3D resultRoot = new SimpleObject3D();
	Object3DSet resultSet = new SimpleObject3DSet();
	LinkSet resultLinks = new SimpleLinkSet();
	resultRoot.setName("stems");
		Object3DSet strandSet = Object3DTools.collectByClassName(root, "RnaStrand"); // Get all RNA strands.
	strandSet.merge(Object3DTools.collectByClassName(root, "DnaStrand")); // RNA and DNA strand are put into same container.
	log.fine("Detected " + strandSet.size() + " strands.");
	for (int i = 0; i < strandSet.size(); ++i) {
	    NucleotideStrand strand1 = (NucleotideStrand)(strandSet.get(i));
	    assert strand1.size() > 0;
	    String stemBaseName = "stem" + strand1.getName() + "_";
	    double[][] matrix = generateMatrixFromLinks(strand1, links);
	    Object3DSet newStemSet = generateStemsFromSymmetricMatrix(matrix,
	      minStemLength, thresh, strand1, stemBaseName);
	    resultSet.merge(newStemSet);
// 	    for (int k = 0; k < newStemSet.size(); ++k) {
// 		RnaStem3D stem3D = ((RnaStem3D)(newStemSet));
// 		Stem stemInfo = stem3D.getStemInfo();
// 		for (int m = 0; m < stemInfo.getLength(); ++m) {
// 		    Interaction interaction = stemInfo.get(m);
		    
// 		    resultLinks.add(new InteractionLinkImp(obj1, obj2, interaction));
// 		}
// 	    }
	    newStemSet.clear();
	    for (int j = i+1; j < strandSet.size(); ++j) {
		NucleotideStrand strand2 = (NucleotideStrand)(strandSet.get(j));
		stemBaseName = "stem" + strand1.getName() + "_" + strand2.getName()+"_";
		matrix = generateMatrixFromLinks(strand1, strand2, links);
		assert matrix.length == strand1.getResidueCount();
		assert matrix[0].length == strand2.getResidueCount();
		newStemSet = generateStemsFromUnsymmetricMatrix(matrix,
					minStemLength, thresh,
					strand1, strand2, stemBaseName);
		if (newStemSet.size() > 0) {
		    log.fine("Found " + newStemSet.size() + " + stems between strands " + strand1.getFullName() + " and "
			     + strand2.getFullName());
		}
		resultSet.merge(newStemSet);
		newStemSet.clear();
	    }
	}
	// remove conflicting stems: : now done much earlier!
	// 	log.info("Starting removeConflictingStems with size: " + resultSet.size());
	// 	removeConflictingStems(resultSet);
	// log.fine("Finished removeConflictingStems with size: " + resultSet.size());
	// translate into tree structure:
	for (int i = 0; i < resultSet.size(); ++i) {
	    resultRoot.insertChild(resultSet.get(i));
	}
	log.fine("Finished generateStemsFromLinks: " + resultRoot.size());
	return new SimpleObject3DLinkSetBundle(resultRoot, resultLinks);
    }

    /** returns true if nucleotide is part of stem */
    public static boolean isPartOfStem(Nucleotide3D residue, Stem stem) {
// 	Sequence residueSequence = residue.getSequence();
// 	if (residueSequence == null) {
// 	    return false; // no sequence attached to nucleotide
// 	}
	Sequence seq1 = stem.getSequence1();
	int pos = residue.getPos();
	// if pointing towards same sequence:
	if (residue.isSameSequence(seq1.getResidue(0))) {
	    if ((pos >= stem.getStartPos()) && (pos < (stem.getStartPos()+stem.size()))) {
		return true;
	    }
	}
	Sequence seq2 = stem.getSequence2();
	// if pointing towards same sequence:
	if (residue.isSameSequence(seq2.getResidue(0))) {
	    if ((pos <= stem.getStopPos()) && (pos > (stem.getStopPos()-stem.size()))) {
		return true;
	    }
	}
	return false;
    }

    /** returns true if nucleotide is part of stem3d object */
    public static boolean isPartOfStem(Nucleotide3D residue, RnaStem3D stem3D) {
	return isPartOfStem(residue, stem3D.getStemInfo());
    }

    /** searches all known stems */
    public static boolean isPartOfStem(Nucleotide3D residue, Object3DSet stemSet) {
	for (int i = 0; i < stemSet.size(); ++i) {
	    Object3D child = stemSet.get(i);
	    if (child instanceof RnaStem3D) {
		boolean result = isPartOfStem(residue, (RnaStem3D)child);
		if (result == true) {
		    return true;
		}
	    }
	}
	return false;
    }

    /** searches all known stems TODO : no recursive search*/
    public static Object3DSet collectStems3D(Object3D root) {
	Object3DSet stemSet = new SimpleObject3DSet();
	for (int i = 0; i < root.size(); ++i) {
	    Object3D child = root.getChild(i);
	    if (child instanceof RnaStem3D) {
		stemSet.add(child);
	    }
	}
	return stemSet;
    }

    /** searches all known stems */
    public static boolean isPartOfStem(Nucleotide3D residue, Object3D root) {
	Object3DSet stemSet = collectStems3D(root);
	return isPartOfStem(residue, stemSet);
    }

}
